/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */
package com.clustercontrol.notify.monitor.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.notify.monitor.ejb.entity.StatusInfoBean;
import com.clustercontrol.notify.monitor.ejb.entity.StatusInfoPK;

/**
 * ステータス情報のDAOインターフェースを実装するクラス<BR>
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.notify.monitor.ejb.entity.StatusInfoBean
 * @see com.clustercontrol.notify.monitor.dao.StatusInfoDAO
 */
public class StatusInfoDAOImpl implements StatusInfoDAO {

	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(StatusInfoDAOImpl.class);

	public void init() {
	}

	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.notify.monitor.dao.StatusInfoDAO#load(com.clustercontrol.notify.monitor.ejb.entity.StatusInfoPK, com.clustercontrol.notify.monitor.ejb.entity.StatusInfoBean)
	 */
	public void load(StatusInfoPK pk, StatusInfoBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			// SQL文の定義
			String sql = "SELECT * FROM cc_status_info WHERE monitor_id = ? AND plugin_id = ? AND facility_id = ?";
			// SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.monitorId);
			stmt.setString(2, pk.pluginId);
			stmt.setString(3, pk.facilityId);

			res = stmt.executeQuery();

			if (res.next()) {
				// 取得した値をBeanにセット
				ejb.setMonitorId(res.getString("monitor_id"));
				ejb.setPluginId(res.getString("plugin_id"));
				ejb.setFacilityId(res.getString("facility_id"));

				ejb.setOutputDate(res.getTimestamp("output_date"));
				ejb.setGenerationDate(res.getTimestamp("generation_date"));
				ejb.setApplication(res.getString("application"));
				ejb.setMessageId(res.getString("message_id"));
				ejb.setMessage(res.getString("message"));
				ejb.setPriority(res.getInt("priority"));
				ejb.setExpirationFlg(res.getInt("expiration_flg"));
				ejb.setExpirationDate(res.getTimestamp("expiration_date"));

			} else {
				String msg = "StatusInfo data is not found.";
				m_log.error("load() error : " + pk.toString()
						+ " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString()
						+ " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}

	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.notify.monitor.dao.StatusInfoDAO#store(com.clustercontrol.notify.monitor.ejb.entity.StatusInfoBean)
	 */
	public void store(StatusInfoBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getPluginId() + ", "
				+ ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", "
				+ ejb.getOutputDate());

		Connection conn = null;
		PreparedStatement stmt = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			// SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("UPDATE cc_status_info SET ");
			sql.append("generation_date = ?, ");
			sql.append("output_date = ?, ");
			sql.append("application = ?, ");
			sql.append("message_id = ?, ");
			sql.append("message = ?, ");
			sql.append("priority = ?, ");
			sql.append("expiration_flg = ?, ");
			sql.append("expiration_date = ? ");
			sql.append("WHERE plugin_id = ? AND ");
			sql.append("monitor_id = ? AND ");
			sql.append("facility_id = ?");
			
			// SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setTimestamp(1, ejb.getGenerationDate());
			stmt.setTimestamp(2, ejb.getOutputDate());
			stmt.setString(3, ejb.getApplication());
			stmt.setString(4, ejb.getMessageId());
			stmt.setString(5, ejb.getMessage());
			stmt.setInt(6, ejb.getPriority());
			stmt.setInt(7, ejb.getExpirationFlg());
			stmt.setTimestamp(8, ejb.getExpirationDate());

			stmt.setString(9, ejb.getPluginId());
			stmt.setString(10, ejb.getMonitorId());
			stmt.setString(11, ejb.getFacilityId());

			int row = stmt.executeUpdate();

			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getPluginId() + ", "
						+ ejb.getMonitorId() + ", " + ejb.getFacilityId()
						+ " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : " + ejb.getPluginId() + ", "
					+ ejb.getMonitorId() + ", " + ejb.getFacilityId()
					+ " SQLException");
			e.printStackTrace();
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : " + ejb.getPluginId() + ", "
						+ ejb.getMonitorId() + ", " + ejb.getFacilityId()
						+ " SQLException");
				e1.printStackTrace();
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : " + ejb.getPluginId() + ", "
				+ ejb.getMonitorId() + ", " + ejb.getFacilityId());
	}

	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.notify.monitor.dao.StatusInfoDAO#remove(com.clustercontrol.notify.monitor.ejb.entity.StatusInfoPK)
	 */
	public void remove(StatusInfoPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			// SQL文の定義
			String sql = "DELETE FROM cc_status_info WHERE plugin_id = ? AND monitor_id = ? AND facility_id = ?";
			// SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.pluginId);
			stmt.setString(2, pk.monitorId);
			stmt.setString(3, pk.facilityId);

			int row = stmt.executeUpdate();

			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString()
						+ " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());

		} finally {
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString()
						+ " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}

	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.notify.monitor.dao.StatusInfoDAO#create(com.clustercontrol.notify.monitor.ejb.entity.StatusInfoBean)
	 */
	public StatusInfoPK create(StatusInfoBean ejb) throws CreateException,
			EJBException {
		m_log.debug("create() start : " + ejb.getPluginId() + ", "
				+ ejb.getMonitorId() + ", " + ejb.getFacilityId());

		Connection conn = null;
		PreparedStatement stmt = null;
		StatusInfoPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			// SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_status_info (");
			sql.append("plugin_id, ");
			sql.append("monitor_id, ");
			sql.append("facility_id, ");
			sql.append("output_date, ");
			sql.append("generation_date, ");
			sql.append("application, ");
			sql.append("message_id, ");
			sql.append("message, ");
			sql.append("priority, ");
			sql.append("expiration_flg, ");
			sql.append("expiration_date) ");
			sql.append("values (?,?,?,?,?,?,?,?,?,?,?)");
			
			// SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setString(1, ejb.getPluginId());
			stmt.setString(2, ejb.getMonitorId());
			stmt.setString(3, ejb.getFacilityId());
			stmt.setTimestamp(4, ejb.getOutputDate());
			stmt.setTimestamp(5, ejb.getGenerationDate());
			stmt.setString(6, ejb.getApplication());
			stmt.setString(7, ejb.getMessageId());
			stmt.setString(8, ejb.getMessage());
			stmt.setInt(9, ejb.getPriority());
			stmt.setInt(10, ejb.getExpirationFlg());
			stmt.setTimestamp(11, ejb.getExpirationDate());

			int row = stmt.executeUpdate();
			pk = new StatusInfoPK(ejb.getMonitorId(), ejb.getPluginId(), ejb
					.getFacilityId());

			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getPluginId() + ", "
						+ ejb.getMonitorId() + ", " + ejb.getFacilityId()
						+ " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if (e.getSQLState().equals("23505")) {
				m_log.error("create() error : " + ejb.getPluginId() + ", "
						+ ejb.getMonitorId() + ", " + ejb.getFacilityId()
						+ " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		} finally {
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getPluginId() + ", "
						+ ejb.getMonitorId() + ", " + ejb.getFacilityId()
						+ " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getPluginId() + ", "
				+ ejb.getMonitorId() + ", " + ejb.getFacilityId());
		return pk;
	}

	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @return プライマリキーのコレクション
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.notify.monitor.ejb.entity.StatusInfoBean#ejbFindAll()
	 * @see com.clustercontrol.notify.monitor.dao.StatusInfoDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");

		ArrayList<StatusInfoPK> ret = new ArrayList<StatusInfoPK>();

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			// SQL文の定義
			String sql = "SELECT facility_id,monitor_id,plugin_id FROM cc_status_info";
			// SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();

			while (res.next()) {
				StatusInfoPK pk = new StatusInfoPK(res.getString("monitor_id"),
						res.getString("plugin_id"), res
								.getString("facility_id"));
				ret.add(pk);
			}

		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}

	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @param pk プライマリキー
	 * @return プライマリキー
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.notify.monitor.ejb.entity.StatusInfoBean#ejbFindByPrimaryKey(com.clustercontrol.notify.monitor.ejb.entity.StatusInfoPK)
	 * @see com.clustercontrol.notify.monitor.dao.StatusInfoDAO#findByPrimaryKey(com.clustercontrol.notify.monitor.ejb.entity.StatusInfoPK)
	 */
	public StatusInfoPK findByPrimaryKey(StatusInfoPK pk)
			throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			// SQL文の定義
			String sql = "SELECT facility_id,monitor_id,plugin_id FROM cc_status_info WHERE plugin_id = ? AND monitor_id = ? AND facility_id = ?";
			// SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.pluginId);
			stmt.setString(2, pk.monitorId);
			stmt.setString(3, pk.facilityId);

			res = stmt.executeQuery();

			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log
						.error("find() error : " + pk.toString()
								+ " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}

	/**
	 * 引数で指定された値でSELECT文を発行します。<BR>
	 * 重要度が最高 かつ 受信日時が最新のデータを検索します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @param facilityId ファシリティIDの配列
	 * @param outputFromDate 最終変更日時（開始）
	 * @param outputToDate 最終変更日時（終了）
	 * @param generationFromDate 出力日時（開始）
	 * @param generationToDate 出力日時（終了）
	 * @param application アプリケーション
	 * @param message メッセージ
	 * @return プライマリキーのコレクション
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.notify.monitor.ejb.entity.StatusInfoBean#ejbFindHighPriorityStatus(String[], Timestamp, Timestamp, Timestamp, Timestamp, String, String)
	 * @see com.clustercontrol.notify.monitor.dao.StatusInfoDAO#findHighPriorityStatus(String[], Timestamp, Timestamp, Timestamp, Timestamp, String, String)
	 */
	public Collection findHighPriorityStatus(String[] facilityId,
			Timestamp outputFromDate, Timestamp outputToDate,
			Timestamp generationFromDate, Timestamp generationToDate,
			String application, String message) throws FinderException {
		return findHighPriorityStatus(facilityId, outputFromDate, outputToDate, generationFromDate,
				generationToDate, application, message, true);
	}
		
	/**
	 * 引数で指定された値でSELECT文を発行します。<BR>
	 * 重要度が最高 かつ 受信日時が最新のデータを検索します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @param facilityId ファシリティIDの配列
	 * @param outputFromDate 最終変更日時（開始）
	 * @param outputToDate 最終変更日時（終了）
	 * @param generationFromDate 出力日時（開始）
	 * @param generationToDate 出力日時（終了）
	 * @param application アプリケーション
	 * @param message メッセージ
	 * @param orderFlg 受信日時のソートをするか否か
	 * @return プライマリキーのコレクション
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.notify.monitor.ejb.entity.StatusInfoBean#ejbFindHighPriorityStatus(String[], Timestamp, Timestamp, Timestamp, Timestamp, String, String)
	 * @see com.clustercontrol.notify.monitor.dao.StatusInfoDAO#findHighPriorityStatus(String[], Timestamp, Timestamp, Timestamp, Timestamp, String, String)
	 */
	public Collection findHighPriorityStatus(String[] facilityId,
			Timestamp outputFromDate, Timestamp outputToDate,
			Timestamp generationFromDate, Timestamp generationToDate,
			String application, String message, boolean orderFlg) throws FinderException {
		m_log.debug("findHighPriorityStatus() start : ");

		ArrayList<StatusInfoPK> ret = new ArrayList<StatusInfoPK>();

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			// SQL文の定義
			StringBuffer sql = new StringBuffer();
			sql.append("SELECT facility_id,monitor_id,plugin_id FROM cc_status_info ");
			sql.append("WHERE ");

			// ファシリティID設定
			sql.append("facility_id IN (");
			for (int index = 0; index < facilityId.length; index++) {
				if (index != 0)
					sql.append(",");
				sql.append("?");
			}
			sql.append(")");

			// 受信日時（自）設定
			if (outputFromDate != null) {
				sql.append(" and output_date >= ?");
			}
			// 受信日時（至）設定
			if (outputToDate != null) {
				sql.append("and output_date <= ?");
			}
			// 出力日時（自）設定
			if (generationFromDate != null) {
				sql.append(" and generation_date >= ?");
			}
			// 出力日時（至）設定
			if (generationToDate != null) {
				sql.append(" and generation_date <= ?");
			}
			// アプリケーション設定
			if (application != null && !"".equals(application)) {
				sql.append(" and application like ?");
			}
			// メッセージ設定
			if (message != null && !"".equals(message)) {
				sql.append(" and message like ?");
			}
			
			if (orderFlg) {
				sql.append(" ORDER BY priority, output_date DESC LIMIT 1");
			} else {
				sql.append(" ORDER BY priority LIMIT 1");
			}
			
			// SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			int count = 1;
			// ファシリティID設定
			if (facilityId != null && facilityId.length > 0) {
				for (int index = 0; index < facilityId.length; index++) {
					stmt.setString(count, facilityId[index]);
					count++;
				}
			}
			// 受信日時（自）設定
			if (outputFromDate != null) {
				stmt.setTimestamp(count, outputFromDate);
				count++;
			}
			// 受信日時（至）設定
			if (outputToDate != null) {
				stmt.setTimestamp(count, outputToDate);
				count++;
			}
			// 出力日時（自）設定
			if (generationFromDate != null) {
				stmt.setTimestamp(count, generationFromDate);
				count++;
			}
			// 出力日時（至）設定
			if (generationToDate != null) {
				stmt.setTimestamp(count, generationToDate);
				count++;
			}
			// アプリケーション設定
			if (application != null && !"".equals(application)) {
				stmt.setString(count, "%" + application + "%");
				count++;
			}
			// メッセージ設定
			if (message != null && !"".equals(message)) {
				stmt.setString(count, "%" + message + "%");
				count++;
			}

			res = stmt.executeQuery();

			while (res.next()) {
				StatusInfoPK pk = new StatusInfoPK(res.getString("monitor_id"),
						res.getString("plugin_id"), res
								.getString("facility_id"));
				ret.add(pk);
			}

		} catch (SQLException e) {
			m_log.error("findHighPriorityStatus() error :  SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findHighPriorityStatus() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findHighPriorityStatus() end : ");
		return ret;
	}

	/**
	 * 引数で指定された値でSELECT文を発行します。<BR>
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @param facilityId ファシリティIDの配列
	 * @param priority 重要度
	 * @param outputFromDate 最終変更日時（開始）
	 * @param outputToDate 最終変更日時（終了）
	 * @param generationFromDate 出力日時（開始）
	 * @param generationToDate 出力日時（終了）
	 * @param application アプリケーション
	 * @param message メッセージ
	 * @return プライマリキーのコレクション
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.bean.PriorityConstant
	 * @see com.clustercontrol.notify.monitor.ejb.entity.StatusInfoBean#ejbFindStatus(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String)
	 * @see com.clustercontrol.notify.monitor.dao.StatusInfoDAO#findStatus(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String)
	 */
	public Collection findStatus(String[] facilityId, Integer priority,
			Timestamp outputFromDate, Timestamp outputToDate,
			Timestamp generationFromDate, Timestamp generationToDate,
			String application, String message) throws FinderException {
		m_log.debug("findStatus() start : ");

		ArrayList<StatusInfoPK> ret = new ArrayList<StatusInfoPK>();

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			// SQL文の定義
			StringBuffer sql = new StringBuffer();
			sql.append("SELECT facility_id,monitor_id,plugin_id FROM cc_status_info ");
			
			StringBuffer sqlWhere = new StringBuffer();

			// ファシリティID設定
			if(facilityId.length > 0){
				sqlWhere.append("WHERE facility_id IN (");
				for (int index = 0; index < facilityId.length; index++) {
					if (index != 0)
						sqlWhere.append(",");
					sqlWhere.append("?");
				}
				sqlWhere.append(")");
			}

			// 重要度設定
			if (priority != null) {
				if(sqlWhere.length() > 0)
					sqlWhere.append(" AND priority = ?");
				else
					sqlWhere.append(" WHERE priority = ?");
			}
			// 更新日時（自）設定
			if (outputFromDate != null) {
				if(sqlWhere.length() > 0)
					sqlWhere.append(" AND output_date >= ?");
				else
					sqlWhere.append(" WHERE output_date >= ?");
			}
			// 更新日時（至）設定
			if (outputToDate != null) {
				if(sqlWhere.length() > 0)
					sqlWhere.append(" AND output_date <= ?");
				else
					sqlWhere.append(" WHERE output_date <= ?");
			}
			// 出力日時（自）設定
			if (generationFromDate != null) {
				if(sqlWhere.length() > 0)
					sqlWhere.append(" AND generation_date >= ?");
				else
					sqlWhere.append(" WHERE generation_date >= ?");
			}
			// 出力日時（至）設定
			if (generationToDate != null) {
				if(sqlWhere.length() > 0)
					sqlWhere.append(" AND generation_date <= ?");
				else
					sqlWhere.append(" WHERE generation_date <= ?");
			}
			// アプリケーション設定
			if (application != null && !"".equals(application)) {
				if(sqlWhere.length() > 0)
					sqlWhere.append(" AND application like ?");
				else
					sqlWhere.append(" WHERE application like ?");
			}
			// メッセージ設定
			if (message != null && !"".equals(message)) {
				if(sqlWhere.length() > 0)
					sqlWhere.append(" AND message like ?");
				else
					sqlWhere.append(" WHERE message like ?");
			}
			
			sql.append(sqlWhere);

			// SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			int count = 1;
			// ファシリティID設定
			if (facilityId != null && facilityId.length > 0) {
				for (int index = 0; index < facilityId.length; index++) {
					stmt.setString(count, facilityId[index]);
					count++;
				}
			}
			// 重要度設定
			if (priority != null) {
				stmt.setInt(count, priority);
				count++;
			}
			// 受信日時（自）設定
			if (outputFromDate != null) {
				stmt.setTimestamp(count, outputFromDate);
				count++;
			}
			// 受信日時（至）設定
			if (outputToDate != null) {
				stmt.setTimestamp(count, outputToDate);
				count++;
			}
			// 出力日時（自）設定
			if (generationFromDate != null) {
				stmt.setTimestamp(count, generationFromDate);
				count++;
			}
			// 出力日時（至）設定
			if (generationToDate != null) {
				stmt.setTimestamp(count, generationToDate);
				count++;
			}
			// アプリケーション設定
			if (application != null && !"".equals(application)) {
				stmt.setString(count, "%" + application + "%");
				count++;
			}
			// メッセージ設定
			if (message != null && !"".equals(message)) {
				stmt.setString(count, "%" + message + "%");
				count++;
			}

			res = stmt.executeQuery();

			while (res.next()) {
				StatusInfoPK pk = new StatusInfoPK(res.getString("monitor_id"),
						res.getString("plugin_id"), res
								.getString("facility_id"));
				ret.add(pk);
			}

		} catch (SQLException e) {
			m_log.error("findStatus() error :  SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findStatus() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findStatus() end : ");
		return ret;
	}

	/**
	 * 引数で指定された値でSELECT文を発行します。<BR>
	 * 指定された日時より、存続日時が古いデータを検索します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @param expirationDate 存続期間日時
	 * @return プライマリキーのコレクション
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.notify.monitor.ejb.entity.StatusInfoBean#ejbFindExpirationStatus(Timestamp)
	 * @see com.clustercontrol.notify.monitor.dao.StatusInfoDAO#findExpirationStatus(Timestamp)
	 */
	public Collection findExpirationStatus(Timestamp expirationDate)
			throws FinderException {
		m_log.debug("findExpirationStatus() start : ");

		ArrayList<StatusInfoPK> ret = new ArrayList<StatusInfoPK>();

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			// SQL文の定義
			StringBuffer sql = new StringBuffer();
			sql.append("SELECT facility_id,monitor_id,plugin_id FROM cc_status_info ");
			sql.append("WHERE expiration_date <= ?");

			// SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setTimestamp(1, expirationDate);

			res = stmt.executeQuery();

			while (res.next()) {
				StatusInfoPK pk = new StatusInfoPK(res.getString("monitor_id"),
						res.getString("plugin_id"), res
								.getString("facility_id"));
				ret.add(pk);
			}

		} catch (SQLException e) {
			m_log.error("findExpirationStatus() error :  SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findExpirationStatus() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findExpirationStatus() end : ");
		return ret;
	}
}
