/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.run.factory;

import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.SchedulerException;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.MonitorNotFoundException;
import com.clustercontrol.NotifyNotFoundException;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.hinemosagent.bean.AgentCheckInfo;
import com.clustercontrol.http.bean.HttpCheckInfo;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorNumericValueInfo;
import com.clustercontrol.monitor.run.bean.MonitorStringValueInfo;
import com.clustercontrol.monitor.run.bean.MonitorTruthValueInfo;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoPK;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.monitor.run.ejb.entity.MonitorNumericValueInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorStringValueInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorTruthValueInfoLocal;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.ejb.session.NotifyControllerLocal;
import com.clustercontrol.notify.ejb.session.NotifyControllerUtil;
import com.clustercontrol.performance.monitor.bean.PerfCheckInfo;
import com.clustercontrol.ping.bean.PingCheckInfo;
import com.clustercontrol.port.bean.PortCheckInfo;
import com.clustercontrol.process.bean.ProcessCheckInfo;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.snmp.bean.SnmpCheckInfo;
import com.clustercontrol.sql.bean.SqlCheckInfo;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 監視情報を検索する抽象クラス<BR>
 * <p>
 * 監視種別（真偽値，数値，文字列）の各クラスで継承してください。
 *
 * @version 3.0.0
 * @since 2.0.0
 */
abstract public class SelectMonitor {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( SelectMonitor.class );
	
	/** 監視情報のローカルコンポーネント。 */
	protected MonitorInfoLocal m_monitor;
	
	/** 監視情報。 */
	protected MonitorInfo m_monitorInfo;
	
	/** 監視対象ID。 */
	protected String m_monitorTypeId;
	
	/** 監視項目ID。 */
	protected String m_monitorId;
	
	/**
	 * 引数で指定された監視情報を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定された監視情報を取得します。</li>
	 * <li>Quartzより、有効/無効を取得します。</li>
	 * <li>監視情報より判定情報を取得します。各監視種別（真偽値，数値，文字列）のサブクラスで実装します（{@link #getJudgementInfo()}）。</li>
	 * <li>監視情報よりチェック条件を取得します。各監視管理のサブクラスで実装します（{@link #getCheckInfo()}）。</li>
	 * </ol>
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @param monitorId 監視項目ID
	 * @return 監視情報
	 * @throws CreateException
	 * @throws SchedulerException
	 * @throws NamingException
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 * @throws MonitorNotFoundException 
	 * @see com.clustercontrol.monitor.run.ejb.entity.MonitorInfoBean
	 * @see com.clustercontrol.monitor.run.factory.SelectSchedule#getValid(String, String)
	 * @see #getJudgementInfo()
	 * @see #getCheckInfo()
	 */
	
	public MonitorInfo getMonitor(String monitorTypeId, String monitorId)
		throws CreateException, SchedulerException, NamingException, NotifyNotFoundException, HinemosUnknownException, MonitorNotFoundException {
		
		m_monitorTypeId = monitorTypeId;
		m_monitorId = monitorId;
		
		MonitorInfo bean = null;
		try 
		{
			// 監視情報を取得
			MonitorInfoPK pk = new MonitorInfoPK(m_monitorId, m_monitorTypeId);
			m_monitor = MonitorInfoUtil.getLocalHome().findByPrimaryKey(pk);
			
            // スコープの取得
            RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
			String facilityPath = repository.getFacilityPath(m_monitor.getFacilityId(), null);
			
			NotifyControllerLocal nc = NotifyControllerUtil.getLocalHome().create();
			Collection notifyId = nc.getNotifyRelation(m_monitor.getNotifyGroupId());
			
			
			// Quartzより有効/無効の取得
			int valid = new SelectSchedule().getValid(m_monitorTypeId, m_monitorId);
			
			bean = new MonitorInfo(
					m_monitor.getApplication(),
					m_monitor.getCalendarId(),
					m_monitor.getDescription(),
					m_monitor.getDeterminationId(),
					facilityPath,
					m_monitor.getFacilityId(),
					m_monitor.getFailurePriority(),
					m_monitor.getMonitorBlock(),
					m_monitorId,
					m_monitorTypeId,
					m_monitor.getMonitorType(),
					m_monitor.getRegDate().getTime(),
					m_monitor.getRegUser(),
					m_monitor.getRunInterval(),
					m_monitor.getTimeout(),
					m_monitor.getFailureMessageId(),
					m_monitor.getFailureMessage(),
					m_monitor.getNotifyGroupId(),
					notifyId,
					m_monitor.getUpdateDate().getTime(),
					m_monitor.getUpdateUser(),
					valid,
					getNumericValueInfo(),
					getStringValueInfo(),
					getTruthValueInfo(),
					getAgentCheckInfo(),
					getHttpCheckInfo(),
					getPerfCheckInfo(),
					getPingCheckInfo(),
					getPortCheckInfo(),
					getProcessCheckInfo(),
					getSnmpCheckInfo(),
					getSqlCheckInfo()
			
			);
		} catch (CreateException e) {
			outputLog(e, "getMonitor()", "010");
			throw e;
		} catch (FinderException e) {
			outputLog(e, "getMonitor()", "010");
			throw new MonitorNotFoundException(e.getMessage(), e);
		} catch (SchedulerException e) {
			outputLog(e, "getMonitor()", "010");
			throw e;
		} catch (NamingException e) {
			outputLog(e, "getMonitor()", "010");
			throw e;
		} 
		
		return bean;
	}
	
	/**
	 * 判定情報を返します。
	 * <p>
	 * @return 判定情報（{@link com.clustercontrol.monitor.run.bean.MonitorJudgementInfo}のリスト）
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList<MonitorNumericValueInfo> getNumericValueInfo() throws FinderException, NamingException {
		// 数値監視判定情報を取得
		Collection<MonitorNumericValueInfoLocal> ct = m_monitor.getMonitorNumericValueInfo();
		ArrayList<MonitorNumericValueInfo> valueList = new ArrayList<MonitorNumericValueInfo>();
		
		for (MonitorNumericValueInfoLocal local : ct) {
			MonitorNumericValueInfo value = new MonitorNumericValueInfo(
					local.getMessage(),
					local.getMessageId(),
					local.getMonitorId(),
					local.getMonitorTypeId(),
					local.getPriority(),
					local.getThresholdLowerLimit(),
					local.getThresholdUpperLimit());
			valueList.add(value);
		}
		return valueList;
	}
	
	/**
	 * 判定情報を返します。
	 * <p>
	 * @return 判定情報（{@link com.clustercontrol.monitor.run.bean.MonitorJudgementInfo}のリスト）
	 * @throws FinderException
	 * @throws NamingException
	 * @throws HinemosUnknownException
	 * @throws NotifyNotFoundException 
	 */
	public ArrayList<MonitorStringValueInfo> getStringValueInfo() throws FinderException, NamingException, HinemosUnknownException, NotifyNotFoundException {
		
		// 文字列監視判定情報を取得
		Collection<MonitorStringValueInfoLocal> ct = m_monitor.getMonitorStringValueInfo();
		
		ArrayList<MonitorStringValueInfo> valueList = new ArrayList<MonitorStringValueInfo>();
		
		try {
			
			//通知用のセッションビーンを初期化
			NotifyControllerLocal	nc = NotifyControllerUtil.getLocalHome().create();
			Collection<NotifyRelationInfo> notifyId;
			
			for (MonitorStringValueInfoLocal local : ct) {
				notifyId = nc.getNotifyRelation(local.getNotifyGroupId());
				
				MonitorStringValueInfo value = new MonitorStringValueInfo(
						local.getMonitorTypeId(),
						local.getMonitorId(),
						local.getOrderNo().intValue(),
						local.getDescription(),
						local.getProcessType().intValue(),
						local.getPattern(),
						local.getPriority().intValue(),
						local.getMessageId(),
						local.getMessage(),
						local.getNotifyGroupId(),
						notifyId,
						ValidConstant.typeToBoolean(local.getValidFlg().intValue())
				);
				valueList.add(value);
			}
			return valueList;
		} catch (CreateException e) {
			e.printStackTrace();
		}
		return null;
	}
	
	/**
	 * 判定情報を返します。
	 * <p>
	 * @return 判定情報（{@link com.clustercontrol.monitor.run.bean.MonitorJudgementInfo}のリスト）
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList<MonitorTruthValueInfo> getTruthValueInfo() throws FinderException, NamingException {
		// 真偽値監視判定情報を取得
		Collection<MonitorTruthValueInfoLocal> ct = m_monitor.getMonitorTruthValueInfo();
		
		ArrayList<MonitorTruthValueInfo> valueList = new ArrayList<MonitorTruthValueInfo>();
		
        for (MonitorTruthValueInfoLocal local : ct) {
            MonitorTruthValueInfo value = new MonitorTruthValueInfo(
            		local.getMessage(),
            		local.getMessageId(),
            		local.getMonitorId(),
            		local.getMonitorTypeId(),
            		local.getPriority(),
            		local.getTruthValue());
            valueList.add(value);
        }
		return valueList;
	}
	
	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	public AgentCheckInfo getAgentCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	public HttpCheckInfo getHttpCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	public PerfCheckInfo getPerfCheckInfo() throws FinderException, NamingException { 
		return null;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	public PingCheckInfo getPingCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	public PortCheckInfo getPortCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ProcessCheckInfo getProcessCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	public SnmpCheckInfo getSnmpCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	public SqlCheckInfo getSqlCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * 監視情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定された監視対象の監視情報を取得します。</li>
	 * <li>１監視情報をテーブルのカラム順（{@link com.clustercontrol.monitor.run.bean.MonitorTabelDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１監視情報を保持するリストを、監視情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>監視情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 監視情報1 {カラム1の値, カラム2の値, … }, 監視情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @return 監視情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws SchedulerException
	 * @throws NamingException
	 * @throws HinemosUnknownException 
	 * @throws MonitorNotFoundException 
	 * @throws NotifyNotFoundException 
	 * @see com.clustercontrol.monitor.run.bean.MonitorTabelDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList<MonitorInfo> getMonitorList(String monitorTypeId) throws CreateException, SchedulerException, NamingException, HinemosUnknownException, MonitorNotFoundException, NotifyNotFoundException {

		ArrayList<MonitorInfo> list = new ArrayList<MonitorInfo>();
		try 
		{
			// 監視情報一覧を取得
			Collection<MonitorInfoLocal> ct =
				MonitorInfoUtil.getLocalHome().findAll(monitorTypeId);
			
			for (MonitorInfoLocal info : ct) {
				list.add(getMonitor(info.getMonitorTypeId(), info.getMonitorId()));
			}
		} catch (CreateException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (FinderException e) {
			outputLog(e, "getMonitorList()", "011");
			throw new MonitorNotFoundException(e.getMessage(), e);
		} catch (SchedulerException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (NamingException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (HinemosUnknownException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (NotifyNotFoundException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		}
		return list;
	}
	
	
	/**
	 * アプリケーションログにログを出力します。
	 * 
	 * @param e 例外
	 * @param method メソッド名
	 * @param index アプリケーションログのインデックス
	 */
	private void outputLog(Exception e, String method, String index) {
		AplLogger apllog = new AplLogger("MON", "mon");
		String[] args = {m_monitorTypeId, m_monitorId };
        apllog.put("SYS", index, args);
        m_log.debug(method + ":" + e.getMessage());
	}
}
