/*
 
Copyright (C) since 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.commons.jmx;

import java.sql.Connection;
import java.sql.Statement;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import javax.management.ObjectName;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.jboss.jmx.adaptor.rmi.RMIAdaptor;
import org.jboss.system.ServiceMBeanSupport;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.util.apllog.AplLogger;

public class HinemosService extends ServiceMBeanSupport implements HinemosServiceMBean {

	/** DBMSスケジューラのJNDI名 */
	private String jndiNameDBMSScheduler = null;
	
	/** RAMスケジューラのJNDI名 */
	private String jndiNameRAMScheduler = null;
	
	/** スケジューラの開始遅延時間（起動処理に時間がかかる場合にスケジューラの開始を遅らせることが可能） */
	private int delay = 10;
	
	private static ScheduledExecutorService scheduler;
	
	/**
	 * HinemosServiceのコンストラクタ
	 */
	public HinemosService() {
		jndiNameDBMSScheduler = null;
		jndiNameRAMScheduler = null;
		
		scheduler = Executors.newSingleThreadScheduledExecutor(
				new ThreadFactory() {
					public Thread newThread(Runnable r) {
						return new Thread(r, "HinemosServiceScheduler");
					}
				}
		);
	}
	
	/**
	 * DBMSスケジューラのJNDI名を登録する
	 */
	public void setJndiNameDBMSScheduler(String jndiName) {
		jndiNameDBMSScheduler = jndiName;
	}
	
	/**
	 * 設定されているDBMSスケジューラのJNDI名を取得する
	 */
	public String getJndiNameDBMSScheduler() {
		return jndiNameDBMSScheduler;
	}
	
	/**
	 * RAMスケジューラのJNDI名を登録する
	 */
	public void setJndiNameRAMScheduler(String jndiName) {
		jndiNameRAMScheduler = jndiName;
	}
	
	/**
	 * 設定されているRAMスケジューラのJNDI名を取得する
	 */
	public String getJndiNameRAMScheduler() {
		return jndiNameRAMScheduler;
	}
	
	/**
	 * 設定されているスケジューラの状態を取得する
	 */
	public String printSchedulerStatus() throws Exception {
		/** ローカル変数 */
		InitialContext initCtx = null;
		Scheduler scheduler = null;
		String msg1 = null;
		String msg2 = null;
		
		/** メイン処理 */
		msg1 = "";
		msg2 = "";
		try {
			initCtx = new InitialContext();
			
			scheduler = (Scheduler)initCtx.lookup(jndiNameDBMSScheduler);
			msg1 = "status of DBMS scheduler : ";
			if (scheduler.isPaused()) {
				msg1 = msg1 + "[ PAUSED ]";
			} else if (scheduler.isShutdown()) {
				msg1 = msg1 + "[ DISABLED ]";
			} else {
				msg1 = msg1 + "[ RUNNING ]";
			}
			
			scheduler = (Scheduler)initCtx.lookup(jndiNameRAMScheduler);
			msg2 = "status of RAM scheduler  : ";
			if (scheduler.isPaused()) {
				msg2 = msg2 + "[ PAUSED ]";
			} else if (scheduler.isShutdown()) {
				msg2 = msg2 + "[ DISABLED ]";
			} else {
				msg2 = msg2 + "[ RUNNING ]";
			}
		} catch (NamingException e) {
			log.warn("failure to lookup scheduler.", e);
			throw e;
		}
		
		return msg1 + System.getProperty("line.separator") + msg2;
	}
	
	/**
	 * 設定されているスケジューラ情報を取得する
	 */
	public String printSchedulerDetail() throws Exception {
		/** ローカル変数 */
		InitialContext initCtx = null;
		Scheduler scheduler = null;
		Trigger trigger = null;
		
		String startFireTime = null;
		String prevFireTime = null;
		String nextFireTime = null;
		int state = 0;
		String msg = null;
		
		String lineSeparator = null;

		/** メイン処理 */
		lineSeparator = System.getProperty("line.separator");
		msg = "";
		try {
			initCtx = new InitialContext();
			
			msg = msg + lineSeparator;
			msg = msg + "- DBMS Scheduler - " + lineSeparator;
			msg = msg + lineSeparator;
			scheduler = (Scheduler)initCtx.lookup(jndiNameDBMSScheduler);
			for (String triggerGroupName : scheduler.getTriggerGroupNames()) {
				for (String triggerName : scheduler.getTriggerNames(triggerGroupName)) {
					trigger = scheduler.getTrigger(triggerName, triggerGroupName);
					state = scheduler.getTriggerState(triggerName, triggerGroupName);
					startFireTime = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.getStartTime());
					prevFireTime = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.getPreviousFireTime());
					nextFireTime = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.getNextFireTime());
					
					msg = msg + String.format("%s (in %s) :", triggerName, triggerGroupName) + lineSeparator;
					msg = msg + String.format("   start fire time - %s", startFireTime) + lineSeparator;
					msg = msg + String.format("   last fire time  - %s", prevFireTime) + lineSeparator;
					msg = msg + String.format("   next fire time  - %s", nextFireTime) + lineSeparator;
					msg = msg + String.format("   current state   - %s", state == Trigger.STATE_PAUSED ? "PAUSED" : "ACTIVE (not PAUSED)") + lineSeparator;
					msg = msg + lineSeparator;
				}
			}
			
			msg = msg + lineSeparator;
			msg = msg + "- RAM Scheduler -" + lineSeparator;
			msg = msg + lineSeparator;
			scheduler = (Scheduler)initCtx.lookup(jndiNameRAMScheduler);
			for (String triggerGroupName : scheduler.getTriggerGroupNames()) {
				for (String triggerName : scheduler.getTriggerNames(triggerGroupName)) {
					trigger = scheduler.getTrigger(triggerName, triggerGroupName);
					state = scheduler.getTriggerState(triggerName, triggerGroupName);
					startFireTime = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.getStartTime());
					prevFireTime = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.getPreviousFireTime());
					nextFireTime = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.getNextFireTime());
					
					msg = msg + String.format("%s (in %s) :", triggerName, triggerGroupName) + lineSeparator;
					msg = msg + String.format("   start fire time - %s", startFireTime) + lineSeparator;
					msg = msg + String.format("   last fire time  - %s", prevFireTime) + lineSeparator;
					msg = msg + String.format("   next fire time  - %s", nextFireTime) + lineSeparator;
					msg = msg + String.format("   current state   - %s", state == Trigger.STATE_PAUSED ? "PAUSED" : "ACTIVE (not PAUSED)") + lineSeparator;
					msg = msg + lineSeparator;
				}
			}
			
		} catch (NamingException e) {
			log.warn("failure to lookup scheduler.", e);
			throw e;
		}
		
		return msg;
	}
	
	/**
	 * 通知抑制の履歴情報（最終重要度および通知日時）をリセットする
	 */
	public void resetNotificationLogger() throws Exception {
		/** ローカル変数 */
		Connection con = null;
		Statement stmt = null;
		InitialContext ic = new InitialContext();
		
		/** メイン処理 */
		log.info("resetting notification counter...");
		
		// DB内の抑制情報を初期化
		con = ConnectionManager.getConnectionManager().getConnection();
		con.setAutoCommit(false);
		stmt = con.createStatement();
		
		log.info("clean up cc_monitor_statue table (latest priority)...");
		stmt.executeUpdate("TRUNCATE TABLE cc_monitor_status");
		log.info("clean up cc_notify_history table (latest date)...");
		stmt.executeUpdate("TRUNCATE TABLE cc_notify_history");
		
		con.commit();
		stmt.close();
		con.close();
		
		// EJB Container内の抑制情報を初期化
		RMIAdaptor server = (RMIAdaptor) ic.lookup("jmx/invoker/RMIAdaptor");
		
		log.info("flush MonitorStatusLocal cache (latest priority)...");
		server.invoke(new ObjectName("jboss.j2ee:jndiName=MonitorStatusLocal,service=EJB"), "flushCache", null, null);
		log.info("flush NotifyHistoryLocal cache (latest date)...");
		server.invoke(new ObjectName("jboss.j2ee:jndiName=NotifyHistoryLocal,service=EJB"), "flushCache", null, null);
		
		log.info("notify counter resetted successfully.");
	}
	
	/**
	 * MBeanの初期化処理
	 */
	@Override
	public void createService() throws Exception {
		/** メイン処理 */
		log.info("creating a MBean of HinemosService...");
		log.info("successful in creating a MBean of HinemosService...");
	}
	
	/**
	 * MBeanの開始処理
	 */
	@Override
	public void startService() throws Exception {
		/** ローカル変数 */
		String mode = null;
		
		/** メイン処理 */
		log.info("starting a MBean of HinemosService...");
		mode = System.getProperty("hinemos.manager.mode");
		if (mode != null && mode.equals("maintenance")) {
			log.info("do nothing, because of maintenance mode.");
		} else {
			startScheduler(delay);
		}
		
		AplLogger apllog = new AplLogger("MNG", "mng");
		String[] args = {"start"};
		apllog.put("SYS", "001", args);
		
		log.info("successful in starting a MBean of HinemosService...");
	}
	
	/**
	 * MBeanの停止処理
	 */
	@Override
	public void stopService() throws Exception {
		/** メイン処理 */
		log.info("stopping a MBean of HinemosService...");
		shutdownScheduler();
		
		AplLogger apllog = new AplLogger("MNG", "mng");
		String[] args = {"shutdown"};
		apllog.put("SYS", "002", args);
		
		log.info("successful in stopping a MBean of HinemosService...");
	}
	
	/**
	 * MBeanの削除処理
	 */
	@Override
	public void destroyService() throws Exception {
		/** メイン処理 */
		log.info("destroying a MBean of HinemosService...");
		jndiNameDBMSScheduler = null;
		jndiNameRAMScheduler = null;
		log.info("successful in destroying a MBean of HinemosService...");
	}
	
	/**
	 * 全スケジューラを開始する
	 */
	public void startScheduler() throws Exception {
		/** メイン処理 */
		startScheduler(0);
	}
	
	/**
	 * 指定時間後に全スケジューラを開始する
	 */
	public void startScheduler(int delay) throws Exception {
		/** メイン処理 */
		log.info("start scheduler after " + delay + " second...");
		scheduler.schedule(
				new HinemosServiceStarter(this), 
				delay * 1000, 
				TimeUnit.MILLISECONDS
		);
	}
	
	/**
	 * DBMSスケジューラを開始する
	 */
	public void startDBMSScheduler() throws Exception {
		/** メイン処理 */
		log.info("starting a scheduler using DBMS...");
		startScheduler(jndiNameDBMSScheduler);
		log.info("successful in starting a scheduler using DBMS...");
	}
	
	/**
	 * RAMスケジューラを開始する
	 */
	public void startRAMScheduler() throws Exception {
		/** メイン処理 */
		log.info("starting a scheduler using RAM...");
		startScheduler(jndiNameRAMScheduler);
		log.info("successful in starting a scheduler using RAM...");
	}
	
	/**
	 * 全スケジューラを一時停止する
	 */
	public void pauseScheduler() throws Exception {
		/** メイン処理 */
		pauseDBMSScheduler();
		pauseRAMScheduler();
	}
	
	/**
	 * DBMSスケジューラを一時停止する
	 */
	public void pauseDBMSScheduler() throws Exception {
		/** メイン処理 */
		log.info("pausing a scheduler using DBMS...");
		pauseScheduler(jndiNameDBMSScheduler);
		log.info("successful in pausing a scheduler using DBMS...");
	}
	
	/**
	 * RAMスケジューラを一時停止する
	 */
	public void pauseRAMScheduler() throws Exception {
		/** メイン処理 */
		log.info("pausing a scheduler using RAM...");
		pauseScheduler(jndiNameRAMScheduler);
		log.info("successful in pausing a scheduler using RAM...");
	}
	
	/**
	 * 全スケジューラを停止する
	 */
	public void shutdownScheduler() throws Exception {
		/** メイン処理 */
		shutdownDBMSScheduler();
		shutdownRAMScheduler();
	}
	
	/**
	 * DBMSスケジューラを停止する
	 * @throws Exception
	 */
	public void shutdownDBMSScheduler() throws Exception {
		/** メイン処理 */
		log.info("shutting down a scheduler using DBMS...");
		shutdownScheduler(jndiNameDBMSScheduler);
		log.info("successful in shutting down a scheduler using DBMS...");
	}
	
	/**
	 * RAMスケジューラを停止する
	 * @throws Exception
	 */
	public void shutdownRAMScheduler() throws Exception {
		/** メイン処理 */
		log.info("shutting down a scheduler using RAM...");
		shutdownScheduler(jndiNameRAMScheduler);
		log.info("successful in shutting down a scheduler using RAM...");
	}
	
	/**
	 * スケジューラを開始する
	 * @param jndiNameSchduler スケジューラのJNDI名
	 * @throws NamingException
	 * @throws SchedulerException
	 */
	private void startScheduler(String jndiNameSchduler) throws NamingException, SchedulerException {
		/** ローカル変数 */
		InitialContext initCtx = null;
		Scheduler scheduler = null;
		
		/** メイン処理 */
		try {
			initCtx = new InitialContext();
			scheduler = (Scheduler)initCtx.lookup(jndiNameSchduler);
			
			scheduler.start();
		} catch (NamingException e) {
			log.warn("failure to lookup scheduler. (jndiName = " + jndiNameSchduler + "", e);
			throw e;
		} catch (SchedulerException e) {
			log.warn("failure to start scheduler. (jndiName = " + jndiNameSchduler + "", e);
			throw e;
		}
	}
	
	/**
	 * スケジューラを一時停止する
	 * @param jndiNameSchduler スケジューラのJNDI名
	 * @throws NamingException
	 * @throws SchedulerException
	 */
	private void pauseScheduler(String jndiNameSchduler) throws NamingException, SchedulerException {
		/** ローカル変数 */
		InitialContext initCtx = null;
		Scheduler scheduler = null;
		
		/** メイン処理 */
		try {
			initCtx = new InitialContext();
			scheduler = (Scheduler)initCtx.lookup(jndiNameSchduler);
			
			scheduler.pause();
		} catch (NamingException e) {
			log.warn("failure to lookup scheduler. (jndiName = " + jndiNameSchduler + "", e);
			throw e;
		} catch (SchedulerException e) {
			log.warn("failure to pause scheduler. (jndiName = " + jndiNameSchduler + "", e);
			throw e;
		}
	}
	
	/**
	 * スケジューラを停止する
	 * @param jndiNameSchduler スケジューラのJNDI名
	 * @throws NamingException
	 * @throws SchedulerException
	 */
	private void shutdownScheduler(String jndiNameSchduler) throws NamingException, SchedulerException {
		/** ローカル変数 */
		InitialContext initCtx = null;
		Scheduler scheduler = null;
		
		/** メイン処理 */
		try {
			initCtx = new InitialContext();
			scheduler = (Scheduler)initCtx.lookup(jndiNameSchduler);
			
			scheduler.shutdown();
		} catch (NamingException e) {
			log.warn("failure to lookup scheduler. (jndiName = " + jndiNameSchduler + "", e);
			throw e;
		} catch (SchedulerException e) {
			log.warn("failure to shutdown scheduler. (jndiName = " + jndiNameSchduler + "", e);
			throw e;
		}
	}

	public int getDelay() {
		return delay;
	}

	public void setDelay(int delay) {
		this.delay = delay;
	}
}
