/*
 
Copyright (C) 2009 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.util;

import java.io.IOException;
import java.rmi.RemoteException;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
//
import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.OutputBasicInfo;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.ejb.entity.NotifyCommandInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyCommandInfoPK;
import com.clustercontrol.notify.ejb.entity.NotifyCommandInfoUtil;
import com.clustercontrol.util.CommonCreateCommands;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * コマンドを実行するクラス<BR>
 *
 * @version 3.0.0
 * @since 3.0.0
 */
public class ExecCommand implements Notifier {
	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( ExecCommand.class );
	
	private static final String THREAD_POOL_COUNT_KEY="common.notify.command.thread.pool.count";
	private static final String THREAD_POOL_COUNT_DEFAULT="10";
	
	private static final String COMMAND_CREATE_MODE="common.notify.command.create.mode";
	
	// コマンド実行にタイムアウトを設けるため、2段のExecutorの構成とする
	// 1.「コマンドを実行するスレッド（CommandTask）」を実行するExecutor
	//   （スレッド起動後直ぐに制御を戻す）
	// 2.コマンドを実行するスレッド（CommandTask）用のExecutor
	//   （スレッド起動後、コマンドの実行終了もしくは、タイムアウトを待つ）
	
	// 「コマンドを実行するスレッド（CommandTask）」を実行し、その終了を待つスレッド（CommandCallerTask）用の
	// スレッドプールを保持するExecutorService
	private static final ExecutorService _callerExecutorService;

	// コマンドを実行するスレッド（CommandTask）用のスレッドプールを保持するExecutorService
	private static final ExecutorService _commandExecutorService;
	
	// ジョブを起こす（Runtime.execする）際にシリアル化するための専用ロックオブジェクト
	private static final Object lockNewProcess = new Object();
	
	// コマンド作成モード（OS毎に異なる）
	private static final String _mode;
	
	static {
		// hinemos.propertiesからスレッドプール数を取得する
		int threadPoolSize = Integer.parseInt(HinemosProperties.getProperty(
				THREAD_POOL_COUNT_KEY, THREAD_POOL_COUNT_DEFAULT));

		// コマンド作成モードをプロパティから取得する
		_mode = HinemosProperties.getProperty(COMMAND_CREATE_MODE, "");
		
		// 「コマンドを実行するスレッド（CommandTask）」を実行し、その終了を待つスレッド（CommandCallerTask）用の
		// スレッドプールを保持するExecutorService
		_callerExecutorService = Executors.newFixedThreadPool(threadPoolSize,
				new CommandTaskThreadFactory());
		
		// コマンドを実行するスレッド（CommandTask）用のスレッドプールを保持するExecutorService
		_commandExecutorService = Executors.newFixedThreadPool(threadPoolSize,
				new CommandCallerTaskThreadFactory());
	}
	
	/**
	 * 指定されたコマンドを呼出します。
	 * 
	 * @param outputInfo 通知・抑制情報
	 * @throws RemoteException
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FinderException
	 */
	public synchronized void notify(NotifyRequestMessage message)
			throws RemoteException, NamingException, CreateException, FinderException {	
		if(m_log.isDebugEnabled()){
			m_log.debug("notify() " + message); 
		}
	
		executeCommand(message.getOutputInfo(), message.getNotifyId());
	}
	
	// 文字列を置換する
	public String getCommandString(OutputBasicInfo outputInfo, String beforeCommand){
		// 文字列を置換する
		try {
			return TextReplacer.substitution(beforeCommand, outputInfo);
		} catch (Exception e) {
			m_log.error(e.getMessage(), e);
			
			// 例外が発生した場合は、置換前の文字列を返す
			return beforeCommand;
		}
	}

	/**
	 * 指定されたコマンドを呼出します。
	 * 
	 * @param outputInfo 通知・抑制情報
	 * @throws RemoteException
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FinderException
	 */
	public synchronized void executeCommand(
			OutputBasicInfo outputInfo,
			String notifyId
	) {	
		if(m_log.isDebugEnabled()){
			m_log.debug("notify() " + outputInfo);
		}

		NotifyCommandInfoLocal commandInfo;
		try {
			commandInfo = NotifyCommandInfoUtil.getLocalHome().findByPrimaryKey(
					new NotifyCommandInfoPK(notifyId, outputInfo.getPriority()));
			

			// 「commandInfo.getCommand()」と「command」の違いに注意が必要。
			// 「commandInfo.getCommand()」は、設定時の実行コマンドで、
			// TextReplacerによる文字列置換前の実行コマンド
			// 「command」は、実際に実行対象のコマンド文字列
			String command = getCommandString(outputInfo, commandInfo.getCommand());

			/**
			 * 実行
			 */
			// 起動ユーザ名取得
			String sysUserName = System.getProperty("user.name");
			String effectiveUser = commandInfo.getEffectiveUser();
			boolean setEnvFlag;
			if(commandInfo.getSetEnvironment() == ValidConstant.TYPE_VALID){
				setEnvFlag = true;
			} else {
				setEnvFlag = false;
			}
			long commadTimeout = commandInfo.getTimeout();

			// JBossの起動ユーザがroot以外の場合で、
			// 起動ユーザとコマンド実行ユーザが異なる場合は、コマンド実行しない
			if ((!sysUserName.equals("root")) && (!sysUserName.equals(effectiveUser))) {
				// 起動失敗
				String detailMsg = "The execution user of the command and jboss's user are different.";
				m_log.error(detailMsg);
				internalErrorNotify(notifyId, "007", detailMsg);
				return;
			} else {
				m_log.debug("NotifyCommand Submit : " + outputInfo + " command=" + command);

				// 並列でコマンド実行
				_callerExecutorService.submit(
						new CommandCallerTask(
								effectiveUser,
								command,
								setEnvFlag,
								outputInfo,
								commadTimeout));
			}
		} catch (FinderException e) {
			String detailMsg = e.getMessage();
			m_log.error(detailMsg, e);
			internalErrorNotify(notifyId, "007", detailMsg);
		} catch (NamingException e) {
			String detailMsg = e.getMessage();
			m_log.error(detailMsg, e);
			internalErrorNotify(notifyId, "007", detailMsg);
		}
	}

	// 並列でコマンドを実行するためのクラス
	class CommandCallerTask implements Callable<Long> {
		// 実効ユーザ
		private final String _effectiveUser;
		
		// 実行するコマンド
		private final String _execCommand;
		
		// ユーザを切り替える際にユーザで設定されているログイン時のスタートアップファイルを読ませるか否かのフラグ
		// 具体的には、suコマンド実行時に引数に「-」を加えるか否か
		private final boolean _setEnvFlag;
		
		private final OutputBasicInfo _outputInfo;
		
		private final long _commadTimeout;
		
		public CommandCallerTask(
				String effectiveUser,
				String execCommand,
				boolean setEnvFlag,
				OutputBasicInfo outputInfo,
				long commadTimeout) {
			_effectiveUser = effectiveUser;
			_execCommand = execCommand;
			_setEnvFlag = setEnvFlag;
			_outputInfo = outputInfo;
			_commadTimeout = commadTimeout;
		}

		/**
		 * CommandTaskを実行しその終了（もしくはタイムアウト）まで待つ処理を実行します
		 */
		public Long call() throws Exception {
			// 初期値（コマンドが時間内に終了せずリターンコードが取得できない場合は、この値が返る）
			long returnValue = Long.MIN_VALUE;
			
			Future<Long> future = _commandExecutorService.submit(
					new CommandTask(
							_effectiveUser,
							_execCommand,
							_setEnvFlag,
							_outputInfo));
			
			try {
				// コマンド実行後ここで指定されている時間が経過しても終了しない場合は強制終了する
				returnValue = future.get(_commadTimeout, TimeUnit.MILLISECONDS);
			} catch (InterruptedException e) {
				m_log.error(e.getMessage(), e);
			} catch (ExecutionException e) {
				m_log.error(e.getMessage(), e);
			} catch (TimeoutException e) {
				// タイムアウトするのは仕様内の動作で異常系ではないため、ログレベルを低く出力する
				m_log.debug(e.getMessage(), e);
			} catch (Exception e) {
				m_log.error(e.getMessage(), e);
			} finally {
				//　タスクを終了させる（既に終了している場合は何もしない）
				future.cancel(true);
			}
			
			return returnValue;
		}
	}
	
	// 並列でコマンドを実行するためのクラス
	class CommandTask implements Callable<Long> {
		// 実効ユーザ
		private final String _effectiveUser;
		
		// 実行するコマンド
		private final String _execCommand;
		
		// ユーザを切り替える際にユーザで設定されているログイン時のスタートアップファイルを読ませるか否かのフラグ
		// 具体的には、suコマンド実行時に引数に「-」を加えるか否か
		private final boolean _setEnvFlag;
		
		private final OutputBasicInfo _outputInfo;
		
		public CommandTask(
				String effectiveUser,
				String execCommand,
				boolean setEnvFlag,
				OutputBasicInfo outputInfo) {
			_effectiveUser = effectiveUser;
			_execCommand = execCommand;
			_setEnvFlag = setEnvFlag;
			_outputInfo = outputInfo;
		}

		/**
		 * Executorからコールされ、コマンドを実行します
		 * @throws IOException 
		 * @throws InterruptedException 
		 */
		public Long call() throws Exception {
			String cmd[] = null;
			CommonCreateCommands createCommands = new CommonCreateCommands();

			if ("windows".equals(_mode)) {
				cmd = createCommands.createCommands(_effectiveUser,
						_execCommand, CommonCreateCommands.WINDOWS);
			} else if ("unix".equals(_mode)) {
				cmd = createCommands.createCommands(_effectiveUser,
						_execCommand, CommonCreateCommands.UNIX);
			} else if ("compatible".equals(_mode)) {
				cmd = createCommands.createCommands(_effectiveUser,
						_execCommand, CommonCreateCommands.COMPATIBLE);
			}
			// default で プラットフォームの自動識別を行う
			else {
				cmd = createCommands.createCommands(_effectiveUser,
						_execCommand, CommonCreateCommands.AUTO);
			}
			
			Process process = null;
			ProcessBuilder builder = new ProcessBuilder(cmd);
			
			try {
				m_log.info("NotifyCommand Start : " + _outputInfo + " command=" + _execCommand);

				// JVM（Windows環境）の不具合（？）でRuntime.execの並列実行多重度を上げた
				// 場合に問題が発生することがあるため、同時にただ１つのスレッドのみがexecを実行
				// できるように、staticなロックオブジェクトのsynchronizedによってシリアル化する
				synchronized(lockNewProcess) {
					process = builder.start();
				}

				if (process != null) {
					// コマンド実行待機
					int exitValue = process.waitFor();
					m_log.debug("NotifyCommand End   : " + _outputInfo + " command=" + _execCommand + " ExitCode=" + exitValue);
					return (long)exitValue;
				}
			} catch (IOException e) {
				m_log.error("NotifyCommand Error : " + _outputInfo + " command=" + _execCommand);
				throw e;
			} catch (InterruptedException e) {
				// 強制終了する必要があるが、finally句ないで処理されるため、ここでは何もしない。
				String log = "NotifyCommand Interrupted : " + _outputInfo + " command=" + _execCommand;
				m_log.warn(log);
				throw e;
			} finally {
				if (process != null) {
					// プロセスが終了していない場合は、強制終了
					process.destroy();
				}
			}
			return Long.MIN_VALUE;
		}
	}
	
	// CommandCallerTaskのスレッドに名前を定義するためFactoryを実装
	private static class CommandCallerTaskThreadFactory implements ThreadFactory {
		private volatile int _count = 0;
		
		public Thread newThread(Runnable r) {
			return new Thread(r, "NotifyCommandCallerTask-" + _count++);
		}
	}
	
	// CommandTaskのスレッドに名前を定義するためFactoryを実装
	private static class CommandTaskThreadFactory implements ThreadFactory {
		private volatile int _count = 0;
		
		public Thread newThread(Runnable r) {
			return new Thread(r, "NotifyCommandTask-" + _count++);
		}
	}

	/**
	 * 通知失敗時の内部エラー通知を定義します
	 */
	public void internalErrorNotify(String notifyId, String msgID, String detailMsg) {
		AplLogger apllog = new AplLogger("NOTIFY", "notify");
		String[] args = { notifyId };
		// 通知失敗メッセージを出力
		apllog.put("SYS", msgID, args, detailMsg);
	}
}
