/*
 
 Copyright (C) 2009 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.notify.queue;

import java.util.Collection;
import java.util.concurrent.ConcurrentHashMap;

import javax.ejb.FinderException;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.ejb.entity.NotifyTypeMstLocal;
import com.clustercontrol.notify.ejb.entity.NotifyTypeMstUtil;

/**
 * ログ出力情報（通知グループIDと通知IDとログ出力基本情報）を順次NotifyExecutorを利用して通知処理します。
 * Message-Driven Bean 実装を利用する場合の SendQueue に相当します。
 */
public class NotifyQueueSender {
	private static Log m_log = LogFactory.getLog( NotifyQueueSender.class );
	
	// 通知種別毎にインスタンスを保持
	// getInstance()呼び出しの度に新規にインスタンスを生成しなくてもよいように保持する
	private static volatile ConcurrentHashMap<Integer, NotifyQueueSender> _instanceMap = null;
	
	// _instanceMapインスタンスへの書き込みを制御するロック
	private static final Object _instanceMapWriteLock = new Object();
	
	// キュー名
	private final String _queueName;
	
	// JMSを利用してメッセージ送信するか否かのフラグ
	private final boolean _useJms;
	
	// 送信先のQueue
	private volatile SendQueue _queue = null;
	private static final Object queueLock = new Object();
	
	/**
	 * 通知種別とキュー名の対応マップを初期化します
	 * @throws NotifyQueueException 
	 */
	public static void init() throws NotifyQueueException {
		synchronized (_instanceMapWriteLock) {
			if(_instanceMap != null){
				return;
			}
			
			// マップのインスタンスを生成する。一度生成すると再度生成されることはない。
			_instanceMap = new ConcurrentHashMap<Integer, NotifyQueueSender>();
			
			try {
				m_log.debug("init");
				// DBからパラーメタを取得する
				Collection<NotifyTypeMstLocal> msts = NotifyTypeMstUtil.getLocalHome().findAll();
				// キュー名やJMS利用の設定を行う
				for (NotifyTypeMstLocal mst : msts) {
					int notifyType = mst.getNotifyType();
					String queueName = mst.getQueueName();
					boolean useJms = false;
					if (mst.getUseJms() == ValidConstant.TYPE_VALID) {
						useJms = true;
					}
					// インスタンスを生成しマップに登録する
					// ここで登録されたインスタンスが使いまわされる
					_instanceMap.put(notifyType, new NotifyQueueSender(notifyType, queueName, useJms));
				}
			} catch (FinderException e) {
				m_log.error(e.getMessage(), e);
				throw new NotifyQueueException("initialize faild. ", e);
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
				throw new NotifyQueueException("initialize faild. ", e);
			}
		}
	}
	/**
	 * インスタンスを生成します。
	 * @param queueName キュー名
	 * @throws NotifyQueueException 
	 */
	public static NotifyQueueSender getInstance(int notifyType) throws NotifyQueueException{
		// 通知種別毎に生成され、マップに登録されているインスタンスを返す
		// マップがまだ生成されていない場合は、生成する
		synchronized (_instanceMapWriteLock) {
			if(_instanceMap == null){
				NotifyQueueSender.init();
			}
		}
		
		return _instanceMap.get(notifyType);
	}
	
	// このクラスのインスタンスは外部から生成できない
	private NotifyQueueSender(int notifyType, String queueName, boolean useJms){
		_queueName = queueName;
		_useJms = useJms;
		
		try {
			synchronized (queueLock) {
				_queue = new SendQueue(_queueName);
			}
		} catch (Exception e) {
			m_log.warn("queue sender initialize failure.", e);
		}
	}
	
	/**
	 * メッセージを NotifyExecutor の該当キューに挿入します
	 * @param info ログ出力情報（通知グループIDと通知IDとログ出力基本情報）
	 * @throws Exception
	 */
	public void send(NotifyRequestMessage info, Integer deliveryMode) throws NotifyQueueException {
		m_log.debug("send() start. " + "Queue=" + _queueName	+ " " + info);
		
		try {

			try{
				// JMSを利用してメッセージ送信（SendQueueはThreadsafeでないため、排他制御）
				synchronized (queueLock) {
					_queue.put(info, deliveryMode);
				}
			} catch (Exception e) {
				m_log.warn("send queue failure.", e);
				synchronized (queueLock) {
					_queue = new SendQueue(_queueName);
					_queue.put(info, deliveryMode);
				}
			}

		} catch (Exception e) {
			new NotifyQueueException("message send failed. " + info, e);
		}
		m_log.debug("send() end.   " + "Queue=" + _queueName	+ " " + info);
	}
}
