/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.ejb.session;

import java.rmi.RemoteException;
import java.sql.SQLException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.SchedulerException;
import org.quartz.jobs.ee.ejb.EJBInvokerJob;

import com.clustercontrol.FacilityNotFoundException;
import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.ViewListInfo;
import com.clustercontrol.monitor.bean.QuartzConstant;
import com.clustercontrol.monitor.factory.DeleteStatus;
import com.clustercontrol.monitor.factory.EventBatchConfirmProperty;
import com.clustercontrol.monitor.factory.EventFilterProperty;
import com.clustercontrol.monitor.factory.EventInfoProperty;
import com.clustercontrol.monitor.factory.ManageStatus;
import com.clustercontrol.monitor.factory.ModifyEventConfirm;
import com.clustercontrol.monitor.factory.SelectEvent;
import com.clustercontrol.monitor.factory.SelectScope;
import com.clustercontrol.monitor.factory.SelectStatus;
import com.clustercontrol.monitor.factory.StatusFilterProperty;
import com.clustercontrol.quartzmanager.ejb.session.QuartzManager;
import com.clustercontrol.quartzmanager.util.QuartzUtil;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;

/**
 * 監視管理機能の管理を行う Session Bean です。<BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 * 
 * <!-- begin-xdoclet-definition --> 
 * @ejb.bean name="MonitorController"	
 *           jndi-name="MonitorController"
 *           type="Stateless" 
 *           transaction-type="Container"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 * <!-- end-xdoclet-definition --> 
 */
public abstract class MonitorControllerBean implements javax.ejb.SessionBean {
	
	/** ファシリティの配下全てのエントリ。 */
	public static final int ALL = RepositoryControllerBean.ALL;
	/** ファシリティの直下のエントリ。 */
	public static final int ONE_LEVEL = RepositoryControllerBean.ONE_LEVEL;
	/** ファシリティの自エントリ。 */
	public static final int ONLY = -1;
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( MonitorControllerBean.class );

	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
     * Quartzへステータスの定周期管理を登録します。<BR><BR>
     * 
     * このメソッドは監視管理のステータスが有効期限切れになっているかを
     * 定期的にチェックするトリガーをQurtzに設定します。<BR>
     * このメソッドをQurtzSetup以外から呼び出さないでください。
     * 
     * @ejb.interface-method
     * 
     * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
     * 
     * @param cronString Quartzへ登録するスケジュール文字列 （例：5分周期 "0 *<code>/</code>5 * * * ? *"）
     * @throws NamingException
     * @throws ParseException
     * @throws SchedulerException
     * 
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#addSchedule(org.quartz.JobDetail, org.quartz.Trigger)
     * @see #manageStatus()
     * 
     * @since 2.0.0
     */
    public void addQuartz(String cronString) throws NamingException, ParseException, SchedulerException{
        //QuartzのSchedulerをルックアップ
		QuartzManager manager = null;
		try {
			manager = QuartzUtil.getQuartzManager();
		} catch (NamingException e) {
			m_log.warn("addQuartz() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
		
        try {
            //ジョブ削除
        	manager.deleteSchedule(QuartzConstant.JNDI_NAME_MON, 
            		QuartzConstant.GROUP_NAME_MON);
        } catch (SchedulerException e) {
			m_log.warn("addQuartz() deleteJob : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
        } catch (RemoteException e) {
			m_log.warn("addQuartz() deleteJob : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
		} catch (NamingException e) {
			m_log.warn("addQuartz() deleteJob : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
        
        //JobDetail作成
        JobDetail job = new JobDetail(
        		QuartzConstant.JNDI_NAME_MON, 
        		QuartzConstant.GROUP_NAME_MON,
        		EJBInvokerJob.class);
        
        String serverName = System.getProperty("jboss.server.name");
	    if(serverName.equals("all")){
	        job.getJobDataMap().put(
	                EJBInvokerJob.PROVIDER_URL, 
	                "jnp://localhost:1100");
	    }
	    
        //ジョブ完了時に削除されないようにする。
        job.setDurability(true);
        
        //JobDetailに呼び出すクラスとメソッドを設定
        job.getJobDataMap().put(
        		EJBInvokerJob.EJB_JNDI_NAME_KEY, 
                QuartzConstant.JNDI_NAME_MON);
        job.getJobDataMap().put(
        		EJBInvokerJob.EJB_METHOD_KEY, 
				QuartzConstant.METHOD_NAME_STATUS);
        
        //CronTriggerを作成
        CronTrigger cronTrigger = new CronTrigger(QuartzConstant.JNDI_NAME_MON, QuartzConstant.GROUP_NAME_MON);
        
        //起動失敗した場合は、次の起動予定時刻をセットするように設定
        cronTrigger.setMisfireInstruction(CronTrigger.MISFIRE_INSTRUCTION_DO_NOTHING);
        
        // スケジュールを設定
        // cronString = "0 */1 * * * ? *";
        try {
			cronTrigger.setCronExpression(cronString);
		} catch (ParseException e) {
			m_log.warn("addQuartz() addSchedule : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
 		}
        
        //Schedulerにジョブを登録する
        try {
			manager.addSchedule(job, cronTrigger);
		} catch (NamingException e) {
			m_log.warn("addQuartz() addSchedule : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
        } catch (SchedulerException e) {
			m_log.warn("addQuartz() addSchedule : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
        } catch (RemoteException e) {
			m_log.warn("addQuartz() addSchedule : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
		}
    }
    
    /**
     * Quartzからステータスの定周期管理の登録を削除します。<BR><BR>
     *  
     *  このメソッドは監視管理のステータスが有効期限切れになっているかを
     * 定期的にチェックするトリガーをQurtzから削除します。<BR>
     * このメソッドをQurtzSetup以外から呼び出さないでください。
     *  
     * @ejb.interface-method
     * 
     * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
     * 
     * @throws NamingException
     * @throws ParseException
     * @throws SchedulerException
     * 
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#deleteSchedule(java.lang.String, java.lang.String)
     * @see #manageStatus()
     * 
     * @since 2.0.0
     */
    public void deleteQuartz() throws NamingException, ParseException, SchedulerException{
        //QuartzのSchedulerをルックアップ
    	QuartzManager manager = null;
		try {
			manager = QuartzUtil.getQuartzManager();
		} catch (NamingException e) {
			m_log.warn("deleteQuartz() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
		
        try {
            //ジョブ削除
        	manager.deleteSchedule(QuartzConstant.JNDI_NAME_MON, 
            		QuartzConstant.GROUP_NAME_MON);
        } catch (NamingException e) {
			m_log.warn("deleteQuartz() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
        } catch (SchedulerException e) {
			m_log.warn("deleteQuartz() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
        } catch (RemoteException e) {
			m_log.warn("deleteQuartz() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
		}
    }

    /**
	 * ステータス情報フィルタ用プロパティを取得します。<BR><BR>
	 * 
	 * クライアントでステータスのフィルタ条件を設定するために、
	 * 空のpropertyインスタンスをマネージャから取得します。<BR>
	 * (propertyには属性に文字列が入るので、マネージャでlocaleにしたがって
	 *  空のインスタンスを生成します。)
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorRead"
     *     method-intf="Remote"
	 * 
	 * @param locale クライアントのロケール
	 * @return ステータス情報フィルタ用プロパティ
	 * 
	 * @see com.clustercontrol.monitor.factory.StatusFilterProperty#getProperty(Locale)
	 */
	public Property getStatusFilterProperty(Locale locale) {
		StatusFilterProperty status = new StatusFilterProperty();
		return status.getProperty(locale);
	}
	
	/**
	 * イベント情報フィルタ用プロパティを取得します。<BR><BR>
	 * 
	 * クライアントでイベントのフィルタ条件を設定するために、
	 * 空のpropertyインスタンスをマネージャから取得します。<BR>
	 * (propertyには属性に文字列が入るので、マネージャでlocaleにしたがって
	 *  空のインスタンスを生成します。)
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorRead"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール情報
	 * @return イベント情報フィルタ用プロパティ
	 * 
	 * @see com.clustercontrol.monitor.factory.EventFilterProperty#getProperty(Locale)
	 */
	public Property getEventFilterProperty(Locale locale) {
		EventFilterProperty event = new EventFilterProperty();
		return event.getProperty(locale);
	}
	
	/**
	 * イベント情報一括確認用プロパティを取得します。<BR><BR>
	 * 
	 *  クライアントでイベントの一括確認用プロパティを設定するために、
	 * 空のpropertyインスタンスをマネージャから取得します。<BR>
	 * (propertyには属性に文字列が入るので、マネージャでlocaleにしたがって
	 *  空のインスタンスを生成します。)
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorRead"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール情報
	 * @return イベント情報一括確認用プロパティ
	 * 
	 * @see com.clustercontrol.monitor.factory.EventBatchConfirmProperty#getProperty(Locale)
	 */
	public Property getEventBatchConfirmProperty(Locale locale) {
		EventBatchConfirmProperty confirm = new EventBatchConfirmProperty();
		return confirm.getProperty(locale);
	}
	
	/**
	 * スコープ情報一覧を取得します。<BR><BR>
	 * 引数で指定されたファシリティの配下全てのファシリティのスコープ情報一覧を返します。<BR>
	 * 各スコープ情報は、ScopeInfoDataのインスタンスとして保持されます。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 *     
	 * @ejb.transaction
	 *   type="NotSupported"
	 * 
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @return スコープ情報一覧（ScopeInfoDataが格納されたArrayList）
	 * @throws CreateException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.monitor.bean.ScopeInfoData
	 * @see com.clustercontrol.monitor.factory.SelectScope#getScopeList(String)
	 */
	public ArrayList getScopeList(String facilityId)
			throws CreateException, NamingException,
			FacilityNotFoundException, HinemosUnknownException {
		
		return getScopeList(facilityId, true, true, true);
	}

	/**
	 * スコープ情報一覧を取得します。<BR><BR>
	 * 引数で指定されたファシリティの配下全てのファシリティのスコープ情報一覧を返します。<BR>
	 * 各スコープ情報は、ScopeInfoDataのインスタンスとして保持されます。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 *     
	 * @ejb.transaction
	 *   type="NotSupported"
	 * 
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @return スコープ情報一覧（ScopeInfoDataが格納されたArrayList）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.monitor.bean.ScopeInfoData
	 * @see com.clustercontrol.monitor.factory.SelectScope#getScopeList(String)
	 */
	public ArrayList getScopeList(String facilityId, boolean statusFlag, boolean eventFlag, boolean orderFlg)
			throws CreateException, NamingException, FacilityNotFoundException, HinemosUnknownException {

		//ステータス情報を取得
		SelectScope select = new SelectScope();
		ArrayList list = select.getScopeList(facilityId, statusFlag, eventFlag, orderFlg);

		return list;
	}
	
	/**
	 * ステータス情報一覧を取得します。<BR>
	 * 引数で指定されたファシリティの配下全てのファシリティのステータス情報一覧を返します。<BR>
	 * 各ステータス情報は、StatusInfoDataのインスタンスとして保持されます。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @ejb.transaction
	 *   type="NotSupported"
	 *   
	 * @param facilityId 取得対象の親ファシリティID
	 * @return ステータス情報一覧（StatusInfoDataが格納されたArrayList）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.monitor.bean.StatusInfoData
	 * @see com.clustercontrol.monitor.factory.SelectStatus#getStatusList(String, int)
	 */
	public ArrayList getStatusList(String facilityId) throws CreateException, FacilityNotFoundException, NamingException {

		//ステータス情報を取得
		SelectStatus select = new SelectStatus();
		ArrayList list = select.getStatusList(facilityId, ALL);

		return list;
	}
	
	/**
	 * 引数で指定されたステータス情報を削除します。<BR>
	 * 
	 * 引数のlistは、StatusInfoDataが格納されたListとして渡されます。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param list 削除対象のステータス情報一覧（StatusInfoDataが格納されたList）
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * 
	 * @since 2.0.0
	 * 
	 * @see com.clustercontrol.monitor.bean.StatusInfoData
	 * @see com.clustercontrol.monitor.factory.DeleteStatus#delete(List)
	 */
	public boolean deleteStatus(List list) throws FinderException, RemoveException, NamingException {

		//ステータス情報を削除
		DeleteStatus status = new DeleteStatus();
		
		boolean ret = false;
		
		try {
			ret = status.delete(list);
		} catch(NamingException e){
			m_log.warn("deleteStatus() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch(RemoveException e){
			m_log.warn("deleteStatus() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (FinderException e) {
			m_log.warn("deleteStatus() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
		return ret;
	}
	
	/**
	 * 引数で指定された条件に一致するステータス情報一覧を取得します。<BR>
	 * 各ステータス情報は、StatusInfoDataのインスタンスとして保持されます。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 *
	 * @ejb.transaction
	 *   type="NotSupported"
	 *  
	 * @param facilityId 取得対象の親ファシリティID
	 * @param property 検索条件
	 * @return ステータス情報一覧（StatusInfoDataが格納されたArrayList）
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FacilityNotFoundException 
	 * 
	 * @see com.clustercontrol.monitor.bean.StatusInfoData
	 * @see com.clustercontrol.monitor.factory.SelectStatus#getStatusList(String, Property)
	 */
	public ArrayList getStatusList(String facilityId, Property property)
			throws CreateException, NamingException, FacilityNotFoundException {

		//ステータス情報を取得
		SelectStatus select = new SelectStatus();
		ArrayList list = select.getStatusList(facilityId, property);

		return list;
	}
	
	/**
	 * 存続期間を経過したステータス情報を削除 または 更新します。<BR><BR>
	 * 
	 * Quartzから呼び出されるコールバックメソッド<BR>
	 * Quartz以外から呼び出さないでください。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @throws RemoveException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @since 2.0.0
	 *
	 * @see com.clustercontrol.monitor.factory.ManageStatus#execute()
	 * @see #addQuartz(String)
	 * @see #deleteQuartz()
	 */
	public void manageStatus() throws RemoveException, FinderException, NamingException {

		// 期限切れのステータス情報を削除/更新する
		ManageStatus manage = new ManageStatus();
		manage.execute();
	}
	
	/**
	 * イベント一覧情報を取得します。(クライアントview用) <BR><BR>
	 * 引数で指定されたファシリティの配下全てのファシリティのイベント情報を取得します。<BR>
	 * 各イベント情報は、EventLogDataインスタンスとして保持されます。<BR>
	 * 戻り値のViewListInfoは、クライアントにて表示用の形式に変換されます。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param messages 表示イベント数
	 * @return ビュー一覧情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * @throws SQLException 
	 * 
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogData
	 * @see com.clustercontrol.monitor.factory.SelectEvent#getEventList(String, int, int)
	 */
	public ViewListInfo getEventList(String facilityId, int messages) throws CreateException, FinderException, NamingException, SQLException {

		// イベントログ情報を取得
		SelectEvent select = new SelectEvent();
		ViewListInfo list = select.getEventList(facilityId, ALL, messages);

		return list;
	}
	
	/**
	 * 引数で指定された条件に一致するイベント一覧情報を取得します。(クライアントview用)<BR><BR>
	 * 
	 * 引数のpropertyには、com.clustercontrol.monitor.factory.StatusFilterPropertyの属性が１つ以上含まれます。<BR>
	 * 各イベント情報は、EventLogDataインスタンスとして保持されます。<BR>
	 * 戻り値のViewListInfoは、クライアントにて表示用の形式に変換されます。
	 * 	
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param property 検索条件
	 * @param messages 表示イベント数
	 * @return ビュー一覧情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * @throws SQLException 
	 * 
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogData
	 * @see com.clustercontrol.monitor.factory.SelectEvent#getEventList(String, Property, int)
	 */
	public ViewListInfo getEventList(String facilityId, Property property, int messages) throws CreateException, FinderException, NamingException, SQLException {

		// イベントログ情報を取得
		SelectEvent select = new SelectEvent();
		ViewListInfo list = select.getEventList(facilityId, property, messages);

		return list;
	}
	
	/**
	 * 引数で指定された条件に一致する帳票出力用イベント情報一覧を返します。<BR><BR>
	 * 
	 * 引数のpropertyには、com.clustercontrol.monitor.factory.StatusFilterPropertyの属性が
	 * １つ以上含まれます。<BR> 
	 * 戻り値のArrayListはArrayListのArrayListであり、内部のArrayListには、
	 * com.clustercontrol.monitor.bean.ReportEventInfoがリストとして格納されます。
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param property 検索条件
	 * @return 帳票出力用イベント情報一覧（{@link com.clustercontrol.monitor.bean.ReportEventInfo}のリスト）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * @throws IndexOutOfBoundsException
	 * 
	 * @since 2.1.0
	 * 
	 * @see com.clustercontrol.monitor.factory.SelectEvent#getEventListForReport(String, Property)
	 */
	public ArrayList getEventListForReport(String facilityId, Property property) throws CreateException, FinderException, NamingException, IndexOutOfBoundsException {

		// 帳票出力用イベントログ情報を取得
		SelectEvent select = new SelectEvent();
		ArrayList list = select.getEventListForReport(facilityId, property);

		return list;
	}
	
	/**
	 * イベント詳細表示用プロパティを返します。<BR><BR>
	 * 
	 * 
	 *<p>戻り値のプロパティの項目は、下記の通りです。
	 * <p>
	 * <ul>
	 *  <li>プロパティ（親。ダミー）</li>
	 *  <ul>
	 *   <li>重要度（子。テキスト）</li>
	 *   <li>受信日時（子。テキスト）</li>
	 *   <li>出力日時（子。テキスト）</li>
	 *   <li>プラグインID（子。テキスト）</li>
	 *   <li>監視項目ID（子。テキスト）</li>
	 *   <li>ファシリティID（子。テキスト）</li>
	 *   <li>スコープ（子。テキスト）</li>
	 *   <li>アプリケーション（子。テキスト）</li>
	 *   <li>メッセージID（子。テキスト）</li>
	 *   <li>メッセージ（子。テキスト）</li>
	 *   <li>オリジナルメッセージ（子。テキストエリア）</li>
	 *   <li>確認（子。テキスト）</li>
	 *   <li>確認済み日時（子。テキスト）</li>
	 *   <li>確認ユーザ（子。テキスト）</li>
	 *   <li>重複カウンタ（子。テキスト）</li>
	 *  </ul>
	 * </ul>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorRead"
     *     method-intf="Remote"
	 * 
	 * @param monitorId 取得対象の監視項目ID
	 * @param pluginId 取得対象のプラグインID
	 * @param facilityId 取得対象のファシリティID
	 * @param outputDate 取得対象の受信日時
	 * @param locale ロケール情報
	 * @return イベント詳細情報表示用プロパティ
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.monitor.factory.EventInfoProperty#getProperty(String, String, String, Date, Locale)
	 */
	public Property getEventInfoProperty(
			String monitorId,
			String pluginId,
			String facilityId,
			Date outputDate,
			Locale locale) throws FinderException, NamingException {
		
		// イベント情報詳細を取得
		EventInfoProperty event = new EventInfoProperty();
		return event.getProperty(monitorId, pluginId, facilityId, outputDate, locale);
	}
	
	/**
	 * 引数で指定されたイベント情報の確認を更新します。<BR><BR>
	 * 確認ユーザとして、操作を実施したユーザを設定します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param monitorId 更新対象の監視項目ID
	 * @param pluginId 更新対象のプラグインID
	 * @param facilityId 更新対象のファシリティID
	 * @param outputDate 更新対象の受信日時ID
	 * @param confirmDate 確認済み日時（更新値）
	 * @param confirmType 確認タイプ（未／済）（更新値）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.monitor.factory.ModifyEventConfirm#modifyConfirm(String, String, String, Date, Date, int)
	 * 
	 */
	public void modifyConfirm(String monitorId,
			String pluginId,
			String facilityId,
			int priority,
			Date generateDate,
			Date outputDate,
			Date confirmDate,
			int confirmType
			) throws CreateException, FinderException, NamingException {

		String confirmUser = m_context.getCallerPrincipal().getName();
				
		// イベントの確認状態を更新する
		ModifyEventConfirm modify = new ModifyEventConfirm();
		
		try{
			modify.modifyConfirm(monitorId, pluginId, facilityId, priority, generateDate, outputDate, confirmDate, confirmType, confirmUser);
		}catch(NamingException e){
			m_log.warn("modifyConfirm() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (FinderException e) {
			m_log.warn("modifyConfirm() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
	}
	
	/**
	 * 引数で指定されたイベント情報一覧の確認を更新します。<BR><BR>
	 * 確認ユーザとして、操作を実施したユーザを設定します。<BR>
	 * 複数のイベント情報を更新します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param list 更新対象のイベント情報一覧（EventLogDataが格納されたList）
	 * @param confirmType 確認タイプ（未／済）（更新値）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.monitor.factory.ModifyEventConfirm#modifyConfirm(List, int)
	 */
	public void modifyConfirm(List list,
			int confirmType
			) throws CreateException, FinderException, NamingException {

		String confirmUser = m_context.getCallerPrincipal().getName();
				
		// イベントの確認状態を更新する
		ModifyEventConfirm modify = new ModifyEventConfirm();
		
		try{
			modify.modifyConfirm(list, confirmType, confirmUser);
		}catch(NamingException e){
			m_log.warn("modifyConfirm() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (FinderException e) {
			m_log.warn("modifyConfirm() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
	}
	
	/**
	 * 引数で指定された条件に一致するイベント情報の確認を一括更新します。<BR><BR>
	 * 確認ユーザとして、操作を実施したユーザを設定します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param confirmType 確認タイプ（未／済）（更新値）
	 * @param facilityId 更新対象の親ファシリティID
	 * @param property 更新条件
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws SQLException
	 * 
	 * @see com.clustercontrol.bean.ConfirmConstant 
	 * @see com.clustercontrol.monitor.factory.ModifyEventConfirm#modifyBatchConfirm(int, String, Property)
	 */
	public void modifyBatchConfirm(int confirmType, String facilityId, Property property) throws CreateException, NamingException, FinderException, SQLException {

		String confirmUser = m_context.getCallerPrincipal().getName();
		
		// イベントの確認状態を一括更新する
		ModifyEventConfirm modify = new ModifyEventConfirm();
		
		try{
			modify.modifyBatchConfirm(confirmType, facilityId, property, confirmUser);
		}catch(NamingException e){
			m_log.warn("modifyBatchConfirm() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (CreateException e) {
			m_log.warn("modifyBatchConfirm() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (FinderException e) {
			m_log.warn("modifyBatchConfirm() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (SQLException e) {
			m_log.warn("modifyBatchConfirm() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
	}

}
