/*
 
Copyright (C) 2010 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.jobmanagement.ejb.entity.JobUserRelationMasterBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobUserRelationMasterPK;

/**
 * ジョブユーザリレーションマスタのDAOインタフェースを実装するクラス<BR>
 * 
 * @version 3.2.0
 * @since 3.2.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobUserRelationMasterBean
 * @see com.clustercontrol.jobmanagement.dao.JobUserRelationMasterDAO
 * 
 */
public class JobUserRelationMasterDAOImpl implements JobUserRelationMasterDAO {

	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(JobUserRelationMasterDAOImpl.class);
	
	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobUserRelationMasterDAO#create(JobUserRelationMasterBean)
	 * 
	 */
	public JobUserRelationMasterPK create(JobUserRelationMasterBean ejb) throws CreateException, EJBException {
		
		m_log.debug("create() start : jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id()
				+ ", userId=" + ejb.getUser_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		JobUserRelationMasterPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			StringBuffer sql = new StringBuffer();
			
			sql.append("INSERT INTO cc_job_user_relation_mst (");
			sql.append("jobunit_id, ");
			sql.append("job_id, ");
			sql.append("user_id) ");
			sql.append("VALUES (?,?,?)");
			
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getJobunit_id());
			stmt.setString(2, ejb.getJob_id());
			stmt.setString(3, ejb.getUser_id());
			
			int row =	stmt.executeUpdate();
			pk = new JobUserRelationMasterPK(ejb.getJobunit_id(), ejb.getJob_id(), ejb.getUser_id());
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : jobunitId=" + ejb.getJobunit_id()
						+ ", jobId=" + ejb.getJob_id()
						+ ", userId=" + ejb.getUser_id() + " SQLException");
				throw new EJBException(msg);
			}
			
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : jobunitId=" + ejb.getJobunit_id()
						+ ", jobId=" + ejb.getJob_id()
						+ ", userId=" + ejb.getUser_id() + " SQLException"
						+ " DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : jobunitId=" + ejb.getJobunit_id()
						+ ", jobId=" + ejb.getJob_id()
						+ ", userId=" + ejb.getUser_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		
		m_log.debug("create() end : jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id()
				+ ", userId=" + ejb.getUser_id());
		
		return pk;
	}
	
	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobUserRelationMasterBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobUserRelationMasterDAO#findAll()
	 * 
	 */
	public Collection<JobUserRelationMasterPK> findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<JobUserRelationMasterPK> ret = new ArrayList<JobUserRelationMasterPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT jobunit_id, job_id, user_id FROM cc_job_user_relation_mst";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobUserRelationMasterPK pk = new JobUserRelationMasterPK(
						res.getString("jobunit_id"),
						res.getString("job_id"),
						res.getString("user_id"));
				
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		} finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		
		return ret;
	}

	public Collection<JobUserRelationMasterPK> findByJobunitId(String jobunitId) throws FinderException {
		m_log.debug("findByJobunitId() start : jobunitId=" + jobunitId);
		
		ArrayList<JobUserRelationMasterPK> ret = new ArrayList<JobUserRelationMasterPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT jobunit_id, job_id, user_id FROM cc_job_user_relation_mst WHERE jobunit_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, jobunitId);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobUserRelationMasterPK pk = new JobUserRelationMasterPK(
						res.getString("jobunit_id"),
						res.getString("job_id"),
						res.getString("user_id"));
				
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findByJobunitId() error : jobunitId=" + jobunitId + " SQLException");
			throw new EJBException(e.getMessage());
		} finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByJobunitId() error : jobunitId=" + jobunitId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByJobunitId() end : jobunitId=" + jobunitId);
		
		return ret;
	}

	public JobUserRelationMasterPK findByPrimaryKey(JobUserRelationMasterPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : pk=" + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT jobunit_id, job_id, user_id FROM cc_job_user_relation_mst WHERE jobunit_id = ? AND job_id = ? AND user_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJobunit_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getUser_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
			
		} catch (SQLException e) {
			m_log.error("findByPrimaryKey() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByPrimaryKey() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
		
	}

	public Collection<JobUserRelationMasterPK> findByUserId(String userId) throws FinderException {
		m_log.debug("findByUserId() start : userId=" + userId);
		
		ArrayList<JobUserRelationMasterPK> ret = new ArrayList<JobUserRelationMasterPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT jobunit_id, job_id, user_id FROM cc_job_user_relation_mst WHERE user_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, userId);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobUserRelationMasterPK pk = new JobUserRelationMasterPK(
						res.getString("jobunit_id"),
						res.getString("job_id"),
						res.getString("user_id"));
				
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findByUserId() error : userId=" + userId + " SQLException");
			throw new EJBException(e.getMessage());
		} finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByUserId() error : userId=" + userId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByUserId() end : userId=" + userId);
		
		return ret;
		
	}
	
	public void init() {
		
	}

	/**
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobUserRelationMasterDAO#load(JobUserRelationMasterPK, JobUserRelationMasterBean)
	 * 
	 */
	public void load(JobUserRelationMasterPK pk, JobUserRelationMasterBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_user_relation_mst WHERE jobunit_id = ? AND job_id = ? AND user_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJobunit_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getUser_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setJobunit_id(res.getString("jobunit_id"));
				ejb.setJob_id(res.getString("job_id"));
				ejb.setUser_id(res.getString("user_id"))
				;
			} else {
				String msg = "JobUserRelationMaster data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
			
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1){
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		
		m_log.debug("load() end : " + pk.toString());
	}

	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobUserRelationMasterDAO#remove(JobUserRelationMasterPK)
	 */
	public void remove(JobUserRelationMasterPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_job_user_relation_mst WHERE jobunit_id = ? AND job_id = ? AND user_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJobunit_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getUser_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
			
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		
		m_log.debug("remove() end : " + pk.toString());
	}

	/**
	 * データベースの更新を行うのが基本だが、<BR>
	 * 全てのカラムが主キーになっているので、更新を行う必要はない。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobInfoDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoBean)
	 * 
	 */
	public void store(JobUserRelationMasterBean ejb) throws EJBException {
		m_log.debug("store() start : jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id()
				+ ", userId=" + ejb.getUser_id());

		m_log.debug("store() end : jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id()
				+ ", userId=" + ejb.getUser_id());
	}
	
}
