/*
 
Copyright (C) since 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.accesscontrol.ejb.session;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.rmi.AccessException;
import java.rmi.RemoteException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.bean.UserRoleList;
import com.clustercontrol.accesscontrol.factory.LoginUserModifier;
import com.clustercontrol.accesscontrol.factory.LoginUserSelector;
import com.clustercontrol.accesscontrol.factory.UserFilterProperty;
import com.clustercontrol.accesscontrol.factory.UserProperty;
import com.clustercontrol.bean.Property;
import com.clustercontrol.jobmanagement.ejb.entity.JobUserRelationMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobUserRelationMasterUtil;

/**
 * アクセス管理機能を実現するSession Bean<BR>
 * 
 * @ejb.bean name="AccessController"	
 *    jndi-name="AccessController"
 *    type="Stateless" 
 *    transaction-type="Container"
 * 
 * @ejb.transaction
 *    type="Required"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessLockLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=UserLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=UserRoleLocal"
 */
public abstract class AccessControllerBean implements javax.ejb.SessionBean {

	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog( AccessControllerBean.class );

	/** コンテキスト情報 */
	private SessionContext m_context;

	/**
	 * コンテキスト情報を設定する。<BR>
	 * 
	 * Session Beanがインスタンスプールに格納される際に行う処理を実装する。<BR>
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * 自身のユーザ情報を取得する。<BR>
	 * ユーザ情報には、以下の形式で値が格納されている。<BR>
	 * <PRE>
	 * {
	 *   {ユーザID1, ユーザ名1, 説明1, 作成者（ユーザID）1, 作成日時1, 最終更新者（ユーザID）1, 最終更新日時1}
	 *   ...
	 * }
	 * </PRE>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *    role-name="RepositoryRead"
	 *    method-intf="Remote"
	 * 
	 * @return ユーザ一覧情報(自身の情報のみ)
	 * @throws FinderException 
	 * @throws NamingException 
	 * 
	 */
	public ArrayList getOwnData(String userId) throws AccessException {
		/** メイン処理 */
		m_log.debug("getOwnData() userId = " + userId);

		String loginUser = m_context.getCallerPrincipal().getName();
		m_log.debug("getOwnData() loginUser = " + loginUser);
		
		if(userId != null && loginUser != null && userId.equals(loginUser)){
			return LoginUserSelector.getOwnData(userId);
		}else{
			m_log.error("getOwnData() Invalid User Operation(get own user data) userId = " + userId + ", loginUser = " + loginUser);
			throw new AccessException("Invalid User Operation(get own user data) userId = " + userId + ", loginUser = " + loginUser);
		}
	}
	
	/**
	 * ユーザ一覧情報を取得する。<BR>
	 * ユーザ一覧情報には、以下の形式で値が格納されている。<BR>
	 * <PRE>
	 * {
	 *   {ユーザID1, ユーザ名1, 説明1, 作成者（ユーザID）1, 作成日時1, 最終更新者（ユーザID）1, 最終更新日時1}, 
	 *   {ユーザID2, ユーザ名2, 説明2, 作成者（ユーザID）2, 作成日時2, 最終更新者（ユーザID）2, 最終更新日時2},
	 *   ...
	 * }
	 * </PRE>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *    role-name="AccessControlRead"
	 *    method-intf="Remote"
	 * 
	 * @return ユーザ一覧情報
	 * @throws FinderException 
	 * @throws NamingException 
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserSelector#getUserList()
	 */
	public ArrayList getUserList() throws FinderException, NamingException {
		/** メイン処理 */
		return LoginUserSelector.getUserList();
	}

	/**
	 * ユーザフィルタダイアログの入力値に基づき、ユーザ一覧情報を取得する。<BR>
	 * ユーザ一覧情報には、以下の形式で値が格納されている。<BR>
	 * <PRE>
	 * {
	 *   {ユーザID1, ユーザ名1, 説明1, 作成者（ユーザID）1, 作成日時1, 最終更新者（ユーザID）1, 最終更新日時1}, 
	 *   {ユーザID2, ユーザ名2, 説明2, 作成者（ユーザID）2, 作成日時2, 最終更新者（ユーザID）2, 最終更新日時2},
	 *   ...
	 * }
	 * </PRE>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *    role-name="AccessControlRead"
	 *    method-intf="Remote"
	 * 
	 * @param property ユーザフィルタダイアログの入力値
	 * @return ユーザ一覧情報
	 * @throws FinderException 
	 * @throws NamingException 
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserSelector#getUserList(Property)
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getUserList(Property property) throws FinderException, NamingException {
		/** メイン処理 */
		return LoginUserSelector.getUserList(property);
	}

	/**
	 * ユーザフィルタダイアログの入力カラム定義情報を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *    role-name="AccessControlRead"
     *    method-intf="Remote"
     * 
	 * @param locale ロケール情報
	 * @return ユーザフィルタダイアログの入力カラム定義情報
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.UserFilterProperty#getProperty(Locale)
	 */
	public Property getUserFilterProperty(Locale locale) {
		/** メイン処理 */
		return UserFilterProperty.getProperty(locale);
	}

	/**
	 * ユーザ情報を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *    role-name="AccessControlRead"
	 *    method-intf="Remote"
	 * 
	 * @param userId ユーザID
	 * @param mode 取得モード
	 * @param locale ロケール情報
	 * @return ユーザ情報
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.bean.PropertyConstant
	 * @see com.clustercontrol.accesscontrol.factory.UserProperty#getProperty(String, int, Locale)
	 */
	public Property getUserProperty(String userId, int mode, Locale locale) throws FinderException, NamingException {
		/** メイン処理 */
		return UserProperty.getProperty(userId, mode, locale);
		
	}

	/**
	 * ユーザを追加する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *    role-name="AccessControlWrite"
	 *    method-intf="Remote"
	 * 
	 * @param property ユーザ情報
	 * @throws NamingException 
	 * @throws CreateException 
	 * @throws FinderException 
	 * @throws ParseException 
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#addUser(Property, String, String)
	 */
	public void addUser(Property property) throws NamingException, CreateException, FinderException, ParseException {
		try {
			/** メイン処理 */
			LoginUserModifier.addUser(property, m_context.getCallerPrincipal().getName());
		} catch (CreateException e) {
			m_log.warn("addUser() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (ParseException e) {
			m_log.warn("addUser() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
	}

	/**
	 * ユーザを変更する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *    role-name="AccessControlWrite"
	 *    method-intf="Remote"
	 * 
	 * @param property ユーザ情報
	 * @throws NamingException 
	 * @throws FinderException 
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#modifyUser(Property, String)
	 */
	public void modifyUser(Property property) throws NamingException, FinderException {
		/** メイン処理 */
		LoginUserModifier.modifyUser(property, m_context.getCallerPrincipal().getName());
	}

	/**
	 * ユーザを削除する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *    role-name="AccessControlWrite"
	 *    method-intf="Remote"
	 * 
	 * @param userId ユーザID
	 * @throws NamingException 
	 * @throws FinderException 
	 * @throws RemoveException 
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#deleteUser(String, String)
	 */
	public void deleteUser(String userId) throws NamingException, RemoveException, FinderException {
		/** メイン処理 */
		LoginUserModifier.deleteUser(userId, m_context.getCallerPrincipal().getName());
	}

	/**
	 * 自分自身のパスワードを変更する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *    role-name="RepositoryRead"
	 *    method-intf="Remote"
	 * 
	 * @param userId ユーザID
	 * @param password パスワード
	 * @throws NamingException 
	 * @throws FinderException 
	 * @throws RemoveException
	 * 
	 */
	public void changeOwnPassword(String userId, String password) throws AccessException {
		/** メイン処理 */
		m_log.debug("changeOwnPassword() userId = " + userId + ", password = " + password);

		String loginUser = m_context.getCallerPrincipal().getName();
		m_log.debug("changeOwnPassword() loginUser = " + loginUser);
		
		if(userId != null && loginUser != null && userId.equals(loginUser)){
			m_log.debug("changeOwnPassword() change passwd");
			LoginUserModifier.modifyUserPassword(userId, password);
		}
		else{
			m_log.error("changeOwnPassword() Invalid User Operation(change password) userId = " + userId + ", loginUser = " + loginUser);
			throw new AccessException("Invalid User Operation(change password) userId = " + userId + ", loginUser = " + loginUser);
		}
	}
	
	/**
	 * パスワードを変更する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *    role-name="AccessControlWrite"
	 *    method-intf="Remote"
	 * 
	 * @param userId ユーザID
	 * @param password パスワード
	 * @throws NamingException 
	 * @throws FinderException 
	 * @throws RemoveException
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#modifyUserPassword(String, String)
	 */
	public void changePassword(String userId, String password) throws NamingException, RemoveException, FinderException {
		/** メイン処理 */
		LoginUserModifier.modifyUserPassword(userId, password);
	}

	/**
	 * ログインしているユーザが指定したユーザ権限を持っているかどうかを確認する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *    unchecked="true"
	 *    method-intf="Remote"
	 * 
	 * @param userRole ユーザ権限名
	 * @return ユーザ権限を保持していればtrue, そうでなければfalse
	 */
	public boolean isPermission(String userRole) throws NamingException {
		/** メイン処理 */
		return m_context.isCallerInRole(userRole);
	}

	/**
	 * ログインユーザのユーザ名を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *    unchecked="true"
	 *    method-intf="Remote"
	 * 
	 * @return ユーザ名
	 * @throws NamingException 
	 * @throws FinderException 
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserSelector#getUserName(String)
	 */
	public String getUserName() throws FinderException, NamingException {
		/** メイン処理 */
		return LoginUserSelector.getUserName(m_context.getCallerPrincipal().getName());
	}

	/**
	 * バージョン番号を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *    unchecked="true"
	 *    method-intf="Remote"
	 * 
	 * @return バージョン番号
	 */
	public String getVersion() {
		/** ローカル変数 */
		String homedir = null;
		String path = null;
		FileInputStream versionFile = null;
		InputStreamReader isr = null;
		BufferedReader reader = null;
		String version = null;
		
		/** メイン処理 */
		homedir = System.getProperty("hinemos.home.dir");
		path = homedir + File.separator + "_version";
		
		try {
			versionFile = new FileInputStream(path);
			isr = new InputStreamReader(versionFile);
			reader = new BufferedReader(isr);
			
			version = reader.readLine();
		} catch (Exception e) {
			return "unknown";
		}
		
		return version;
	}
	
	/**
	 * 引数で指定されたユーザがジョブユニットの管理ユーザに設定されているか確認する<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *    role-name="AccessControlWrite"
	 *    method-intf="Remote"
	 * 
	 * @param userId
	 * 	 * @return 引数で指定されたユーザが管理ユーザに設定されているジョブユニットIDのリスト
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo
	 */
	public ArrayList<String> checkJobManagementUser(String userId) throws FinderException, NamingException {
		ArrayList<String> ret = new ArrayList<String>();
		
		Collection<JobUserRelationMasterLocal> userList = JobUserRelationMasterUtil.getLocalHome().findByUserId(userId);
		
		for (JobUserRelationMasterLocal userInfo : userList) {
			ret.add(userInfo.getJobunit_id());
		}
		
		return ret;
	}
	
	/**
	 * ユーザの権限リストを返す<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *    unchecked="true"
	 *    method-intf="Remote"
	 * 
	 * @return ユーザの権限リスト
	 */
	public UserRoleList getUserRoleList() {
		return UserProperty.getRoleList();
	}
}

