/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.agent;

import java.util.ArrayList;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.Date;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;
import javax.naming.InitialContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.util.AgentProperties;
import com.clustercontrol.agent.util.InternalLogger;
import com.clustercontrol.agent.util.RunHistoryUtil;
import com.clustercontrol.jobmanagement.bean.CommandConstant;
import com.clustercontrol.jobmanagement.bean.CommandTypeConstant;
import com.clustercontrol.jobmanagement.bean.RunStatusConstant;
import com.clustercontrol.jobmanagement.bean.TopicConstant;
import com.clustercontrol.jobmanagement.message.RunInstructionInfo;
import com.clustercontrol.jobmanagement.message.RunResultInfo;
import com.clustercontrol.util.Dummy;


/**
 * Topicを受信するクラス<BR>
 * Topicへの接続と、メッセージの受信を行います。
 * 
 * 
 * Topicでマネージャからのジョブ実行指示を受け取ります。
 * 
 * @version $Revision: 3167 $
 * @since 1.0.0
 * 
 */
public class ReceiveTopic implements MessageListener, ExceptionListener {

	private static final String TOPIC_CON_FACTORY = "ConnectionFactory";
	private static final String TOPIC_USER_NAME = "topic.user.name";
	private static final String TOPIC_USER_PASSWORD = "topic.user.password";

	protected Agent m_agent;
	
	protected ArrayList m_facilityIdList;
    protected SendQueue m_sendQueue;
    
    protected static Hashtable<String, Date> m_runHistory;

	private TopicConnectionFactory m_factory;

    protected TopicConnection m_con;
    protected Topic m_topic;
    protected TopicSession m_session;
    protected TopicSubscriber m_subscriber;

    private long m_jobTopicReconnectionInterval = 30;
    private long m_runhistoryClearDelay = -1;
    
	private volatile boolean m_errFlg = false;
	private static boolean m_clearFlg = false;
	
	// heartbeat送受完了日時(jms妥当性判定用)
	private volatile static Date heartbeatReceiveDate = new Date();
	public static final Object _instanceHeartbeatDateLock = new Object();

	/** ReceiveTopicクラス用スケジューラ * */
	private ScheduledExecutorService m_scheduler = Executors.newScheduledThreadPool(10);
    private ScheduledFuture<?> m_future;

	
	/** 実行履歴削除用スケジューラ **/
	private ScheduledExecutorService m_schedulerClear = Executors.newScheduledThreadPool(10);
    private ScheduledFuture<?> m_futureClear;

	//ロガー
	private static Log log = LogFactory.getLog(ReceiveTopic.class);
	
	/**
	 * コンストラクタ
	 * @param agent ジョブエージェント
	 * @param facilityIdList ファシリティIDのリスト　
	 * @param sendQueue　メッセージ送信用クラス
	 * @param props　プロパティファイル情報
	 */
	@SuppressWarnings("unchecked")
	public ReceiveTopic(Agent agent, Collection facilityIdList, SendQueue sendQueue) {
		super();
		m_agent = agent;
		m_facilityIdList = new ArrayList(facilityIdList);
		m_sendQueue = sendQueue;
		m_runHistory = new Hashtable<String, Date>();

		// 再接続処理実行間隔取得
		String interval = AgentProperties.getProperty("job.topic.reconnection.interval");
		if (interval != null) {
			try {
				// プロパティファイルには秒で記述
				m_jobTopicReconnectionInterval = Integer.parseInt(interval);
				log.info("job.topic.reconnection.interval = " + m_jobTopicReconnectionInterval + " sec");
			} catch (NumberFormatException e) {
				log.error("job.topic.reconnection.interval",e);
			}
		}
		
		// ジョブ実行履歴の削除実行時間を取得
		String delay = AgentProperties.getProperty("runhistory.clear.delay");
		if (delay != null) {
			try {
				// プロパティファイルには秒で記述
				m_runhistoryClearDelay = Integer.parseInt(delay);
				log.info("runhistory.clear.delay = " + m_runhistoryClearDelay + " sec");
			} catch (NumberFormatException e) {
				log.error("runhistory.clear.delay",e);
			}
		}
		
		//接続処理
        initial();
	}
    
    
    /**
     * ファシリティIDリスト設定.
     * 設定されたファイシリティIDのリストが、前回と違う場合は、トピック受信の再接続を行う
     * @param facilityIdList　
     * @since
     */
	@SuppressWarnings("unchecked")
	synchronized public void setFacilityIdList(Collection facilityIdList) {
	
		//取得した情報が変化しているかどうか確認
		if(facilityIdList.size() == m_facilityIdList.size()){
			if( m_facilityIdList.containsAll(facilityIdList) ){
				return;
			}
		}
		
		//ファイシリティリスト更新
		m_facilityIdList = new ArrayList(facilityIdList);
		//Topic再接続
		if( !isErrFlg() ){
		    initialTopic();
		}
	}
    /* 
     * トピック受信処理
     * (non-Javadoc)
     * @see javax.jms.MessageListener#onMessage(javax.jms.Message)
     */
    public void onMessage(Message message) {
        
        
        log.debug("onMessage start");
        
        if(message instanceof ObjectMessage){
            ObjectMessage objectMessage = (ObjectMessage)message;
            Object obj;
			try {
				obj = objectMessage.getObject();
			} catch (JMSException e) {
				log.error("message is incorrect : ", e);
				return;
			}

			if(obj instanceof RunInstructionInfo){
            	log.debug("onMessage get RunInstructionInfo");
                
                RunInstructionInfo info = (RunInstructionInfo)obj;
                
                log.debug("onMessage SessionID=" + info.getSessionId()
                		+ ", JobID=" + info.getJobId()
                		+ ", CommandType=" + info.getCommandType());
                
                if(info.getCommandType() == CommandTypeConstant.CHECK){
                	//受信メッセージ応答
                	log.debug("onMessage CommandType = CHECK");
                    
                    RunResultInfo resultInfo = new RunResultInfo();
                    resultInfo.setSessionId(info.getSessionId());
                    resultInfo.setJobunitId(info.getJobunitId());
                    resultInfo.setJobId(info.getJobId());
                    resultInfo.setFacilityId(info.getFacilityId());
                    resultInfo.setCommandType(info.getCommandType());
	                
                    m_sendQueue.put(resultInfo);
                }
                else if(info.getCommandType() == CommandTypeConstant.DELETE_NORMAL_HISTORY || 
                		info.getCommandType() == CommandTypeConstant.DELETE_STOP_HISTORY){
                	//受信メッセージ応答
                	log.debug("onMessage CommandType = DELETE_NORMAL_HISTORY or DELETE_STOP_HISTORY");
                	
                	//実行履歴削除
                	RunHistoryUtil.delRunHistory(info, m_runHistory);
                }
                else {
                	log.debug("onMessage CommandType != CHECK");
                	
                	//実行履歴チェック
                	Date startDate = RunHistoryUtil.findRunHistory(info, m_runHistory);
                	if(startDate != null){
                		//実行履歴がある場合、開始メッセージを送信する
                		
                		RunResultInfo resultInfo = new RunResultInfo();
                		resultInfo.setSessionId(info.getSessionId());
                		resultInfo.setJobunitId(info.getJobunitId());
                		resultInfo.setJobId(info.getJobId());
                		resultInfo.setFacilityId(info.getFacilityId());
                		resultInfo.setCommand(info.getCommand());
                		resultInfo.setCommandType(info.getCommandType());
                		resultInfo.setStatus(RunStatusConstant.START);
                		resultInfo.setTime(startDate);
                		
                		log.info("run SessionID=" + info.getSessionId() + ", JobID=" + info.getJobId());

                		//送信
                		m_sendQueue.put(resultInfo);
                	}
                	else{
	                	if(info.getCommand().equals(CommandConstant.GET_PUBLIC_KEY) || 
	                			info.getCommand().equals(CommandConstant.ADD_PUBLIC_KEY) || 
	                			info.getCommand().equals(CommandConstant.DELETE_PUBLIC_KEY)){
	                    	//公開鍵用スレッド実行
	                    	log.debug("onMessage CommandType = GET_PUBLIC_KEY or ADD_PUBLIC_KEY or DELETE_PUBLIC_KEY");
	                    	
	                    	PublicKeyThread thread = new PublicKeyThread(info, m_sendQueue, m_runHistory);
	    	                thread.start();
	                    }
	                	else if(info.getCommand().equals(CommandConstant.GET_FILE_LIST)){
	                    	//ファイルリスト用スレッド実行
	                    	log.debug("onMessage CommandType = GET_FILE_LIST");
	                    	
	                    	FileListThread thread = new FileListThread(info, m_sendQueue, m_runHistory);
	    	                thread.start();
	                    }
	                	else if(info.getCommand().equals(CommandConstant.GET_CHECKSUM) || 
	                			info.getCommand().equals(CommandConstant.CHECK_CHECKSUM)){
	                    	//チェックサム用スレッド実行
	                    	log.debug("onMessage CommandType = GET_CHECKSUM or CHECK_CHECKSUM");
	                    	
	                    	CheckSumThread thread = new CheckSumThread(info, m_sendQueue, m_runHistory);
	    	                thread.start();
	                    }
	                	else{
	                    	//コマンド実行
	    	                CommandThread thread = new CommandThread(info, m_sendQueue, m_runHistory);
	    	                thread.start();
	                	}
                	}
                }
            }else if(obj instanceof Dummy){
				// TOPIC接続疎通のためのオブジェクトのため何もしない
				log.debug("onMessage() get Dummy");
				
				// 最終受信日時を格納する
				synchronized (_instanceHeartbeatDateLock) {
					heartbeatReceiveDate = new Date();
				}
            }else{
				log.error("message is incorrect : " + obj.toString());
				return;
            }
        }else{
			log.error("message is incorrect : " + message.getClass());
			return;
        }
        log.debug("onMessage end");
    }
    
	
    /* 通信エラーハンドラ
     * (non-Javadoc)
     * @see javax.jms.ExceptionListener#onException(javax.jms.JMSException)
     */
    public void onException(JMSException arg0) {

    	log.error(arg0.getMessage(), arg0);
    	
    	// syslogにログ出力
		InternalLogger.error("hinemos_jobagent: A problem has been detected with the connection to MGR");
    	
    	m_agent.reConnection();
    }

    /**
	 * @param errFlg
	 *            errFlg を設定。
	 */
	synchronized private void setErrFlg(boolean errFlg) {
		log.debug("setErrFlg() start : errFlg = " + errFlg);
		if (m_errFlg == false && errFlg == true) {

			log.debug("setErrFlg() set ReSetupTask to timer");
			m_future = m_scheduler.scheduleWithFixedDelay(
					new ReSetupTask(), 
					m_jobTopicReconnectionInterval, 
					m_jobTopicReconnectionInterval, 
					TimeUnit.SECONDS);

			// 履歴情報クリアのタイマーを起動する
			if (m_runhistoryClearDelay >= 0) {
				log.debug("setErrFlg() set RunHistoryClearTask to timer");
				m_futureClear = m_schedulerClear.schedule(
						new RunHistoryClearTask(), 
						m_runhistoryClearDelay, 
						TimeUnit.SECONDS);
			}

		}
		m_errFlg = errFlg;
		log.debug("setErrFlg() finish");
	}

	/**
	 * @return errFlg を戻します。
	 */
	synchronized private boolean isErrFlg() {
		return m_errFlg;
	}
	/**
	 *  
	 */
	private boolean reInitial() {
		log.info("reInitial() start");
		terminate();

		boolean ret = false;
		synchronized (this) {
			if (initial()) {

				log.info("reInitial() success!");
				ret = true;

				//エラーフラグ解除
				setErrFlg(false);
				//履歴クリアフラグ解除
				setClearFlg(false);
				
			} else {
				log.warn("reInitial() fail!");
			}
		}
		log.info("reInitial() finish");
		return ret;
	}
	/**
	 * サーバ接続の終了処理
	 *  
	 */
	public void terminate() {
		log.debug("terminate() start");

		terminateSumscriber();

		
        try {
			if (m_session != null)
				m_session.close();
		} catch (JMSException e) {
			log.debug("terminate() session close error", e);
		}

		try {
			if (m_con != null)
				m_con.close();
		} catch (JMSException e1) {
			log.debug("terminate() connection close error", e1);
		}
		log.debug("terminate() finish");
	}
	/**
	 * トピック受信処理の終了 
	 */
	private void terminateSumscriber() {
		try {
			if (m_subscriber != null)
		        m_subscriber.close();
		} catch (JMSException e) {
			log.debug("terminateSumscriber() subscriber close error", e);
		}
	}



	/**
	 * 初期化処理
	 * JMSへの接続、トピック受信設定を行う
	 * @return
	 */
	private boolean initial() {

		log.info("initial() start");

		InitialContext con = null;

		try {
			//InitialContextの生成
			con = new InitialContext(AgentProperties.getProperties());

			//コネクションファクトリ生成
			m_factory = (TopicConnectionFactory) con.lookup(TOPIC_CON_FACTORY);

			//コネクション生成
			if (AgentProperties.getProperty(TOPIC_USER_NAME) != null) {
				//ユーザ認証
				m_con = m_factory.createTopicConnection(
						AgentProperties.getProperty(TOPIC_USER_NAME),
						AgentProperties.getProperty(TOPIC_USER_PASSWORD));
			} else {
				//ユーザ認証なし
				m_con = m_factory.createTopicConnection();
			}

			
			//セッション生成
			m_session = m_con.createTopicSession(false,
					Session.AUTO_ACKNOWLEDGE);

			//メッセージTopic取得
			m_topic = (Topic) con.lookup(TopicConstant.TOPIC_NAME_EXECUTE);

			
			//エラーハンドらセット
	        m_con.setExceptionListener(this);

	        m_con.start();

	        //トピック接続開始
	        initialTopic();
			

		} catch (Exception e) {
			log.error("initial() fail!", e);
			setErrFlg(true);
			return false;
		} finally {
			try {
				if (con != null)
					con.close();
			} catch (Exception e1) {
				log.warn("initial() close fail!", e1);
			}
		}
		log.info("initial() success!");
		return true;

	}

	/**
	 * トピック受信設定
	 * @return
	 * @since
	 */
	private boolean initialTopic() {


		//現在のTopic受信を終了
		terminateSumscriber();
		
		//新しいファシリティIDでトピック受信開始
		try {
	        
	        //ファシリティIDでメッセージセレクターを作成
	        StringBuffer msgSelector = new StringBuffer();
	        for (Iterator iter = m_facilityIdList.iterator(); iter.hasNext();) {
				String facilityId = (String) iter.next();
				msgSelector.append("FacilityId='");
				msgSelector.append(facilityId);
				msgSelector.append("'");
				if( iter.hasNext() ){
					msgSelector.append(" OR ");
				}
			}
	        
	        if(msgSelector.length() != 0){
	        
		        m_subscriber = m_session.createSubscriber(m_topic, msgSelector.toString(), false);
		        
		        //コールバックを登録する
		        m_subscriber.setMessageListener(this);

	        }else{
	        	log.debug("facility is null");
	        }
			

		} catch (Exception e) {
			log.error("TopicInit", e);
			setErrFlg(true);
			return false;
		} finally {
		}
		return true;

	}
	
	/**
	 * 再接続処理を行う
	 */
	synchronized public void reconnect() {
		setErrFlg(true);
	}
	
	/**
	 * 接続状況を返す
	 * @return 接続状況(再接続中 : true, 接続中　: false)
	 */
	synchronized public boolean isReconnecting() {
		return m_errFlg;
	}
	
	/**
	 * Heartbeanメッセージの最終受信日時を取得するメソッド
	 * @return Heartbeatメッセージの最終受信日時
	 */
	public static Date getHeartbeatReceiveDate() {
		synchronized (_instanceHeartbeatDateLock) {
			return heartbeatReceiveDate;
		}
	}
	
	/**
	 * clearFlgの設定
	 * @param clearFlg
	 */
	synchronized private void setClearFlg(boolean clearFlg) {
		m_clearFlg = clearFlg;
	}
	
	/**
	 * clearFlgの取得
	 * @return
	 */
	synchronized static public boolean isHistoryClear(){
		return m_clearFlg;
	}
	
    /**  
     * 実行履歴をログに出力  
     * @return  
     */  
    protected void logHistory() {

            synchronized (m_runHistory) {

                    try {
                            for(String key : m_runHistory.keySet()) {
                                    log.info("A running job is out of control due to stopped agent : " + key);
                            }
                    } catch (ConcurrentModificationException e) {
                            log.warn("Log output process is stopped due to job execution history updated at the same time.");
                    }
            }
    }

    /**
	 * EJB再接続タイマータスク
	 * 通信エラーとなった場合に定周期で呼ばれ再接続を行う 
	 */
	protected class ReSetupTask implements Runnable {

		/**
		 * デフォルトコンストラクタ
		 */
		public ReSetupTask() {
		}
		
		/**
		 * コネクションクローズ
		 */
		public void run() {
			log.debug("ReceiveTopic.ReSetupTask.run() start");
			try {
				if (reInitial()) {
					//このタスクをタイマーから解除
					log.debug("ReceiveTopic.ReSetupTask.run() task cancel");
					if(m_future != null){
						m_future.cancel(true);
					}
				}				
			} catch (Exception e) {
				log.error("ReceiveTopic.ReSetupTask.run()", e);
			}
		}

	}
	
	/**
	 * ジョブ実行履歴削除タイマーマスク
	 * 通信エラーとなった場合に、一定時間後、再度接続確認を行い、
	 * 接続していない場合は、ジョブ履歴情報を削除する
	 */
	protected class RunHistoryClearTask implements Runnable {

		/**
		 * デフォルトコンストラクタ
		 */
		public RunHistoryClearTask() {
		}
		
		/**
		 * ジョブ履歴削除
		 */
		public void run() {
			log.debug("ReceiveTopic.RunHistoryClearTask.run() start");
			try{
				// まだマネージャとの接続が回復していない場合
				if (isErrFlg()) {
					RunHistoryUtil.clearRunHistory(m_runHistory);
					setClearFlg(true);
					log.info("job history was deleted.");
				}
				
				log.debug("ReceiveTopic.RunHistoryClearTask.run() task cancel");
				if(m_futureClear != null){
					m_futureClear.cancel(true);
				}
			} catch (Exception e) {
				log.error("ReceiveTopic.RunHistoryClearTask.run()", e);
			}
		}
		
	}
	
}
