/*
 
Copyright (C) 2009 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.vm.factory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.ejb.entity.FacilityData;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.vm.RepositoryActionInterface;
import com.clustercontrol.vm.VmException;
import com.clustercontrol.vm.bean.VmMethodTypeConstant;
import com.clustercontrol.vm.ejb.entity.VmMethodMstLocal;
import com.clustercontrol.vm.ejb.entity.VmMethodMstPK;
import com.clustercontrol.vm.ejb.entity.VmMethodMstUtil;

/**
 * 仮想化のリポジトリ操作(割当ノード走査)用のコントローラクラス
 * 
 * @version 3.1.0
 * @since 3.1.0
 *
 */
public class VmNodeController {

	protected static Log m_log = LogFactory.getLog(VmNodeController.class);
	
	/** hostId(facilityId)に対するguestのFacilityDataのマップ */
	private HashMap<String, ArrayList<FacilityData>> guestMap = new HashMap<String, ArrayList<FacilityData>>();

	
	/**
	 * 
	 * 指定したホストノードに現在割り当てられているノード一覧を返却する。<BR>
	 * <BR>
	 * 
	 * @param hostId
	 * @param locale
	 * @return
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws VmException
	 */
	public ArrayList<FacilityData> getCurrentAllocationList(
			String hostId,
	        String vmManagementIp,
			String user,
			String password,
			String protocol,
			String virtSolution,
			Locale locale) 
		throws VmException{
			
		m_log.debug("getCurrentAllocationList() : start " +
				"hostId = " + hostId + ", " +
				"vmManagementIp = " + vmManagementIp + ", " +
				"user = " + user + ", " +
				"password = " + password + ", " +
				"protocol = " + protocol + ", " +
				"virtSolution = " + virtSolution);
		
		//戻り値
		ArrayList<FacilityData> guestList = guestMap.get(hostId);

		if(guestList == null){
	        m_log.debug("getCurrentAllocationList() : new list start hostId = " + hostId);
			
			// 実行クラス名
	        String className = null;
	        // 実行クラスインタフェース
	        RepositoryActionInterface action = null;

			// 実行クラス名を取得する
			try{
		        m_log.debug("getCurrentAllocationList() : get instance class name start");
		        
		        // 仮想化ソリューション名が登録され、かつ内部DBにクラス名定義がある場合
		        if(virtSolution != null && virtSolution.length() > 0){
		        	VmMethodMstLocal local =VmMethodMstUtil.getLocalHome().findByPrimaryKey(new VmMethodMstPK(virtSolution,VmMethodTypeConstant.REPOSITORY)); 
		        	
		        	if(local != null){
		            	m_log.debug("getCurrentAllocationList() : className = " + className);
		            	className = local.getClassName();
		        	}
		        	else{
		        		m_log.error("getCurrentAllocationList() : " + virtSolution + "の実行クラスが内部DBに登録されていません。");
		            	throw new VmException(virtSolution + "の実行クラスが内部DBに登録されていません。");
		        	}
	        	//仮想化ソリューション名称が登録されていない
		        }else{
		    		m_log.error("getCurrentAllocationList() : 仮想化ソリューション名称が登録されていません。");
	            	throw new VmException("仮想化ソリューション名称が登録されていません。");
		        }
				
			} catch (Exception e) {
				m_log.error("getCurrentAllocationList() : ",e);
				VmException ve = new VmException("実行クラスが内部DBに登録されていません。");
				ve.setStackTrace(e.getStackTrace());
            	throw ve;
			}
			
			
	        // 対象の想化ソフトのインスタンスを生成
			try {
		        m_log.debug("getCurrentAllocationList() : new instance class start");

		        action = (RepositoryActionInterface)Class.forName(className).newInstance();

	            if(action != null)
	        		m_log.debug("getCurrentAllocationList() : action class is = " + action.getClass().getCanonicalName());
	            else {
	            	m_log.debug("getCurrentAllocationList() : action class is null");
	            	throw new VmException(className + "クラスの生成に失敗しました。");
	            }
				
			} catch (InstantiationException e) {
				m_log.error("getCurrentAllocationList() : ",e);
				VmException ve = new VmException(className + "クラスの生成に失敗しました。");
				ve.setStackTrace(e.getStackTrace());
            	throw ve;
			} catch (IllegalAccessException e) {
				m_log.error("getCurrentAllocationList() : ",e);
				VmException ve = new VmException(className + "クラスの生成に失敗しました。");
				ve.setStackTrace(e.getStackTrace());
            	throw ve;
			} catch (ClassNotFoundException e) {
				m_log.error("getCurrentAllocationList() : ",e);
				VmException ve = new VmException(className + "クラスの生成に失敗しました。");
				ve.setStackTrace(e.getStackTrace());
            	throw ve;
			} catch (Exception e) {
				m_log.error("getCurrentAllocationList() : ",e);
				VmException ve = new VmException(className + "クラスの生成に失敗しました。");
				ve.setStackTrace(e.getStackTrace());
            	throw ve;
			}
			
            
	        // 実行して結果を取得
			try {
		        m_log.debug("getCurrentAllocationList() : run instance class start");

		        guestList = action.getCurrentAllocationList(hostId, vmManagementIp, user, password, protocol);
		        
			} catch (VmException e) {
				m_log.error("getCurrentAllocationList() : ",e);
				throw e;
			} catch (Exception e) {
				m_log.error("getCurrentAllocationList() : ",e);
				VmException ve = new VmException(className + "の処理に失敗しました");
				ve.setStackTrace(e.getStackTrace());
            	throw ve;
			}
			
			// ゲストノードの仮想化ソリューション名をホストノードのものに統一
			Iterator<FacilityData> guestItr = guestList.iterator();
			while (guestItr.hasNext()) {
				FacilityData facilityData = (FacilityData) guestItr.next();
				facilityData.setVirtSolution(virtSolution);
			}
			
			// guestMapに登録する
			guestMap.put(hostId, guestList);
		}
		// 既に値の取得済み
		else{
	        m_log.debug("getCurrentAllocationList() : list exist!! hostId = " + hostId);
		}
		
		
		m_log.debug("getCurrentAllocationList() : end " +
				"hostId = " + hostId + ", " +
				"vmManagementIp = " + vmManagementIp + ", " +
				"user = " + user + ", " +
				"password = " + password + ", " +
				"protocol = " + protocol + ", " +
				"virtSolution = " + virtSolution);
		
		return guestList;
	}
	
	
	/**
	 * リポジトリに登録済みのguestIdのノードが現在hostIdのノードに割り当てられているか<BR>
	 * @param facilityId
	 * @return
	 */
	public boolean isAllocated(String guestId, String hostId, Locale locale) 
		throws CreateException,NamingException,FinderException,VmException{

		m_log.debug("isAllocated() : start guestId = " + guestId + ", hostId = " + hostId);

		boolean ret = false;

		// 対象ゲストノードの情報をリポジトリより取得
		String vmName = null;
		String vmId = null;
		FacilityData hostData = null;
        String vmManagementIp = null;
		String user = null;
		String password = null;
		String protocol = null;
		String virtSolution = null;

		// guestIdを一意に識別する設定値（VMNAME,VMID）をリポジトリより取得する。
		try {
			m_log.debug("isAllocated() : get guestId settings");

			RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
			ArrayList<String> attribute = new ArrayList<String>();
	        attribute.add(FacilityAttributeConstant.VMNAME);
	        attribute.add(FacilityAttributeConstant.VMID);
	        
	        HashMap<String, String> map = repository.getNodeDetail(guestId, attribute);
	        vmName = map.get(FacilityAttributeConstant.VMNAME);
	        vmId = map.get(FacilityAttributeConstant.VMID);
	        
	        if(vmName == null || vmId == null)
	        	return ret;
	        
	        hostData = repository.getNodeFacilityData(hostId);
	        
			virtSolution = hostData.getVirtSolution();
			user = hostData.getVmUser();
			password = hostData.getVmUserpassword();
			protocol = hostData.getVmProtocol();
			if(hostData.getIpProtocolNumber() == null){
				vmManagementIp = hostData.getIpNetworkNumber();
			}else{
				if(hostData.getIpProtocolNumber().intValue() == 4){
					vmManagementIp = hostData.getIpNetworkNumber();				
				}else{
					vmManagementIp = hostData.getIpNetworkNumberV6();
				}
			}
			m_log.debug("isAllocated() : virtSolution = " + virtSolution);
			m_log.debug("isAllocated() : user = " + user);
			m_log.debug("isAllocated() : password = " + password);
			m_log.debug("isAllocated() : protocol = " + protocol);
			m_log.debug("isAllocated() : vmManagementIp = " + vmManagementIp);
		} catch (NamingException e) {
			m_log.error("isAllocated() : ",e);
			throw e;
		} catch (CreateException e) {
			m_log.error("isAllocated() : ",e);
			throw e;
		} catch (FinderException e) {
			m_log.error("isAllocated() : ",e);
			throw e;
		}
		
		// 指定のホストノードに存在するかを確認する
		m_log.debug("isAllocated() : getCurrentAllocationList start");
		ArrayList<FacilityData> guestList = getCurrentAllocationList(
				hostId,
				vmManagementIp,
				user,
				password,
				protocol,
				virtSolution,
				locale);

		// 存在チェック開始
		m_log.debug("isAllocated() : check start");
		Iterator<FacilityData> fdItr = guestList.iterator();
		while (fdItr.hasNext()) {
			FacilityData guest = (FacilityData) fdItr.next();
			
			// 仮想マシン名と仮想マシンIDの合致するものが存在したら
			if(vmName.equals(guest.getVmName()) && vmId.equals(guest.getVmId())){
				m_log.debug("isAllocated() : match node!! vmName = " + guest.getVmName() + ", vmId = " + guest.getVmId());
				ret = true;
			}
			else{
				m_log.debug("isAllocated() : unmatch node!! vmName = " + guest.getVmName() + ", vmId = " + guest.getVmId());
			}
		}
		
		// 戻り値
		m_log.debug("isAllocated() : return = " + ret);
		return ret;
	}
	
}
