/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.util.code;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.PollerProtocolConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryCollectMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryCollectMstPK;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryCollectMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryMstData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCalcMethodMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCalcMethodMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorScopeSnapLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorScopeSnapUtil;
import com.clustercontrol.performance.util.Repository;

/**
 * 収集項目コードの情報を生成するファクトリクラス
 * 
 * @version 3.0
 * @since 1.0
 */
public class CollectorItemCodeTable {
	private static Log m_log = LogFactory.getLog( CollectorItemCodeTable.class );
	private static final HashMap<String, CollectorItemTreeItem> m_codeTable;
	/**
	 * コンストラクター
	 *
	 */
	public CollectorItemCodeTable() {
	}
	
	static {
		m_codeTable = new HashMap<String, CollectorItemTreeItem>(); // 収集項目コードがキー

		// カテゴリコードからカテゴリ情報を参照するためのテーブル
		HashMap<String, CollectorItemTreeItem> categoryTable = 
			new HashMap<String, CollectorItemTreeItem>();
		
		// カテゴリ情報の読み込み
		Collection<CollectorCategoryMstLocal> cate = null;
		try {
			cate = CollectorCategoryMstUtil.getLocalHome().findAll();
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}
		
		Iterator<CollectorCategoryMstLocal> cateItr = cate.iterator();
		while(cateItr.hasNext()){
			CollectorCategoryMstLocal bean = cateItr.next();
			
			CollectorCategoryMstData category  = 
				new CollectorCategoryMstData(bean.getCategoryCode(), bean.getCategoryName());
			
			CollectorItemTreeItem categoryItem = 
				new CollectorItemTreeItem(null, category, null, null);  // 親の要素はないためnull
			
			// カテゴリコードにマッピングするようにカテゴリ情報を登録
			categoryTable.put(bean.getCategoryCode(), categoryItem);
		}

		
		
		try {			
			// 収集項目コードの読み込み
			Collection<CollectorItemCodeMstLocal> ctItemCodeMst = CollectorItemCodeMstUtil.getLocalHome().findAll();			
			
			Iterator<CollectorItemCodeMstLocal> itrItemCode = ctItemCodeMst.iterator();
			
			while(itrItemCode.hasNext()) {
				CollectorItemCodeMstLocal codeBean = itrItemCode.next();
				String itemCode = codeBean.getItemCode();
				
				if(itemCode == null){
					continue;
				}
				
				if(itemCode != null && codeBean.getCategoryCode() != null
						&& codeBean.getDeviceSupport() != null && codeBean.getGraphRange() != null){
					
					// 収集項目コードマスタ情報のDTOを作成
					CollectorItemCodeMstData codeData = new CollectorItemCodeMstData(
							codeBean.getItemCode(),
							codeBean.getCategoryCode(),
							codeBean.getParentItemCode(),
							codeBean.getItemName(),
							codeBean.getMeasure(),
							codeBean.getDeviceSupport(),
							codeBean.getDeviceType(),
							codeBean.getGraphRange()
					);
				
					// カテゴリ名を調べます
					CollectorItemTreeItem categoryTreeItem = categoryTable.get(codeBean.getCategoryCode());
				
					// 親のオブジェクトを取得（存在しない場合はnull）
					CollectorItemTreeItem parentItem = 
						(CollectorItemTreeItem) m_codeTable.get(codeBean.getParentItemCode());
					
					// 親のコードが存在しない場合はカテゴリの直下の要素とする
					if(parentItem == null){
						parentItem = categoryTreeItem;
					}
					
					CollectorItemTreeItem ctItem = 
						new CollectorItemTreeItem(parentItem, codeData, null, null);
					
					// 収集項目コードをキーとして登録
					m_codeTable.put(itemCode, ctItem);
				}
			}
			
		} catch (Exception e) {
			m_codeTable.clear();
			// エラー処理
			e.printStackTrace();
		}
		
	}
	
	
	
	/**
	 * 与えられたコードのセットで収集項目ツリーを生成する
	 * @param codeSet 有効なコードのセット
	 * @param disableCodeSet 無効なコードのセット（現バージョンでは未使用）
	 * @param deviceSet デバイス情報のセット
	 * @return
	 */
	public static CollectorItemTreeItem[] createTree(
			Set<CollectorItemCodeMstData> codeSet, 
			Set<CollectorItemCodeMstData> disableCodeSet,
			String collectMethod,
			Set<CollectorDeviceInfoData> deviceSet){
		ArrayList<CollectorItemTreeItem> itemTree = new ArrayList<CollectorItemTreeItem>();
		
		// 参照しやすいようにカテゴリコードをマップで管理
		HashMap<String, CollectorItemTreeItem> categoryMap = 
			new HashMap<String, CollectorItemTreeItem>();

		// カテゴリ情報の読み込み
		Collection<CollectorCategoryMstLocal> cate = null;
		try {
			cate = CollectorCategoryMstUtil.getLocalHome().findAll();
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}
		
		Iterator<CollectorCategoryMstLocal> cateItr = cate.iterator();
		while(cateItr.hasNext()){
			CollectorCategoryMstLocal bean = cateItr.next();
			
			CollectorCategoryMstData category  = 
				new CollectorCategoryMstData(bean.getCategoryCode(), bean.getCategoryName());
			
			CollectorItemTreeItem categoryItem = 
				new CollectorItemTreeItem(null, category, null, null);  // 親の要素はないためnull
			 
			// カテゴリコードにマッピングするようにカテゴリ情報を登録
			categoryMap.put(bean.getCategoryCode(), categoryItem);
			 
			// リターン用のリストに追加
			itemTree.add(categoryItem);
		}

		// カテゴリコード順にソート
		CollectorItemTreeItem[] list = itemTree.toArray(new CollectorItemTreeItem[0]);
        Arrays.sort(list, new Comparator<CollectorItemTreeItem>(){
            public int compare(CollectorItemTreeItem o1, CollectorItemTreeItem o2) {
                //比較
            	return o1.getCategory().getCategoryCode().compareTo(o2.getCategory().getCategoryCode());
            }
        });
		
		// 処理済の収集項目コードのマップ
		//　キー ： 収集項目コードとデバイス名をセットにして管理するクラスのインスタンス
		// 値　：　収集項目要素
		HashMap<ItemCodeAndDisplayName, CollectorItemTreeItem> createdCodeItemMap = 
			new HashMap<ItemCodeAndDisplayName, CollectorItemTreeItem>();
		
		// 有効な収集項目コードを設定
		Iterator<CollectorItemCodeMstData> itr = codeSet.iterator();
		
		// 生成済みのコードのセット（重複して生成するのを防ぐ）
		HashSet<String> createdCodeSet = new HashSet<String>();
		while(itr.hasNext()){
			CollectorItemCodeMstData itemCodeData = itr.next();
			createItemTreeItem(createdCodeSet, itemCodeData, codeSet, collectMethod, deviceSet, categoryMap, createdCodeItemMap);
		}
		
		// 配列を返す
//		return itemTree.toArray(new CollectorItemTreeItem[0]);
		return list;
	}
	
	/**
	 * 
	 * @param createdCodeSet 既に生成されたコードを保持（重複して生成するのを防ぐため）
	 * @param itemCodeData
	 * @param codeSet
	 * @param collectMethod
	 * @param deviceSet
	 * @param categoryMap
	 * @param createdCodeMap 既に要素のインスタンスが生成されたもののマップ
	 */
	private static void createItemTreeItem(
			HashSet<String> createdCodeSet,
			final CollectorItemCodeMstData itemCodeData,
			final Set<CollectorItemCodeMstData> codeSet, 
			final String collectMethod,
			final Set<CollectorDeviceInfoData> deviceSet,
			final HashMap<String, CollectorItemTreeItem> categoryMap,
			HashMap<ItemCodeAndDisplayName, CollectorItemTreeItem> createdCodeMap){
		// 既に生成済みのコードの場合は抜ける
		if(createdCodeSet.contains(itemCodeData.getItemCode())){
			return;
		}
		
		// 親の項目コード
		String parentItemCode = itemCodeData.getParentItemCode();
		
		// 親のコードがない場合は再起呼び出し終了
		if(parentItemCode == null){
			// カテゴリ直下のコードとして生成するため、親要素はカテゴリ要素とする
			CollectorItemTreeItem parentItem = categoryMap.get(itemCodeData.getCategoryCode());
			
			// カテゴリが存在しない場合
			if(parentItem == null){
				return;
			}
			
			// デバイスサポートの有無をチェック
			if(itemCodeData.getDeviceSupport() == YesNoConstant.TYPE_NO){
				CollectorItemTreeItem item = new CollectorItemTreeItem(parentItem, itemCodeData, collectMethod, null);
				// 作成済みマップに登録する
				createdCodeMap.put(new ItemCodeAndDisplayName(itemCodeData.getItemCode(), null), item);
			} else {
				// デバイスのリスト全てをチェックする
				Iterator<CollectorDeviceInfoData> dItr = deviceSet.iterator();
				while(dItr.hasNext()){
					CollectorDeviceInfoData dData = dItr.next();
					// 収集項目で定義されているデバイスタイプと一致する場合は要素として追加
					if(itemCodeData.getDeviceType().equals(dData.getDeviceType())){
						CollectorItemTreeItem item = new CollectorItemTreeItem(parentItem, itemCodeData, collectMethod, dData);
						// 作成済みマップに登録する
						createdCodeMap.put(new ItemCodeAndDisplayName(itemCodeData.getItemCode(), dData.getDisplayName()), item);
					}
				}
			}
			// 復帰する
		} else {
			// 親のコードがある場合
			// デバイスサポートの有無をチェック
			if(itemCodeData.getDeviceSupport() == YesNoConstant.TYPE_NO){
				// 生成済みの要素から親の要素を取得する
				CollectorItemTreeItem parentItem = 
					createdCodeMap.get(new ItemCodeAndDisplayName(itemCodeData.getParentItemCode(), null));
				
				// まだ生成されていない場合
				if(parentItem == null){
					// 全ての収集項目データセットから親の項目を特定する
					Iterator<CollectorItemCodeMstData> itr = codeSet.iterator();
					while(itr.hasNext()){
						CollectorItemCodeMstData data = itr.next();
						// コードが一致するものが見つかった場合はその要素を生成する（再起呼び出し）
						if(parentItemCode.equals(data.getItemCode())){
							createItemTreeItem(createdCodeSet, data, codeSet, collectMethod, deviceSet, categoryMap, createdCodeMap);
						}
					}
					
					// 生成した親の要素を取得する
					parentItem = createdCodeMap.get(new ItemCodeAndDisplayName(itemCodeData.getParentItemCode(), null));
					
					if(parentItem == null){
						// 親が存在しないため、カテゴリを親要素とする
						parentItem = categoryMap.get(itemCodeData.getCategoryCode());
					}
				}
				
				CollectorItemTreeItem item = new CollectorItemTreeItem(parentItem, itemCodeData, collectMethod, null);
				// 作成済みマップに登録する
				createdCodeMap.put(new ItemCodeAndDisplayName(itemCodeData.getItemCode(), null), item);				
			} else {
				// デバイスをサポートしている場合
				// デバイスのリスト全てをチェックする
				Iterator<CollectorDeviceInfoData> dItr = deviceSet.iterator();
				while(dItr.hasNext()){
					CollectorDeviceInfoData dData = dItr.next();
					// 収集項目で定義されているデバイスタイプと一致する場合は要素として追加
					if(itemCodeData.getDeviceType().equals(dData.getDeviceType())){
						// 生成済みの要素から親の要素を取得する
						CollectorItemTreeItem parentItem = 
							createdCodeMap.get(new ItemCodeAndDisplayName(itemCodeData.getParentItemCode(), dData.getDisplayName()));
						
						// まだ生成されていない場合
						if(parentItem == null){
							// 全ての収集項目データセットから親の項目を特定する
							Iterator<CollectorItemCodeMstData> itr = codeSet.iterator();
							while(itr.hasNext()){
								CollectorItemCodeMstData data = itr.next();
								// コードが一致するものが見つかった場合はその要素を生成する（再起呼び出し）
								if(parentItemCode.equals(data.getItemCode())){
									createItemTreeItem(createdCodeSet, data, codeSet, collectMethod, deviceSet, categoryMap, createdCodeMap);
								}
							}
							
							// 生成した親の要素を取得する
							parentItem = createdCodeMap.get(new ItemCodeAndDisplayName(itemCodeData.getParentItemCode(), dData.getDisplayName()));
							
							if(parentItem == null){
								// 親が存在しないため、カテゴリを親要素とする
								parentItem = categoryMap.get(itemCodeData.getCategoryCode());
							}
						}
						
						CollectorItemTreeItem item = new CollectorItemTreeItem(parentItem, itemCodeData, collectMethod, dData);
						// 作成済みマップに登録する
						createdCodeMap.put(new ItemCodeAndDisplayName(itemCodeData.getItemCode(), dData.getDisplayName()), item);				
					}
				}
			}
		}
		createdCodeSet.add(itemCodeData.getItemCode());
	}

//	private static class ItemCodeAndDeviceName{
//		private String itemCode;
//		private String deviceName;
//		
//		public ItemCodeAndDeviceName(String itemCode, String deviceName){
//			this.itemCode = itemCode;
//			this.deviceName = deviceName;
//		}
//		
//		public boolean equals(Object other) {		
//			if (other instanceof ItemCodeAndDeviceName) {
//				ItemCodeAndDeviceName info = (ItemCodeAndDeviceName)other;
//				if (this.itemCode == null && this.deviceName == null){
//					if (info.itemCode == null && info.deviceName == null){
//						return true;
//					}
//				} else if (this.itemCode == null && this.deviceName != null){
//					if (info.itemCode == null && this.deviceName.equals(info.deviceName)){
//						return true;
//					}
//				} else if (this.itemCode != null && this.deviceName == null){
//					if (this.itemCode.equals(info.itemCode) && info.deviceName == null){
//						return true;
//					}
//				} else {
//					if (this.itemCode.equals(info.itemCode)){
//						return this.deviceName.equals(info.deviceName);
//					}
//				}
//				return false;
//			} else {
//				return false;
//			}
//		}
//
//		public int hashCode() {
//			int result = 17;
//
//			result = 37 * result + ((this.itemCode != null) ? this.itemCode.hashCode() : 0);
//
//			result = 37 * result + ((this.deviceName != null) ? this.deviceName.hashCode() : 0);
//
//			return result;
//		}
//	}
	
	private static class ItemCodeAndDisplayName{
		private String itemCode;
		private String displayName;
		
		public ItemCodeAndDisplayName(String itemCode, String displayName){
			this.itemCode = itemCode;
			this.displayName = displayName;
		}
		
		public boolean equals(Object other) {		
			if (other instanceof ItemCodeAndDisplayName) {
				ItemCodeAndDisplayName info = (ItemCodeAndDisplayName)other;
				if (this.itemCode == null && this.displayName == null){
					if (info.itemCode == null && info.displayName == null){
						return true;
					}
				} else if (this.itemCode == null && this.displayName != null){
					if (info.itemCode == null && this.displayName.equals(info.displayName)){
						return true;
					}
				} else if (this.itemCode != null && this.displayName == null){
					if (this.itemCode.equals(info.itemCode) && info.displayName == null){
						return true;
					}
				} else {
					if (this.itemCode.equals(info.itemCode)){
						return this.displayName.equals(info.displayName);
					}
				}
				return false;
			} else {
				return false;
			}
		}

		public int hashCode() {
			int result = 17;

			result = 37 * result + ((this.itemCode != null) ? this.itemCode.hashCode() : 0);

			result = 37 * result + ((this.displayName != null) ? this.displayName.hashCode() : 0);

			return result;
		}
	}
	
//	public static class DeviceTypeAndDeviceName{
//		private String deviceType;
//		private String deviceName;
//		
//		public DeviceTypeAndDeviceName(String deviceType, String deviceName){
//			this.deviceType = deviceType;
//			if(deviceName != null){
//				this.deviceName = deviceName;
//			}
//			else {
//				this.deviceName = "";
//			}
//		}
//		
//		public boolean equals(Object other) {
//			if (other instanceof DeviceTypeAndDeviceName) {
//				DeviceTypeAndDeviceName info = (DeviceTypeAndDeviceName)other;
//				
//				if (this.deviceType == null && this.deviceName == null){
//					if (info.deviceType == null && info.deviceName == null){
//						return true;
//					}
//				} else if (this.deviceType == null && this.deviceName != null){
//					if (info.deviceType == null && this.deviceName.equals(info.deviceName)){
//						return true;
//					}
//				} else if (this.deviceType != null && this.deviceName == null){
//					if (this.deviceType.equals(info.deviceType) && info.deviceName == null){
//						return true;
//					}
//				} else {
//					if (this.deviceType.equals(info.deviceType)){
//						return this.deviceName.equals(info.deviceName);
//					}
//				}
//				return false;
//			} else {
//				return false;
//			}
//		}
//
//		public int hashCode() {
//			int result = 17;
//
//			result = 37 * result + ((this.deviceType != null) ? this.deviceType.hashCode() : 0);
//
//			result = 37 * result + ((this.deviceName != null) ? this.deviceName.hashCode() : 0);
//
//			return result;
//		}
//
//		public String getDeviceType() {
//			return deviceType;
//		}
//
//		public String getDeviceName() {
//			return deviceName;
//		}
//	}	
	
	public static class DeviceTypeAndDisplayName{
		private String deviceType;
		private String displayName;
		
		public DeviceTypeAndDisplayName(String deviceType, String displayName){
			this.deviceType = deviceType;
			if(displayName != null){
				this.displayName = displayName;
			}
			else {
				this.displayName = "";
			}
		}
		
		public boolean equals(Object other) {
			if (other instanceof DeviceTypeAndDisplayName) {
				DeviceTypeAndDisplayName info = (DeviceTypeAndDisplayName)other;
				
				if (this.deviceType == null && this.displayName == null){
					if (info.deviceType == null && info.displayName == null){
						return true;
					}
				} else if (this.deviceType == null && this.displayName != null){
					if (info.deviceType == null && this.displayName.equals(info.displayName)){
						return true;
					}
				} else if (this.deviceType != null && this.displayName == null){
					if (this.deviceType.equals(info.deviceType) && info.displayName == null){
						return true;
					}
				} else {
					if (this.deviceType.equals(info.deviceType)){
						return this.displayName.equals(info.displayName);
					}
				}
				return false;
			} else {
				return false;
			}
		}

		public int hashCode() {
			int result = 17;

			result = 37 * result + ((this.deviceType != null) ? this.deviceType.hashCode() : 0);

			result = 37 * result + ((this.displayName != null) ? this.displayName.hashCode() : 0);

			return result;
		}

		public String getDeviceType() {
			return deviceType;
		}

		public String getDisplayName() {
			return displayName;
		}
	}	
	
	private static class PlatformIdAndSubPlatformId {
		private String m_platformId;
		private String m_subPlatformId;
		
		public PlatformIdAndSubPlatformId(String platformId, String subPlatformId){
			m_platformId = platformId;
			m_subPlatformId = subPlatformId;
		}
		
		public String getPlatformId(){
			return m_platformId;
		}
		
		public String getSubPlatformId(){
			return m_subPlatformId;
		}
		
		public boolean equals(Object other) {
			if (other instanceof PlatformIdAndSubPlatformId) {
				PlatformIdAndSubPlatformId info = (PlatformIdAndSubPlatformId)other;
				
				if (this.m_platformId == null && this.m_subPlatformId == null){
					if (info.m_platformId == null && info.m_subPlatformId == null){
						return true;
					}
				} else if (this.m_platformId == null && this.m_subPlatformId != null){
					if (info.m_platformId == null && this.m_subPlatformId.equals(info.m_subPlatformId)){
						return true;
					}
				} else if (this.m_platformId != null && this.m_subPlatformId == null){
					if (this.m_platformId.equals(info.m_platformId) && info.m_subPlatformId == null){
						return true;
					}
				} else {
					if (this.m_platformId.equals(info.m_platformId)){
						return this.m_subPlatformId.equals(info.m_subPlatformId);
					}
				}
				return false;
			} else {
				return false;
			}
		}

		public int hashCode() {
			int result = 17;

			result = 37 * result + ((this.m_platformId != null) ? this.m_platformId.hashCode() : 0);

			result = 37 * result + ((this.m_subPlatformId != null) ? this.m_subPlatformId.hashCode() : 0);

			return result;
		}
	}
	
	/**
	 * 指定の収集IDの収集のデバイス情報をDBから取得し収集項目ツリーを生成して返します。
	 * 指定の収集IDがnullの場合はリポジトリからデバイス情報を取得する。
	 * 
	 * @param facilityId ファシリティID
	 * @param collectorId 収集ID
	 * @return
	 */
	public static CollectorItemTreeItem[] getCollectorTree(String facilityId, String collectorId){
		return getCollectorTree(facilityId, collectorId, null);
	}

	/**
	 * 指定の収集IDの収集のデバイス情報をDBから取得し収集項目ツリーを生成して返します。
	 * 指定の収集IDがnullの場合はリポジトリからデバイス情報を取得する。
	 * 
	 * @param facilityId ファシリティID
	 * @param collectorId 収集ID
	 * @param collectMethod 収集方法
	 * @return
	 */
	private static CollectorItemTreeItem[] getCollectorTree(String facilityId, String collectorId, String collectMethod){
		Set<CollectorDeviceInfoData> deviceSet = new HashSet<CollectorDeviceInfoData>();

		// 全てのノードのプラットフォームIDのセット
		Set<PlatformIdAndSubPlatformId> platformCombSet = new HashSet<PlatformIdAndSubPlatformId>();
		
		// いずれかのノードに含まれる（OR条件）デバイス名の情報 (Ver3.1.0現在では未使用)
		Set<DeviceTypeAndDisplayName> deviceSetAdd = new HashSet<DeviceTypeAndDisplayName>();
		
		// 全てのノードに含まれる（AND条件）デバイス名の情報
		Set<DeviceTypeAndDisplayName> deviceSetRetain = null;
		
		// DeviceTypeAndDisplayNameに対応するdeviceName情報
		HashMap<DeviceTypeAndDisplayName, String> deviceNameMap = new HashMap<DeviceTypeAndDisplayName, String>();
		
		if(collectorId == null){
			if(facilityId != null){
				// リポジトリからデバイス情報を取得
				Repository repository = new Repository();
				
				// ファシリティがノードの場合
				if(repository.isNode(facilityId)){
					// プラットフォームIDを取得
					String platformId = Repository.getPlatformId(facilityId);
					String subPlatformId = Repository.getSubPlatformId(facilityId);
					
					PlatformIdAndSubPlatformId platformIdComb = new PlatformIdAndSubPlatformId(platformId, subPlatformId);
					platformCombSet.add(platformIdComb);
					
					List<CollectorDeviceInfoData> deviceList = repository.getDeviceList(facilityId);
					Iterator<CollectorDeviceInfoData> itr = deviceList.iterator();
					
					Set<DeviceTypeAndDisplayName> tmp = new HashSet<DeviceTypeAndDisplayName>();
					while(itr.hasNext()){
						CollectorDeviceInfoData data = itr.next();
						
						DeviceTypeAndDisplayName tan = new DeviceTypeAndDisplayName(data.getDeviceType(), data.getDisplayName());
						tmp.add(tan);
						deviceNameMap.put(tan, data.getDeviceName());
					}
					
					deviceSetAdd.addAll(tmp);
					deviceSetRetain = new HashSet<DeviceTypeAndDisplayName>(tmp);
				}
				// ファシリティがスコープの場合
				else {
					// スコープ配下に存在する全てのノードを取得する
					List<ArrayList<String>> nodeList = repository.getNodeList(facilityId);
					
					// デバイス情報をノード単位で取得
					Iterator<ArrayList<String>> nodeItr = nodeList.iterator();
					
					// スコープ内に存在するノードでループ
					while(nodeItr.hasNext()){
						String nodeFacilityId = (nodeItr.next()).get(0).toString();
						
						// プラットフォームIDを取得
						String platformId = Repository.getPlatformId(nodeFacilityId);
						String subPlatformId = Repository.getSubPlatformId(nodeFacilityId);
						
						PlatformIdAndSubPlatformId platformIdComb = new PlatformIdAndSubPlatformId(platformId, subPlatformId);
						platformCombSet.add(platformIdComb);
						
						// デバイス情報を取得
						List<CollectorDeviceInfoData> deviceList = repository.getDeviceList(nodeFacilityId);		
						Iterator<CollectorDeviceInfoData> itr = deviceList.iterator();
						
						// 現在着目しているノードに含まれる全てのデバイスを格納
						Set<DeviceTypeAndDisplayName> tmp = new HashSet<DeviceTypeAndDisplayName>();
						while(itr.hasNext()){
							CollectorDeviceInfoData data = itr.next();
							DeviceTypeAndDisplayName tan = new DeviceTypeAndDisplayName(data.getDeviceType(), data.getDisplayName());
							tmp.add(tan);
							deviceNameMap.put(tan, data.getDeviceName());
						}
						
						// 全て追加する
						deviceSetAdd.addAll(tmp);
						
						// 同じものだけ追加する
						if(deviceSetRetain == null){
							deviceSetRetain = new HashSet<DeviceTypeAndDisplayName>(tmp);
						} else {
							deviceSetRetain.retainAll(tmp);
						}
					}		
				}
			}
		} else {
			// DBからノード情報を取得
			Collection<CollectorScopeSnapLocal> facilitys;
			try {
				facilitys = CollectorScopeSnapUtil.getLocalHome().findByCollectorId(collectorId);
			} catch (FinderException e) {
				m_log.error(e.getMessage(), e);
				return null;
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
				return null;
			}
			
			// DBから全てのデバイス情報を取得（複数ノードのデバイスが混在する）
			Collection<CollectorDeviceInfoLocal> col;
			try {
				col = CollectorDeviceInfoUtil.getLocalHome().findByCollectorId(collectorId);
			} catch (FinderException e) {
				m_log.error(e.getMessage(), e);
				return null;
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
				return null;
			}
			
			Iterator<CollectorScopeSnapLocal> facilityItr = facilitys.iterator();
			while(facilityItr.hasNext()){
				CollectorScopeSnapLocal bean = facilityItr.next();
				
				// ファシリティ種別がノードの場合
				if(FacilityConstant.TYPE_NODE_STRING.equals(bean.getFacilityType())){
					
					String platformId = bean.getPlatformId();
					String subPlatformId = bean.getSubPlatformId();
					
					platformCombSet.add(new PlatformIdAndSubPlatformId(platformId, subPlatformId));

					// ノード毎のデバイス情報を取得し、
					// どれかに含まれるデバイスセット（deviceSetAdd）、
					// どれにも含まれるデバイスセット（deviceSetRetain）を生成する。
					String targetFacilityId = bean.getFacilityId();

					// デバイス情報をノード単位で取得
					Iterator<CollectorDeviceInfoLocal> nodeItr = col.iterator();

					// ターゲットのノードが保持するデバイスの情報を格納するセット
					Set<DeviceTypeAndDisplayName> tmp = new HashSet<DeviceTypeAndDisplayName>();
					while(nodeItr.hasNext()){
						CollectorDeviceInfoLocal deviceInfo = nodeItr.next();

						// ターゲットのノードのデバイス情報のみ抽出する
						if(targetFacilityId.equals(deviceInfo.getFacilityId())){
							DeviceTypeAndDisplayName tan = 
								new DeviceTypeAndDisplayName(deviceInfo.getDeviceType(), deviceInfo.getDisplayName());
							tmp.add(tan);
							deviceNameMap.put(tan, deviceInfo.getDeviceName());
						}
					}

					// 全て追加する
					deviceSetAdd.addAll(tmp);

					// 同じものだけ追加する
					if(deviceSetRetain == null){
						deviceSetRetain = new HashSet<DeviceTypeAndDisplayName>(tmp);
					} else {
						deviceSetRetain.retainAll(tmp);
					}
				}
			}
		}
		
		// 選択可能な収集項目コード情報のセット
		Set<CollectorItemCodeMstData> codeSet = null;
		
		// 選択不可な収集項目コード情報のセット
		Set<CollectorItemCodeMstData> disableCodeSet = new HashSet<CollectorItemCodeMstData>();
		
		Iterator<PlatformIdAndSubPlatformId> platformItr = platformCombSet.iterator();
		// プラットフォームごとにループをまわす
		while(platformItr.hasNext()){
			PlatformIdAndSubPlatformId platformComb = platformItr.next();

			m_log.debug("getCollectorTree() " +
					"platformId : " + platformComb.getPlatformId() + 
					", subPlatformId : " + platformComb.getSubPlatformId() +
					", collectMethod : " + collectMethod);
			
			// プラットフォーム、サブプラットフォーム単位のカテゴリコード、収集方法を取得
			Collection<CollectorCategoryCollectMstLocal> collects;
			// プラットフォーム、サブプラットフォーム単位の収集方法、アイテムコード、算出方法を取得
			Collection<CollectorItemCalcMethodMstLocal> calsMethods;
			try {
				collects = CollectorCategoryCollectMstUtil.getLocalHome().findByPlatformIdAndSubPlatformId(
						platformComb.getPlatformId(),
						platformComb.getSubPlatformId());
				
				calsMethods = CollectorItemCalcMethodMstUtil.getLocalHome().findByPlatformIdAndSubPlatformId(
						platformComb.getPlatformId(),
						platformComb.getSubPlatformId());
			} catch (FinderException e) {
				m_log.error(e.getMessage(), e);
				return null;
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
				return null;
			}
			
			// categoryCodeをキーにcollectMethodを取得するマップ
			HashMap<String, String> categoryMap = new HashMap<String, String>();
			Iterator<CollectorCategoryCollectMstLocal> collectsItr = collects.iterator();
			while(collectsItr.hasNext()) {
				CollectorCategoryCollectMstLocal collect = collectsItr.next();
				categoryMap.put(collect.getCategoryCode(), collect.getCollectMethod());
			}
			
			// プラットフォームごとに収集可能な収集項目コードのリストを求める
			HashSet<CollectorItemCodeMstData> tmpByPlatform = new HashSet<CollectorItemCodeMstData>();
			
			// 計算方法管理テーブルの要素ごとにループをまわす
			Iterator<CollectorItemCalcMethodMstLocal> itr = calsMethods.iterator();
			while(itr.hasNext()){
				CollectorItemCalcMethodMstLocal bean = itr.next();
					
				// 収集方法に拠らない場合
				if (collectMethod == null || collectMethod.equals("")){
					
					CollectorItemCodeMstLocal codeBean;
					try {
						codeBean = CollectorItemCodeMstUtil.getLocalHome().findByPrimaryKey(bean.getItemCode());
					} catch (FinderException e) {
						m_log.error(e.getMessage(), e);
						return null;
					} catch (NamingException e) {
						m_log.error(e.getMessage(), e);
						return null;
					}
					
					// カテゴリ別収集情報の収集方法と収集項目マスタの収集方法を比較し、同じならば追加する
					if(categoryMap.get(codeBean.getCategoryCode()) != null && 
							categoryMap.get(codeBean.getCategoryCode()).equals(bean.getCollectMethod())) {
						// 収集項目コードマスタ情報のDTOを作成
						CollectorItemCodeMstData codeData = new CollectorItemCodeMstData(
								codeBean.getItemCode(),
								codeBean.getCategoryCode(),
								codeBean.getParentItemCode(),
								codeBean.getItemName(),
								codeBean.getMeasure(),
								codeBean.getDeviceSupport(),
								codeBean.getDeviceType(),
								codeBean.getGraphRange()
						);
						
						tmpByPlatform.add(codeData);
					}
					
				}
				// 収集方法が一致する場合(収集方法が指定されている場合)
				else if(collectMethod.equals(bean.getCollectMethod())){
					CollectorItemCodeMstLocal codeBean;
					try {
						codeBean = CollectorItemCodeMstUtil.getLocalHome().findByPrimaryKey(bean.getItemCode());
					} catch (FinderException e) {
						m_log.error(e.getMessage(), e);
						return null;
					} catch (NamingException e) {
						m_log.error(e.getMessage(), e);
						return null;
					}

					// カテゴリ別収集情報の収集方法と収集項目マスタの収集方法を比較し、同じならば追加する
					if(categoryMap.get(codeBean.getCategoryCode()) != null && 
							categoryMap.get(codeBean.getCategoryCode()).equals(bean.getCollectMethod())) {
						// 収集項目コードマスタ情報のDTOを作成
						CollectorItemCodeMstData codeData = new CollectorItemCodeMstData(
								codeBean.getItemCode(),
								codeBean.getCategoryCode(),
								codeBean.getParentItemCode(),
								codeBean.getItemName(),
								codeBean.getMeasure(),
								codeBean.getDeviceSupport(),
								codeBean.getDeviceType(),
								codeBean.getGraphRange()
						);
					
						tmpByPlatform.add(codeData);
					}
				}
			}
			// プラットフォームごとに収集可能な項目をANDもしくはOR条件でセットに格納する
			// 同じものだけ追加する(AND条件)
			if(codeSet == null){
				codeSet = new HashSet<CollectorItemCodeMstData>(tmpByPlatform);
			} else {
				codeSet.retainAll(tmpByPlatform);
			}
			
			// まずは全て追加する（あとでcodeSetに含まれる要素は削除する）
			disableCodeSet.addAll(tmpByPlatform);
		}

		// デバイスがひとつもない場合はここで空のセットを初期化する
		if(deviceSetRetain == null){
			deviceSetRetain = new HashSet<DeviceTypeAndDisplayName>();
		}
		
		// デバイス情報のセットを生成
		// デバイス情報を設定
		Iterator<DeviceTypeAndDisplayName> deviceItr = deviceSetRetain.iterator();
		while(deviceItr.hasNext()){
			DeviceTypeAndDisplayName info = deviceItr.next();

			// デバイス名とデバイス種別のみのインスタンスを生成
			CollectorDeviceInfoData deviceData = new CollectorDeviceInfoData(
					null,  // 収集ID
					null,  // ファシリティID
					deviceNameMap.get(info),
					info.getDisplayName(),
					null,
					null,
					info.getDeviceType());
			
			deviceSet.add(deviceData);
		}

		// コードがひとつもない場合はここで空のセットを初期化する
		if(codeSet == null){
			codeSet = new HashSet<CollectorItemCodeMstData>();
		}
		
		// 全ての項目からcodeSetに含まれる要素は削除する
		disableCodeSet.removeAll(codeSet);
		
		return createTree(codeSet, disableCodeSet, collectMethod, deviceSet);
	}
	
	/**
	 * 収集項目コードとリポジトリ表示名から収集項目表示用文字列を生成し返します。
	 * 形式 ：　収集名[リポジトリ表示名]
	 * 
	 * @param itemCode 収集項目コード
	 * @param displayName リポジトリ表示名
	 * @return 収集項目表示
	 */
	public static String getFullItemName(String itemCode, String displayName){
		try {
			CollectorItemCodeMstLocal bean = CollectorItemCodeMstUtil.getLocalHome().findByPrimaryKey(itemCode);
			String itemName = bean.getItemName();
			if(bean.getDeviceSupport().intValue() == YesNoConstant.TYPE_YES){
				itemName = itemName + "[" + displayName + "]";
			}
			return itemName;
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}
		return itemCode + " is not found.";
	}
	
	/**
	 * 収集項目コードをキーにした、CollectorItemTreeItemのHashMapを返します。
	 * @return 
	 */
	public static HashMap<String, CollectorItemTreeItem> getItemCodeMap(){
		return m_codeTable;
	}
	
}