/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.composite;

import java.awt.Color;
import java.awt.Frame;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.awt.SWT_AWT;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.StackedXYAreaRenderer;
import org.jfree.chart.renderer.xy.StandardXYItemRenderer;
import org.jfree.chart.renderer.xy.XYAreaRenderer;
import org.jfree.data.time.Second;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;

import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.performance.action.DrawRecordGraphAction;
import com.clustercontrol.performance.bean.GraphConstant;
import com.clustercontrol.performance.bean.GraphProperty;
import com.clustercontrol.performance.bean.CollectedDataInfo;
import com.clustercontrol.performance.bean.CollectorItemInfo;
import com.clustercontrol.performance.ejb.bmp.RecordCollectorData;
import com.clustercontrol.performance.util.Messages;

/**
 * 実績グラフを表示するコンポジット
 * 
 * @version 1.0
 * @since 1.0
 *  
 */
public class RecordGraphComposite extends Composite {
    private Composite composite = null;

//    private ScrollBar scrollBar;

    private int interval;

    // 収集期間（秒）
    private int periodSecond;

    // 表示期間の基準となる時刻（収集を開始した時刻）
    private double lowerBoundBase;

    private DrawRecordGraphAction m_action;

    private int m_plot = GraphConstant.REALTIME_GRAPH_MAX_PLOT;

    private Frame m_graphFrame;

    private String m_title; // タイトル

    private String m_timeAxisLabel; // 横軸のラベル

    private String m_valueAxisLabel; // 縦軸のラベル

    private JFreeChart m_jfreechart;

    private TimeSeriesCollection m_timeSeriesCollection; // データを保持する

    private final int GRAPH_MAX = 10;

    private TimeSeries[] m_timeseries; // サブスコープグラフ表示のために色の順序を設定した TimeSeries
                                       // を保持

    private Label infoLabel;
    
    // グラフの表示設定を保持
    private GraphProperty graphProperty;

    /**
     * インスタンスを返します。
     * 
     * @param parent 親のコンポジット
     */
    public RecordGraphComposite(final Composite parent) {
        super(parent, SWT.NONE);

        initialize();
    }

    /**
     * グラフを表示しない場合。
     * メッセージを記述したインスタンスを返します。
     * 
     * @param parent
     * @param label	グラフを表示しない理由
     */
    public RecordGraphComposite(final Composite parent, String label) {
        super(parent, SWT.NONE);

        setLabelInfo(label);
    }
    
    /**
     * グラフ描画用のデータを設定します。(グラフ表示対象の収集が変更になった場合に呼びます)
     * 
     * @param collectorData 実績収集定義情報
     * @param itemInfoList 収集項目のリスト
     * @param facilityTree  ファシリティツリー
     */
    public void setMasterData(RecordCollectorData collectorData,
            List<CollectorItemInfo> itemInfoList) {

        this.m_action.setData(collectorData, itemInfoList);
    }

    /**
     * 指定の設定でグラフを表示します。（グラフ表示対象の収集項目が変更になった場合に呼びます）
     * 
     * @param graphProperty グラフ描画設定
     */
    public void drawGraph(GraphProperty graphProperty, Date startTime, Date endTime) {

        this.graphProperty = graphProperty;
        
        // グラフ描画
        this.m_action.drawGraph(
        		graphProperty.getTargetFacility(),
                graphProperty.getTargetItem(),
                graphProperty.getGraphType(),
                graphProperty.getGraphForm(),
                startTime,
                endTime);
    }

    /**
     * グラフ種別を返します。
     *  
     */
    public int getGraphType() {
        return this.m_action.getGraphType();
    }

    /**
     * グラフの形を返します。
     *  
     */
    public int getGraphForm() {
        return this.m_action.getGraphForm();
    }

    /**
     * 対象ファシリティのファシリティツリー項目を返します。
     * 
     * @return 対象ファシリティのファシリティツリー項目
     */
    public FacilityTreeItem getTargetFacility() {
        return this.m_action.getTargetFacility();
    }

    /**
     * 初期化を行います。
     *
     */
    private void initialize() {
        this.setLayout(new FillLayout());

        m_timeSeriesCollection = new TimeSeriesCollection();

        // ここでサブスコープグラフ表示用にSeriesを10個準備します
        m_timeseries = new TimeSeries[GRAPH_MAX];

        for (int i = 0; i < m_timeseries.length; i++) {
            m_timeseries[i] = new TimeSeries("", Second.class);
        }

        composite = new Composite(this, SWT.EMBEDDED);
		
//		scrollBar = this.getHorizontalBar();
//        scrollBar.setIncrement(10); // 増減幅

        // アクションクラスを設定します。
        m_action = new DrawRecordGraphAction();
//        scrollBar.addSelectionListener(m_action);
        m_action.setComposite(this);
    }

    /**
     * JFreeChartのグラフ描画用ChartPanelを生成します。
     * 
     * @param graphType グラフ種別
     * @param rangeFixed グラフの縦軸が固定か否かを指定（固定の場合は、0-100）
     */
    private void createChart(final int graphType, final int graphForm,
    		final boolean rangeFixed) {
        boolean showDetail;
        if (graphType == GraphProperty.TYPE2 || graphType == GraphProperty.TYPE3) {
            showDetail = true;
        } else {
            showDetail = false;
        }

        m_jfreechart = ChartFactory.createTimeSeriesChart(m_title,
                m_timeAxisLabel, m_valueAxisLabel, m_timeSeriesCollection,
                //				true, true, false);
                showDetail, true, false);

        // 初期表示範囲を設定
        // 収集回数が指定プロット回数よりも少ない場合は、収集期間を表示期間とする
        double range = Math.min(this.periodSecond * 1000D, this.interval
                * 1000D * this.m_plot);
        setRange(0, range);

        XYPlot xyplot = m_jfreechart.getXYPlot();
		
        // レンダラと色を選択
		Color[] color = {Color.RED, Color.BLUE, Color.GREEN, Color.ORANGE, Color.CYAN,
				Color.MAGENTA, Color.YELLOW, Color.PINK, Color.GRAY, Color.BLACK,
				Color.LIGHT_GRAY, Color.DARK_GRAY, Color.WHITE};
        if (graphForm == GraphProperty.FORM2) {
        	XYAreaRenderer renderer = new XYAreaRenderer();
			/** グラフの描画色を固定する */
			for (int i = 0; i < 10; i++) {
				renderer.setSeriesPaint(i, color[i]);
			}
        	xyplot.setRenderer(renderer); // グラフを積み上げ面表示にする。(初期値は折線グラフ)
        } else {
            StandardXYItemRenderer renderer = new StandardXYItemRenderer();
			/** グラフの描画色を固定する */
			for (int i = 0; i < 10; i++) {
				renderer.setSeriesPaint(i, color[i]);
			}
            xyplot.setRenderer(renderer); // グラフを折れ線グラフにする。
        }

        // 百分率の場合は0-100。しかし、(積み上げ面かつサブスコープ)の場合は除く。
        // それ以外は下限が0で、上限はオート
        if (rangeFixed && (graphForm != GraphProperty.FORM2 || graphType != GraphProperty.TYPE3)) {
            ValueAxis valueaxis = xyplot.getRangeAxis();
            valueaxis.setRange(0.0D, 100D);
        } else {
            NumberAxis valueaxis = (NumberAxis)(xyplot.getRangeAxis());
            valueaxis.setAutoRangeIncludesZero(true);
        }

        ChartPanel chartPanel = new ChartPanel(m_jfreechart);

        if (m_graphFrame == null) {
            m_graphFrame = SWT_AWT.new_Frame(composite);
        }

        m_graphFrame.add(chartPanel);
    }

    /**
     * グラフを作成します。
     * 
     * @param title グラフのタイトル
     * @param timeAxisLabel 横軸のラベル
     * @param valueAxisLabel 縦軸のラベル
     * @param graphType グラフ種別
     * @param rangeFixed グラフの縦軸が固定か否かを指定（固定の場合は、0-100）
     */
    public void createGraph(
    		final String title, 
    		final String timeAxisLabel,
            final String valueAxisLabel, 
            final int graphType,
            final int graphForm,
            final boolean rangeFixed, 
            final int interval, 
            final Date startDate,
            final Date stopDate) {

        m_title = title; // タイトル
        m_timeAxisLabel = timeAxisLabel; // 横軸のラベル
        m_valueAxisLabel = valueAxisLabel; // 縦軸のラベル

        setTimeScale(interval, startDate, stopDate); // 時間軸のスクロール設定

        // 既にグラフが生成されている場合は削除
        if (m_graphFrame != null) {
            m_graphFrame.removeAll();
            m_graphFrame = null;
        }

        createChart(graphType, graphForm, rangeFixed);
    }

    /**
     * グラフのデータを設定します。
     * 
     * @param name 凡例
     * @param dataSet 描画対象の性能値データのリスト
     */
    public void setGraphData(String name, List<CollectedDataInfo> dataSet) {
        TimeSeries timeseries = m_timeSeriesCollection.getSeries(name);

        if (timeseries == null) {
            timeseries = new TimeSeries(name, Second.class);
            m_timeSeriesCollection.addSeries(timeseries);
        }

        Iterator<CollectedDataInfo> itr = dataSet.iterator();
        while (itr.hasNext()) {
            CollectedDataInfo data = itr.next();

            if (!Double.isNaN(data.getValue())) {
                if (data.getValue() >= 0.0) {
                    timeseries.addOrUpdate(new Second(data.getDate()), data
                            .getValue());
                }
            }
        }
    }

    /**
     * サブスコープグラフのデータを設定します。
     * 
     * @param name 凡例
     * @param dataSet 描画対象の性能値データのリスト
     */
    public void setSubscopeGraphData(int index, List<CollectedDataInfo> dataSet, List<String> facilityNames) {
//        // すべて同じファシリティIDであるため最初の要素のファシリティIDを取得
//        String name = ((CollectedDataInfo) dataSet.get(0)).getFacilityId();

    	// ファシリティ名（ファシリティIDではない）で、TimeSeriesを引く
        TimeSeries timeseries = m_timeSeriesCollection.getSeries(facilityNames.get(index));

        // 指定のファシリティ名で登録されていない場合
        if (timeseries == null) {
//          m_timeseries[index].setName(name);
        	m_timeseries[index].setKey(facilityNames.get(index)); // キーにファシリティ名を設定
            m_timeSeriesCollection.addSeries(m_timeseries[index]);
        }

        Iterator<CollectedDataInfo> itr = dataSet.iterator();

        while (itr.hasNext()) {
            CollectedDataInfo data = itr.next();

            if (!Double.isNaN(data.getValue())) {
                if (data.getValue() >= 0.0) {
                    m_timeseries[index].addOrUpdate(new Second(data.getDate()),
                            data.getValue());
                }
            }
        }
    }

    /**
     * グラフのデータを消去します。
     *  
     */
    public void clearData() {
        m_timeSeriesCollection.removeAllSeries();
    }

    /**
     * 時間尺度を設定します。
     * 
     * @param startDate
     * @param endDate
     */
    private void setTimeScale(int interval, final Date startDate,
            final Date endDate) {
        this.interval = interval;
        // １番最初の時刻の値は計算できないため、２番目の時刻からを表示する
//        this.lowerBoundBase = startDate.getTime() + interval * 1000;
        // １番最初の時刻の値から表示可能となっている
        this.lowerBoundBase = startDate.getTime();
        long upperBound = endDate.getTime();
        // 収集期間を設定
        this.periodSecond = (int) ((upperBound - lowerBoundBase) / 1000);
    }

    /**
     * 表示部を変更します。
     * 
     * @param lowerBound 描画対象データのうち表示対象とする時間の開始時刻
     * @param upperBound 描画対象データのうち表示対象とする時間の終了時刻
     */
    public void setRange(double lowerBound, double upperBound) {
        ValueAxis domainAxis = m_jfreechart.getXYPlot().getDomainAxis();

        domainAxis.setRange(this.lowerBoundBase + lowerBound,
                this.lowerBoundBase + upperBound);
    }

    /**
     * 描画するデータのプロット数を取得します。
     * 
     * @return プロット数
     */
    public int getPlot() {
        return m_plot;
    }

    /**
     * 描画するデータのプロット数を設定します。
     * 
     * @param プロット数
     */
    public void setPlot(int plot) {
        this.m_plot = plot;
    }

    /**
     * グラフ描画定義情報 を取得します。
     * 
     * @return graphProperty　グラフ描画定義情報
     */
    public GraphProperty getGraphProperty() {
        return graphProperty;
    }
    
    /**
     * グラフを表示しない場合、理由を記述したlabelを設定します。
     * 
     * @param message グラフを表示しない理由
     */
    private void setLabelInfo(String message){
    	this.setLayout(new FillLayout(SWT.VERTICAL));
        infoLabel = new Label(this, SWT.NONE);
        infoLabel.setText(message);	
    }
    
}