/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.logagent;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.logagent.util.EjbConnectionManager;
import com.clustercontrol.repository.ejb.session.RepositoryController;

/**
 * ログ転送エージェントメインクラス<BR>
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class Agent {
	
	/** プラグインID */
	static public final String PLUGIN_ID = "LOGAGENT";
	
	private Properties m_props;
	
	private EjbConnectionManager m_ejbConnectionManager;
	
	private ReceiveTopic m_receiveTopic;
	
	private SendQueue m_sendQueue;

	private UpdateRepositoryInfoReceiveTopic m_updateRepository;

	/** ファシリティID更新用タイマー * */
	protected Timer m_timer = new Timer(true);
	
	private TransferLogManager m_transferManager;
	
	// マネージャへのポーリング周期（秒）
	private int m_pollingInterval = 10;
	
	
	
	//ロガー
	static private Log log = LogFactory.getLog(Agent.class);
	
	
	
	/**
	 * メイン処理<BR>
	 * 
	 * ログ転送エージェントを起動するとこのメソッドがコールされる。
	 * 
	 * @param args プロパティファイル名
	 */
	public static void main(String[] args) throws Exception{
		
		log.info("starting Hinemos log agent...");
		log.info("java.vm.version = " + System.getProperty("java.vm.version"));
		log.info("java.vm.vendor = " + System.getProperty("java.vm.vendor"));
		log.info("java.home = " + System.getProperty("java.home"));
		log.info("os.name = " + System.getProperty("os.name"));
		log.info("os.arch = " + System.getProperty("os.arch"));
		// 起動ユーザを表示
		log.info("user.name = " + System.getProperty("user.name"));
		// 作業ディレクトリを表示
		log.info("user.dir = " + System.getProperty("user.dir"));
		
		Agent agent = new Agent(args[0]);
		
		//エージェント処理開始
		agent.exec();
		
		log.info("Hinemos log agent started");
		
		//終了待ち
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		while (true) {
			try {
				String line = br.readLine();
				if (line.equals("quit") ) {
					break;
				}
			} catch (Exception e) {
			}
			synchronized (agent) {
				agent.wait();
			} 
		}
		
		
		//------------
		//-- 終了処理
		//------------
		
		agent.terminate();
		
		
	}
	
	/**
	 * コンストラクタ
	 */
	public Agent(String propFileName) throws Exception{
		
		//------------
		//-- 初期処理
		//------------
		
		//プロパティファイル読み込み
		m_props = new Properties();
		try {
			m_props.load(new FileInputStream(propFileName));
		} catch (Exception e) {
			log.error(e);
			throw e;
		}
		
		// マネージャへのポーリング周期（秒）
		String pollingInterval = m_props.getProperty("manager.polling.interval");
		if (pollingInterval != null) {
			try {
				m_pollingInterval = Integer.parseInt(pollingInterval);
			} catch (NumberFormatException e) {
				log.error("manager.polling.interval",e);
			}
		}
		
		m_ejbConnectionManager = new EjbConnectionManager(m_props);
		m_sendQueue = new SendQueue(this, m_props);
		m_transferManager = new TransferLogManager(m_ejbConnectionManager, m_sendQueue, m_props);
		
		//終了呼び出し設定
		Runtime.getRuntime().addShutdownHook(new Thread() {
			public void run() { terminate(); }
		});
	}
	
	
	/**
	 * ログ転送エージェント処理を実行します。<BR>
	 * 
	 * トピックへの接続を行う。 
	 */
	public void exec() {
		//-----------------
		//-- トピック接続
		//-----------------
		if (log.isDebugEnabled())
			log.debug("トピック接続");
		
		//受信トピック生成
		try {
			m_receiveTopic = new ReceiveTopic(this, getFacility(), m_transferManager, m_sendQueue, m_props);
		} catch (RemoteException e) {
			// マネージャに接続できなかった場合
			HashMap<String, String> map = new HashMap<String, String>();
			m_receiveTopic = new ReceiveTopic(this, map.values(), m_transferManager, m_sendQueue, m_props);
			
			log.error("exec(): Unable to connect to Hinemos manager. ",e);
		} catch (NamingException e) {
			// マネージャに接続できなかった場合
			HashMap<String, String> map = new HashMap<String, String>();
			m_receiveTopic = new ReceiveTopic(this, map.values(), m_transferManager, m_sendQueue, m_props);
			
			log.error("exec(): Unable to connect to Hinemos manager. ",e);
		} catch (CreateException e) {
			// マネージャに接続できなかった場合
			HashMap<String, String> map = new HashMap<String, String>();
			m_receiveTopic = new ReceiveTopic(this, map.values(), m_transferManager, m_sendQueue, m_props);
			
			log.error("exec(): Unable to connect to Hinemos manager. ",e);
		} catch (Exception e) {
			// 上記以外の例外が発生した場合
			HashMap<String, String> map = new HashMap<String, String>();
			m_receiveTopic = new ReceiveTopic(this, map.values(), m_transferManager, m_sendQueue, m_props);
			
			log.error("exec(): Other exceptions were generated. ",e);
		}	
		
		
		m_updateRepository = new UpdateRepositoryInfoReceiveTopic(this, m_props);

		// キャッシュ更新タイマー開始
		m_timer.schedule(new ReflashFilterTask(), m_pollingInterval*1000, m_pollingInterval*1000);
	}
	
	
	/**
	 * 全ログファイルの転送を終了します。<BR>
	 *  
	 */
	public void terminateTransfer() {
		
		// ログ転送停止
		m_transferManager.stopTransfer();
	}
	
	

	/**
	 * ファシリティIDを設定します。<BR>
	 * 
	 * Hinemosでは、ファシリティIDを定期的に取得し
	 * セットします。
	 * 
	 */
	public void setFacility() {
		
		try {
			m_receiveTopic.setFacilityIdList(getFacility());
		} catch (RemoteException e) {
			// マネージャに接続できなかった場合			
			log.error("Unable to connect to Hinemos manager. ",e);
		} catch (NamingException e) {
			// マネージャに接続できなかった場合
			log.error("Unable to connect to Hinemos manager. ",e);
		} catch (CreateException e) {
			// マネージャに接続できなかった場合
			log.error("Unable to connect to Hinemos manager. ",e);
		} catch (Exception e) {
			// 上記以外の例外が発生した場合
			log.error("Other exceptions were generated. ",e);
		}
		
	}
	
	/**
	 * ファシリティIDを取得します。<BR>
	 * 
	 * 自局のIPアドレスと、ホスト名でリポジトリからファシリティIDを取得します。
	 * 
	 * @return　ファシリティIDのCollection
	 * @throws Exception 
	 * @since
	 */
	public Collection<String> getFacility() throws Exception {
		
		//RepositoryControllerを取得
		
		HashMap<String, String> map = new HashMap<String, String>();
		
		RepositoryController repository = null;
		try {
			repository = m_ejbConnectionManager.getRepositoryController();
		} catch (Exception e1) {
			throw e1;
		}
		
		try {
			//ネットワーク情報取得
			Enumeration networkInterfaces = NetworkInterface
			.getNetworkInterfaces();
			if (null != networkInterfaces) {
				
				while (networkInterfaces.hasMoreElements()) {
					
					NetworkInterface ni = (NetworkInterface) networkInterfaces
					.nextElement();
					
					if (log.isDebugEnabled()) {
						try {
							log.debug("ネットワーク検出:\t"
									+ new String(ni.getDisplayName().getBytes(
									"iso-8859-1")));
						} catch (UnsupportedEncodingException e) {
							log.debug("ネットワーク検出:\t", e);
						}
					}
					
					Enumeration inetAddresses = ni.getInetAddresses();
					while (inetAddresses.hasMoreElements()) {
						
						InetAddress in4 = (InetAddress) inetAddresses.nextElement();
						
						if (log.isDebugEnabled())
							log.debug("IP/HOST検出:\t" + in4.getHostAddress()
									+ "/" + in4.getHostName());
						
						//ローカルホスト以外の時場合
						if (in4.getHostAddress().compareTo("127.0.0.1") != 0) {
							
							//IPとホスト名で、ファイシリティIDを特定
							ArrayList facilityIdList = repository.getFacilityIdList(in4.getHostName(), in4.getHostAddress());
							
							if (log.isDebugEnabled()) {
								if (facilityIdList.size() == 0) {
									log.debug("対応ファシリティなし");
								}
							}
							
							for (Iterator iter = facilityIdList.iterator(); iter.hasNext();) {
								String facilityId = (String) iter.next();
								
								if (log.isDebugEnabled())
									log.debug("対応ファシリティ:" + facilityId);
								
								//ファシリティの対象に追加
								if (map.containsKey(facilityId) == false) {
									map.put(facilityId, facilityId);
								}
							}
						}
					}
					
				}
			}
			
			
		} catch (SocketException e) {
			log.debug("NetWorkInterfacesの情報がありません");
			log.error(e.getMessage(), e);
			
		} catch (RemoteException e) {
			log.debug("ファシリティ情報取得失敗");
			log.error(e.getMessage(), e);
			
		} catch (FinderException e) {
			log.debug("ファシリティ情報取得失敗");
			log.error(e.getMessage(), e);
			
		} catch (NamingException e) {
			log.debug("ファシリティ情報取得失敗");
			log.error(e.getMessage(), e);
		}
		
		return map.values();
		
	}

	/**
	 * ログ転送エージェントを終了します。<BR>
	 */
	public void terminate() {
		
		m_receiveTopic.terminate();
		
		m_updateRepository.terminate();
		
		m_sendQueue.terminate();
		
		m_timer.cancel();
		
		log.info("Hinemos log agent stopped");
		
	}
	
	/**
	 * 全てのTopicとQueueの再接続処理を行う
	 * 
	 */
	synchronized public void reConnection() {
		
		log.info("Hinemos log agent reconnect start");
		
		m_receiveTopic.reconnect();
		
		m_updateRepository.reconnect();
		
		m_sendQueue.reconnect();
		
		log.info("Hinemos log agent reconnect end");
	}
	
	/**
	 * ファシリティID再取得タイマータスク<BR>
	 */
	protected class ReflashFilterTask extends TimerTask {

		/**
		 * キャッシュ更新
		 */
		public void run() {

			setFacility();

		}
	}
}
