/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterPK;

/**
 * ジョブ待ち条件ジョブマスタのDAOインターフェースを実装するクラス<BR>
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterBean
 * @see com.clustercontrol.jobmanagement.dao.JobStartJobMasterDAO
 */
public class JobStartJobMasterDAOImpl implements JobStartJobMasterDAO {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(JobStartJobMasterDAOImpl.class);
	
	public void init() {
	}
	
	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobMasterDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterPK, com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterBean)
	 */
	public void load(JobStartJobMasterPK pk, JobStartJobMasterBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_start_job_mst WHERE job_id = ? AND target_job_id = ? AND target_job_type = ? AND target_job_end_value = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			stmt.setString(2, pk.getTarget_job_id());
			stmt.setInt(3, pk.getTarget_job_type());
			stmt.setInt(4, pk.getTarget_job_end_value());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setJob_id(res.getString("job_id"));
				ejb.setTarget_job_end_value(res.getInt("target_job_end_value"));
				ejb.setTarget_job_type(res.getInt("target_job_type"));
				ejb.setTarget_job_id(res.getString("target_job_id"));
			} else {
				String msg = "JobStartJobMaster data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}
	
	/**
	 * データベースの更新を行うのが基本だが、<BR>
	 * 全てのカラムが主キーになっているので、更新を行う必要はない。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobMasterDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterBean)
	 */
	public void store(JobStartJobMasterBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getJob_id() + ", " + ejb.getTarget_job_id() + ", " + ejb.getTarget_job_type() + ", " + ejb.getTarget_job_end_value());
		
		m_log.debug("store() end : " + ejb.getJob_id() + ", " + ejb.getTarget_job_id() + ", " + ejb.getTarget_job_type() + ", " + ejb.getTarget_job_end_value());
	}
	
	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobMasterDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterPK)
	 */
	public void remove(JobStartJobMasterPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_job_start_job_mst WHERE job_id = ? AND target_job_id = ? AND target_job_type = ? AND target_job_end_value = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			stmt.setString(2, pk.getTarget_job_id());
			stmt.setInt(3, pk.getTarget_job_type());
			stmt.setInt(4, pk.getTarget_job_end_value());
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}
	
	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobMasterDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterBean)
	 */
	public JobStartJobMasterPK create(JobStartJobMasterBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : " + ejb.getJob_id() + ", " + ejb.getTarget_job_id() + ", " + ejb.getTarget_job_type() + ", " + ejb.getTarget_job_end_value());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		JobStartJobMasterPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_start_job_mst (");
			sql.append("job_id, ");
			sql.append("target_job_id, ");
			sql.append("target_job_type, ");
			sql.append("target_job_end_value) ");
			sql.append("VALUES (?,?,?,?)");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getJob_id());
			stmt.setString(2, ejb.getTarget_job_id());
			stmt.setInt(3, ejb.getTarget_job_type());
			stmt.setInt(4, ejb.getTarget_job_end_value());
			
			int row =	stmt.executeUpdate();
			pk = new JobStartJobMasterPK(
					ejb.getJob_id(),
					ejb.getTarget_job_id(),
					ejb.getTarget_job_type(), 
					ejb.getTarget_job_end_value());
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getJob_id() + ", "
						+ ejb.getTarget_job_id() + ", " + ejb.getTarget_job_type() + ", " + ejb.getTarget_job_end_value() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getJob_id() + ", "
						+ ejb.getTarget_job_id() + ", " + ejb.getTarget_job_type() + ", " + ejb.getTarget_job_end_value() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getJob_id() + ", "
						+ ejb.getTarget_job_id() + ", " + ejb.getTarget_job_type() + ", " + ejb.getTarget_job_end_value() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getJob_id() + ", "
				+ ejb.getTarget_job_id() + ", " + ejb.getTarget_job_type() + ", " + ejb.getTarget_job_end_value());
		return pk;
	}
	
	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobMasterDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<JobStartJobMasterPK> ret = new ArrayList<JobStartJobMasterPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT job_id, target_job_id, target_job_type, target_job_end_value FROM cc_job_start_job_mst";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobStartJobMasterPK pk= new JobStartJobMasterPK(
						res.getString("job_id"),
						res.getString("target_job_id"),
						res.getInt("target_job_type"), 
						res.getInt("target_job_end_value")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}
	
	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobMasterDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterPK)
	 */
	public JobStartJobMasterPK findByPrimaryKey(JobStartJobMasterPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT job_id, target_job_id, target_job_type, target_job_end_value FROM cc_job_start_job_mst WHERE job_id = ? AND target_job_id = ? AND target_job_type = ? AND target_job_end_value = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			stmt.setString(2, pk.getTarget_job_id());
			stmt.setInt(3, pk.getTarget_job_type());
			stmt.setInt(4, pk.getTarget_job_end_value());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
	
	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたジョブIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterBean#ejbFindByJobId(java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobMasterDAO#findByJobId(java.lang.String)
	 */
	public Collection findByJobId(String job_id) throws FinderException {
		m_log.debug("findByJobId() start : " + job_id);
		
		ArrayList<JobStartJobMasterPK> ret = new ArrayList<JobStartJobMasterPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT job_id, target_job_id, target_job_type, target_job_end_value FROM cc_job_start_job_mst WHERE job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, job_id);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobStartJobMasterPK pk= new JobStartJobMasterPK(
						res.getString("job_id"),
						res.getString("target_job_id"),
						res.getInt("target_job_type"), 
						res.getInt("target_job_end_value")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findByJobId() error : " + job_id + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByJobId() error : " + job_id + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByJobId() end : " + job_id);
		return ret;
	}
}
