/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.jobmanagement.composite;

import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.bean.JobParamTypeConstant;
import com.clustercontrol.bean.SizeConstant;
import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.jobmanagement.action.GetParameterTableDefine;
import com.clustercontrol.jobmanagement.bean.JobParameterInfo;
import com.clustercontrol.jobmanagement.bean.ParameterTableDefine;
import com.clustercontrol.jobmanagement.composite.action.ParameterSelectionChangedListener;
import com.clustercontrol.jobmanagement.dialog.ParameterDialog;
import com.clustercontrol.util.Messages;

/**
 * ジョブ変数タブ用のコンポジットクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class ParameterComposite extends Composite {
	/** テーブルビューア */
	protected CommonTableViewer m_viewer = null;
	/** 追加ボタン */
	protected Button m_createCondition = null;
	/** 変更ボタン */
	protected Button m_modifyCondition = null;
	/** 削除ボタン */
	protected Button m_deleteCondition = null;
	/** ジョブ変数情報のリスト */
	protected ArrayList m_paramList = null;
	/** シェル */
	protected Shell m_shell = null;
	/** 選択アイテム */
	protected ArrayList m_selectItem = null;
	/** 親ジョブID */
	protected String m_parentJobId = null;
	/** ジョブID */
	protected String m_jobId = null;
	
	/**
	 * コンストラクタ
	 * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
	public ParameterComposite(Composite parent, int style) {
		super(parent, style);
		initialize();
		m_shell = this.getShell();
	}
	
	/**
	 * コンポジットを構築します。
	 */
	private void initialize() {
		RowLayout layout = new RowLayout();
		layout.type = SWT.VERTICAL;
		layout.spacing = 1;
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.fill = true;
		this.setLayout(layout);
		
		Composite composite = null;
		RowLayout rowLayout = null;
		
		composite = new Composite(this, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.VERTICAL;
		composite.setLayout(rowLayout);
		
		Label tableTitle = new Label(composite, SWT.NONE);
		tableTitle.setText(Messages.getString("list"));
		
		Table table = new Table(composite, SWT.BORDER | SWT.H_SCROLL
				| SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.SINGLE);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		table.setLayoutData(new RowData(430, 80));
		
		composite = new Composite(composite, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 5;
		composite.setLayout(rowLayout);
		
		//パラメータの追加
		Label dummy1 = new Label(composite, SWT.NONE);
		dummy1.setLayoutData(new RowData(190, SizeConstant.SIZE_LABEL_HEIGHT));
		m_createCondition = new Button(composite, SWT.NONE);
		m_createCondition.setText(Messages.getString("add"));
		m_createCondition.setLayoutData(new RowData(80,
				SizeConstant.SIZE_BUTTON_HEIGHT));
		m_createCondition.addSelectionListener(new SelectionAdapter() {
			@SuppressWarnings("unchecked")
			public void widgetSelected(SelectionEvent e) {
				ParameterDialog dialog = new ParameterDialog(m_shell);
				if (dialog.open() == IDialogConstants.OK_ID) {
					ArrayList info = dialog.getInputData();
					ArrayList list = (ArrayList) m_viewer.getInput();
					if (list == null) {
						list = new ArrayList();
					}
					list.add(info);
					m_viewer.setInput(list);
				}
			}
		});
		
		//パラメータの変更
		m_modifyCondition = new Button(composite, SWT.NONE);
		m_modifyCondition.setText(Messages.getString("modify"));
		m_modifyCondition.setLayoutData(new RowData(80,
				SizeConstant.SIZE_BUTTON_HEIGHT));
		m_modifyCondition.addSelectionListener(new SelectionAdapter() {
			@SuppressWarnings("unchecked")
			public void widgetSelected(SelectionEvent e) {
				ParameterDialog dialog = new ParameterDialog(m_shell);
				if (m_selectItem instanceof ArrayList) {
					dialog.setInputData((ArrayList) m_selectItem);
					if (dialog.open() == IDialogConstants.OK_ID) {
						ArrayList info = dialog.getInputData();
						ArrayList list = (ArrayList) m_viewer.getInput();
						list.remove(m_selectItem);
						list.add(info);
						m_selectItem = null;
						m_viewer.setInput(list);
					}
				} else {
					
				}
			}
		});
		
		//パラメータの削除
		m_deleteCondition = new Button(composite, SWT.NONE);
		m_deleteCondition.setText(Messages.getString("delete"));
		m_deleteCondition.setLayoutData(new RowData(80,
				SizeConstant.SIZE_BUTTON_HEIGHT));
		m_deleteCondition.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if (m_selectItem instanceof ArrayList) {
					ArrayList list = (ArrayList) m_viewer.getInput();
					list.remove(m_selectItem);
					m_selectItem = null;
					m_viewer.setInput(list);
				} else {
					
				}
			}
		});

		m_viewer = new CommonTableViewer(table);
		m_viewer.createTableColumn(GetParameterTableDefine.get(),
				ParameterTableDefine.SORT_COLUMN_INDEX,
				ParameterTableDefine.SORT_ORDER);
		m_viewer.addSelectionChangedListener(
				new ParameterSelectionChangedListener(this));
	}
	
	/**
	 * ジョブ変数情報をコンポジットに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.bean.JobParameterInfo
	 */
	@SuppressWarnings("unchecked")
	public void reflectParamInfo() {
		if (m_paramList instanceof ArrayList) {
			//パラメータ設定
			ArrayList tableData = new ArrayList();
			for (int i = 0; i < m_paramList.size(); i++) {
				JobParameterInfo info = (JobParameterInfo) m_paramList.get(i);
				ArrayList tableLineData = new ArrayList();
				tableLineData.add(info.getParamId());
				tableLineData.add(new Integer(info.getType()));
				if (info.getType() == JobParamTypeConstant.TYPE_SYSTEM) {
					tableLineData.add("(" + Messages.getString("auto") + ")");
				}
				else if (info.getType() == JobParamTypeConstant.TYPE_USER) {
					tableLineData.add(info.getValue());
				}
				tableLineData.add(info.getDescription());
				tableData.add(tableLineData);
			}
			m_viewer.setInput(tableData);
		}
	}
	
	/**
	 * ジョブ変数情報を設定します。
	 * 
	 * @param paramList ジョブ変数情報のリスト
	 */
	public void setParamInfo(ArrayList paramList) {
		m_paramList = paramList;
	}
	
	/**
	 * ジョブ変数情報を返します。
	 * 
	 * @return ジョブ変数情報のリスト
	 */
	public ArrayList getParamInfo() {
		return m_paramList;
	}
	
	/**
	 * コンポジットの情報から、ジョブ変数情報を作成します。
	 * 
	 * @return 入力値の検証結果
	 * 
	 * @see com.clustercontrol.jobmanagement.bean.JobParameterInfo
	 */
	@SuppressWarnings("unchecked")
	public ValidateResult createParamInfo() {
		
		//パラメータ情報のインスタンスを作成・取得
		m_paramList = new ArrayList();
		
		//パラメータ取得
		ArrayList tableData = (ArrayList) m_viewer.getInput();
		HashMap map = new HashMap();
		if (tableData instanceof ArrayList) {
			for (int i = 0; i < tableData.size(); i++) {
				ArrayList tableLineData = (ArrayList) tableData.get(i);
				JobParameterInfo info = new JobParameterInfo();
				Integer type = 
					(Integer) tableLineData.get(ParameterTableDefine.TYPE);
				info.setType(type.intValue());
				if (info.getType() == JobParamTypeConstant.TYPE_SYSTEM) {
					info.setParamId((String)tableLineData.get(
							ParameterTableDefine.PARAM_ID));
					info.setDescription((String)tableLineData.get(
							ParameterTableDefine.DESCRIPTION));
				}
				else if (info.getType() == JobParamTypeConstant.TYPE_USER) {
					info.setParamId((String)tableLineData.get(
							ParameterTableDefine.PARAM_ID));
					info.setValue((String)tableLineData.get(
							ParameterTableDefine.VALUE));
					info.setDescription((String)tableLineData.get(
							ParameterTableDefine.DESCRIPTION));
				}
				
				//重複チェック
				Integer checkValue = (Integer) map.get(info.getParamId());
				if (checkValue == null) {
					m_paramList.add(info);
					map.put(info.getParamId(), new Integer(0));
				}
			}
		}

		return null;
	}
	
	/**
	 * 選択アイテムを返します。
	 * 
	 * @return 選択アイテム
	 */
	public ArrayList getSelectItem() {
		return m_selectItem;
	}
	
	/**
	 * 選択アイテムを設定します。
	 * 
	 * @param selectItem 選択アイテム
	 */
	public void setSelectItem(ArrayList selectItem) {
		m_selectItem = selectItem;
	}
	
	/**
	 * 親ジョブIDを返します。
	 * 
	 * @return 親ジョブID
	 */
	public String geParentJobId() {
		return m_parentJobId;
	}
	
	/**
	 * 親ジョブIDを設定します。
	 * 
	 * @param parentJobId 親ジョブID
	 */
	public void setParentJobId(String parentJobId) {
		m_parentJobId = parentJobId;
	}
	
	/**
	 * ジョブIDを返します。
	 * 
	 * @return ジョブID
	 */
	public String geJobId() {
		return m_jobId;
	}
	
	/**
	 * ジョブIDを設定します。
	 * 
	 * @param jobId ジョブID
	 */
	public void setJobId(String jobId) {
		m_jobId = jobId;
	}
}