/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.agent;

import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import java.util.Hashtable;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.util.RunHistoryUtil;
import com.clustercontrol.jobmanagement.bean.RunStatusConstant;
import com.clustercontrol.jobmanagement.message.RunInstructionInfo;
import com.clustercontrol.jobmanagement.message.RunResultInfo;

/**
 * コマンドを実行するスレッドクラス<BR>
 * 
 * ジョブ実行の際にプロセスを生成して、 終了まで、状態を監視するクラスです。
 * 
 * @version 2.1.0
 * @since 1.0.0
 */
public class CommandThread extends AgentThread {

	// ジョブ実行結果を読み取る際のエンコーディングをプロパティに設定する際のキー
	private final static String INPUT_ENCODING = "input.encoding";
	
	protected String m_resultMsg = "";
	protected String m_errMsg = "";
	protected int m_exitValue;
	
	// ジョブ実行結果を受け取る際のエンコーディング
	private String m_inputEncoding = null;

	// ロガー
	static private Log log = LogFactory.getLog(CommandThread.class);
	
	// ジョブを起こす（Runtime.execする）際にシリアル化するための専用ロックオブジェクト
	private static final Object lockNewProcess = new Object();

	/**
	 * デバッグ用メイン処理
	 * 
	 * @param args
	 */
	public static void main(String[] args) {

		CommandThread agent = new CommandThread();
		agent.execCommand(new String[] { "java -version" });
		agent.execCommand(new String[] { "cmd /c \"dir c;\\ /s \" " });
		agent.execCommand(new String[] { "notepad" });

	}

	/**
	 * デバッグ用コンストラクタ
	 */
	public CommandThread() {
		super();
	}

	/**
	 * コンストラクタ
	 * 
	 * @param info
	 *            実行指示
	 * @param sendQueue
	 *            実行応答用メッセージ送信クラス
	 * @param runHistory
	 *            実行履歴
	 */
	public CommandThread(RunInstructionInfo info, SendQueue sendQueue,
			Hashtable<String, Date> runHistory, Properties props) {
		super(info, sendQueue, runHistory, props);
		
		// ログファイルのエンコーディングを設定
		if(props != null){
			m_inputEncoding =  props.getProperty(INPUT_ENCODING);
		}
	}

	/**
	 * ジョブ（コマンド・スクリプト）を実行するクラス<BR>
	 * 
	 * ReceiveTopicで受け取ったジョブの指示が実行の場合に このメソッドが実行されます。
	 * 
	 */
	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Runnable#run()
	 */
	public void run() {
		log.debug("run start");

		Date startDate = new Date();

		// 実行履歴に追加
		RunHistoryUtil.addRunHistory(m_info, m_runHistory, startDate);

		// ---------------------------
		// -- 開始メッセージ送信
		// ---------------------------

		// メッセージ作成
		RunResultInfo info = new RunResultInfo();
		info.setSessionId(m_info.getSessionId());
		info.setJobId(m_info.getJobId());
		info.setFacilityId(m_info.getFacilityId());
		info.setCommand(m_info.getCommand());
		info.setCommandType(m_info.getCommandType());
		info.setStatus(RunStatusConstant.START);
		info.setTime(startDate);

		log.info("run SessionID=" + m_info.getSessionId() + ", JobID="
				+ m_info.getJobId());

		// 送信
		m_sendQueue.put(info);

		// ---------------------------
		// -- コマンド実行
		// ---------------------------

		// 起動ユーザ名取得
		String sysUserName = System.getProperty("user.name");
		String execCommand = m_info.getCommand();

		String cmd[] = null;
		if (!sysUserName.equals(m_info.getUser())) {
			if (sysUserName.equals("root")) {
				// 起動ユーザがrootの場合、su -c 'コマンド' ユーザ名 を実行する
				// コマンドパラメータは、一つずつ配列の要素となるようにする
				cmd = new String[4];
				cmd[0] = "su";
				cmd[1] = m_info.getUser();
				cmd[2] = "-c";
				cmd[3] = execCommand;

			} else {
				// 起動ユーザがroot以外の場合、エラーとする
				// 終了を送信
				info.setStatus(RunStatusConstant.ERROR);
				info.setTime(new Date());
				String errMsg = "The execution user of the command and agent's user are different. " +
				"execUser=[" + m_info.getUser() + "], agentUser=[" + sysUserName + "]";
				info.setErrorMessage(errMsg);
				info.setMessage("");
				m_sendQueue.put(info);

				log	.error(errMsg);
				log.info("run end");

				// 実行履歴削除メッセージ送信
				sendDeleteHistory(info);
				return;
			}
		} else {
			// 起動ユーザと実行ユーザが一緒の場合はそのまま実行
			cmd = execCommand.split(" ");

		}

		boolean ret = execCommand(cmd);

		// ---------------------------
		// -- 終了メッセージ送信
		// ---------------------------

		if (ret) {

			info.setStatus(RunStatusConstant.END);
			info.setEndValue(m_exitValue);

		} else {

			info.setStatus(RunStatusConstant.ERROR);

		}

		// 終了を送信
		info.setTime(new Date());
		info.setErrorMessage(m_errMsg);
		info.setMessage(m_resultMsg);
		m_sendQueue.put(info);

		// 実行履歴削除メッセージ送信
		sendDeleteHistory(info);

		log.debug("run end");
	}

	/**
	 * run()から呼び出されるコマンド実行部分のメソッド
	 * 
	 * @param cmd
	 *            コマンド
	 * @return 成功/失敗
	 */
	public boolean execCommand(String cmd[]) {
		Runtime runtime = Runtime.getRuntime();

		// コマンド起動
		Process process;

		try {
			if (log.isInfoEnabled()) {
				for (int i = 0; i < cmd.length; i++) {
					log.info("Command Execute:" + cmd[i]);
				}
			}
			// JVM（Windows環境）の不具合（？）でRuntime.execの並列実行多重度を上げた
			// 場合に問題が発生することがあるため、同時にただ１つのスレッドのみがexecを実行
			// できるように、staticなロックオブジェクトのsynchronizedによってシリアル化する
			synchronized(lockNewProcess) {
				process = runtime.exec(cmd);
			}
		} catch (IOException e) {
			log.error(e);
			m_errMsg = e.getMessage();
			return false;
		}
		try {

			// //標準出力、エラー出力読み取り開始
			StreamReader errStreamReader = new StreamReader(process
					.getErrorStream());
			errStreamReader.start();
			StreamReader inStreamReader = new StreamReader(process
					.getInputStream());
			inStreamReader.start();

			// コマンド実行待機
			try {
				m_exitValue = process.waitFor();
			} catch (InterruptedException e) {
				log.error(e);
				m_errMsg = e.getMessage();
				return false;
			}

			log.debug("ExitCode:" + m_exitValue);

			// 標準出力取得
			try {
				inStreamReader.join();
			} catch (InterruptedException e2) {
			}

			m_resultMsg = inStreamReader.getResult();
			log.debug(inStreamReader.getResult());

			// エラー出力取得
			try {
				errStreamReader.join();
			} catch (InterruptedException e2) {
			}

			m_errMsg = errStreamReader.getResult();
			log.debug(m_errMsg);

			return true;

		} finally {
			process.destroy();
		}

	}

	/**
	 * 実行プロセスの標準、エラー出力読み取りスレッドクラス<BR>
	 * コンストラクタで渡されたストリームを読み出し、文字列として格納します。
	 */
	protected class StreamReader extends Thread {

		InputStream m_ist;
		StringBuffer m_result;

		// ストリームから読み出した文字列
		String out_string = "";

		// これまですべてのreadで読み込んだバイト列
		byte[] m_readbuf;

		/**
		 * コンストラクタ
		 * 
		 * @param ist
		 *            入力ストリーム
		 */
		public StreamReader(InputStream ist) {
			super();
			m_ist = ist;
			m_result = new StringBuffer();
			m_readbuf = new byte[m_limit_orgmsg];
		}

		/**
		 * メインスレッド。
		 * 
		 * Agent.propertiesで指定されたサイズ分だけ標準入力、標準エラー入力から読み込み、 out_stringへ代入する。
		 * サイズを超えた場合は切捨て、その旨をログに出力する。
		 * 
		 * バッファサイズのデフォルト設定は以下（単位はByte）。 limit.size.orgmsg=8192
		 * 
		 */
		public void run() {

			// バッファを超えたことをログに出力したかどうかのフラグ
			boolean bufIsFull = false;

			// 一回のreadでストリームから読み込んだバイト数
			int readSize = 0;

			// 読み込んだバイト数の合計
			int total = 0;
			
			log.debug("--- read start ---");
			try {
				// 設定されたバッファサイズから、すでに読み込んだバイト数を引いた分だけ読見込む
				while ((readSize = m_ist.read(m_readbuf, total, m_limit_orgmsg - total)) != -1) {
					log.debug("total : " + total + ", readsize : " + readSize);

					if (bufIsFull) {
						// バッファがフルになっていて、ストリームが終わらない場合は、
						// 読んだものを無視してストリームが終わるまで読むことを繰り返す
						total = 0;
						
					} else {
						
						
						// バッファがフルになった場合、ログを出力して、メッセージ文字列を生成
						if (total == m_limit_orgmsg) {
							log.info("SessionID=" + m_info.getSessionId()
									+ ", JobID=" + m_info.getJobId()
									+ " : stdout / errout limit("
									+ m_limit_orgmsg + " Byte) over");
//							out_string = new String(m_readbuf);
							if(m_inputEncoding != null){
								out_string = new String(m_readbuf, 0, total, m_inputEncoding);
							} else {
								out_string = new String(m_readbuf, 0, total);
							}
							bufIsFull = true;
							
						// バッファがフルでなく、ストリームに残りがある場合は、再度読み込みへ
						} else {
							total = total + readSize;
						}
					}
				}
				
				// バッファがフルになっていなければ、バッファをStringに変換
				if (!bufIsFull) out_string = new String(m_readbuf, 0, total);

			} catch (IOException e) {
				log.error(e);
			}

			log.debug("--- read end ---");

			try {
				m_ist.close();
			} catch (IOException e) {
				log.info(e.getMessage());
			}

		}

		/**
		 * 読み出し結果取得
		 * 
		 * @return ストリームから取り出した文字列
		 */
		public String getResult() {
			return out_string;
		}
	}
}
