/* $Id: entryreal.cc 24 2003-12-20 01:38:09Z takekawa $
 * Copyright (C) 2003 Takashi Takekawa
 * This file is part of the Grs Library
 *
 * This library is free software; You may redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have recieved a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundtion, Inc., 59 Temple Place, Suite 330, Bostom, MA
 * 02111-1307 USA.
 */

#include "entryreal.h"
#include <gdk/gdkkeysyms.h>

#include <boost/bind.hpp>
using boost::bind;
using boost::ref;

namespace Grs {

EntryReal::EntryReal(Property<double>& property, int digits)
    : Gtk::DrawingArea(), m_value(property.get_value()),
      m_digits(digits), m_position(0), m_signal()
{
    property_can_focus() = true;
    add_events(Gdk::KEY_PRESS_MASK);
    add_events(Gdk::BUTTON_PRESS_MASK);

    m_layout = create_pango_layout("-undifined-");
    init_text();
    init_cursol();

    property.connect(bind(&EntryReal::set_value, ref(*this), _1));
    m_signal.connect(bind(&Property<double>::set_value, ref(property), _1));
}

EntryReal::~EntryReal()
{
}

void
EntryReal::set_digits(int digits)
{
    if (digits < 0) digits = 0;
    if (8 < digits) digits = 8;
    m_position += (m_digits - digits);
    m_digits = digits;
    init_text();
    init_cursol();
    queue_draw();
}

void
EntryReal::set_position(int pos)
{
    if (pos < -m_digits) pos = -m_digits;
    if (8-m_digits < pos) pos = 8-m_digits;
    m_position = pos;
    init_cursol();
    queue_draw();
}

void
EntryReal::set_value(double val)
{
    m_value = val;
    m_signal(m_value);
    init_text();
    queue_draw();
}

void
EntryReal::on_style_changed(const Glib::RefPtr<Gtk::Style>&)
{
    m_layout->set_font_description(get_style()->get_font());
    Pango::Rectangle rect = m_layout->get_pixel_logical_extents();
    set_size_request(rect.get_width(), rect.get_height());
}

bool
EntryReal::on_expose_event(GdkEventExpose*)
{
    Gtk::StateType state = get_state();
    Glib::RefPtr<Gdk::Drawable> win = get_window();
    Glib::RefPtr<Gdk::GC> gc = get_style()->get_base_gc(state);
    gc->set_foreground(get_style()->get_text(state));
    gc->set_background(get_style()->get_bg(state));
    win->draw_layout(gc, 0, 0, m_layout);
    return true;
}

int
EntryReal::get_position(double x, double y) const
{
    const int p = 10 - m_digits;
    const int xx = int(x*Pango::SCALE);
    const int yy = int(y*Pango::SCALE);
    int index, trailing;
    m_layout->xy_to_index(xx, yy, index, trailing);
    if (index <= 0)
        return -10;
    if ((index == p) || (10 < index))
        return -11;
    if (index < p)
        return p - index - 1;
    return p - index;
}

bool
EntryReal::on_scroll_event(GdkEventScroll* event)
{
    grab_focus();
    int pos = get_position(event->x, event->y);
    if (pos < -10) return false;
    if (pos == -10) {
        switch (event->direction)
        {
        case GDK_SCROLL_UP:
            set_digits(m_digits + 1);
            return true;
        case GDK_SCROLL_DOWN:
            set_digits(m_digits - 1);
            return true;
        default:
            break;
        }
    } else {
        set_position(pos);
        switch (event->direction)
        {
        case GDK_SCROLL_UP:
            set_value(m_value+std::pow(10.0, pos));
            return true;
        case GDK_SCROLL_DOWN:
            set_value(m_value-std::pow(10.0, pos));
            return true;
        case GDK_SCROLL_LEFT:
            set_digits(m_digits + 1);
            return true;
        case GDK_SCROLL_RIGHT:
            set_digits(m_digits - 1);
            return true;
        }
    }
    return false;
}

bool
EntryReal::on_button_press_event(GdkEventButton* event)
{
    grab_focus();
    if (event->type != GDK_BUTTON_PRESS) return false;
    int pos = get_position(event->x, event->y);
    if (pos < -10) return false;
    if (pos == -10) {
        switch (event->button)
        {
        case 1:
            set_digits(m_digits + 1);
            return true;
        case 3:
            set_digits(m_digits - 1);
            return true;
        }
    } else {
        set_position(pos);
        switch (event->button)
        {
        case 1:
            set_value(m_value+std::pow(10.0, pos));
            return true;
        case 3:
            set_value(m_value-std::pow(10.0, pos));
            return true;
        }
    }
    return false;
}

bool
EntryReal::on_key_press_event(GdkEventKey* event)
{
    int key = event->keyval;
    switch (key) {
    case GDK_Up:
        set_value(m_value+std::pow(10.0, m_position));
        return true;
    case GDK_Down:
        set_value(m_value-std::pow(10.0, m_position));
        return true;
    case GDK_Left:
        set_position(m_position + 1);
        return true;
    case GDK_Right:
        set_position(m_position - 1);
        return true;
    case GDK_Page_Up:
        set_digits(m_digits + 1);
        return true;
    case GDK_Page_Down:
        set_digits(m_digits - 1);
        return true;
    }
    if ('0' <= key && key <= '9') {
        double absval = std::abs(m_value);
        double newval = double(key) - double('0');
        double oldval = absval * std::pow(10.0, -m_position);
        oldval -= 10.0*std::floor(0.1*oldval);
        oldval = std::floor(oldval);
        set_value(copysign(absval+std::pow(10.0, m_position)*(newval-oldval), m_value));
        set_position(m_position - 1);
        return true;
    }
    return false;
}

bool
EntryReal::on_focus_in_event(GdkEventFocus* event)
{
    set_state(Gtk::STATE_ACTIVE);
    init_cursol();
    queue_draw();
    return true;
}

bool
EntryReal::on_focus_out_event(GdkEventFocus* event)
{
    set_state(Gtk::STATE_NORMAL);
    init_cursol();
    queue_draw();
    return true;
}

void
EntryReal::init_text()
{
    double dval = ::rint(m_value*std::pow(10.0, m_digits));
    if (1e9 <= dval) {
        m_layout->set_text("over flow  ");
        return;
    }
    if (dval <= -1e9) {
        m_layout->set_text("under flow ");
        return;
    }

    const int p = 10-m_digits;
    //             "01234567890"
    char str[12] = "           ";
    long value = long(dval);
    if (value < 0) {
        str[0] = '-';
        value = -value;
    } else {
        str[0] = '+';
    }

    for (int i = 10; i > p; --i) {
        str[i] = '0' + char(value%10);
        value /= 10;
    }

    str[p] = '.';

    for (int i = p-1; i > 0; --i) {
        str[i] = '0' + char(value%10);
        value /= 10;
        if (value == 0) break;
    }
    m_layout->set_text(str);
}

void
EntryReal::init_cursol()
{
    Pango::AttrList list;
    int index = (m_position < 0) ? (10 - m_digits - m_position)
                                 : (9 - m_digits - m_position); 
    Gtk::StateType state = (has_focus() ? Gtk::STATE_SELECTED : get_state());
    if (has_focus()) {
        Gdk::Color c = get_style()->get_text(state);
        Pango::AttrColor attr
            = Pango::Attribute::create_attr_foreground(
                    c.get_red(), c.get_green(), c.get_blue());
        attr.set_start_index(index);
        attr.set_end_index(index+1);
        list.insert(attr);
    }
    if (has_focus()) {
        Gdk::Color c = get_style()->get_bg(state);
        Pango::AttrColor attr
            = Pango::Attribute::create_attr_background(
                    c.get_red(), c.get_green(), c.get_blue());
        attr.set_start_index(index);
        attr.set_end_index(index+1);
        list.insert(attr);
    }
    m_layout->set_attributes(list);
}

}

