/**********************************************************************
 
	Copyright (C) 2006 Hirohisa MORI <joshua@globalbase.org>
 
	This program is free software; you can redistribute it 
	and/or modify it under the terms of the GLOBALBASE 
	Library General Public License (G-LGPL) as published by 

	http://www.globalbase.org/
 
	This program is distributed in the hope that it will be 
	useful, but WITHOUT ANY WARRANTY; without even the 
	implied warranty of MERCHANTABILITY or FITNESS FOR A 
	PARTICULAR PURPOSE.

**********************************************************************/

#include "memory_debug.h"
#include "resource.h"
#include "rs_cache.h"
#include "matrix.h"
#include "h_layering.h"
#include "xlerror.h"
#include "mx_blk_h_layering.h"
#include "gbview.h"
#include "win_flame.h"
#include "queue.h"
#include "pri_level.h"

#define	_LLONG_MAX	0x7fffffffffffffffLL	/* max signed long long */

/*
typedef struct hl_pattern_hdr {
	INTEGER64		next;
	INTEGER64		id;
	INTEGER64		top_dc[]; dim_code
	unsigned char		url[]; UTF-8
} HL_PATTERN_HDR;
*/

typedef struct hl_pattern_sft {
	FAVT64_NODE *		fn;
	INTEGER64		next;
	INTEGER64		id;
	INTEGER64 *		top_dc;
	int			dim;
	L_CHAR *		url;
	int			hdr_size;
} HL_PATTERN_SFT;

typedef struct hl_open_work {
	RESOURCE *		r;
	INTEGER64 *		param_i;
	double *		param_d;
	int			dirty_flag;
	MATRIX_NODE *		param_node;
	double			base_reso;
	REAL1 *			org;
	INTEGER64 *		pixel_size;
} HL_OPEN_WORK;


typedef struct hl_estimate {
	struct hl_estimate *	next;
	MATRIX *		m;
	double			distance;
	INTEGER64 *		dc;
} HL_ESTIMATE;

typedef struct hl_estimate_work {
	HL_ESTIMATE *		queue;
	AVT_NODE *		finish_dc;
} HL_ESTIMATE_WORK;

typedef struct hl_que_data {
	Q_HEADER			h;
	GBVIEW_FLAME *			gf;
	GBVIEW_LAYER_STATUS *		target;
	GBVIEW_LAYER_STATUS *		layer;
} HL_QUE_DATA;

void hl_close_matrix_file(MATRIX*);
void hl_new_token(MATRIX_TOKEN*,int);
void hl_free_token(MATRIX_TOKEN*,int);
void hl_trigger(int type,void* t);

INTEGER64 invalid_hide_time = 300;
SYS_QUEUE hl_que;


int
get_hl_size(HL_OPEN_WORK * w,int dim)
{
int i;
	if ( w->r->h.limit_resolution == 0 )
		return -1;
	w->org = d_alloc(sizeof(REAL1)*(dim));
	w->pixel_size = d_alloc(sizeof(INTEGER64)*dim);
	w->org[0] = w->r->h.minrect.tl.x;
	w->org[1] = w->r->h.minrect.tl.y;
	w->pixel_size[0] = 
		(w->r->h.minrect.br.x - w->r->h.minrect.tl.x)*
		w->r->h.visible_resolution;
	w->pixel_size[1] =
		(w->r->h.minrect.br.y - w->r->h.minrect.tl.y)*
		w->r->h.visible_resolution;
	w->base_reso = w->r->h.visible_resolution;
	for ( ; w->base_reso < w->r->h.limit_resolution ; ) {
		w->base_reso *= 2;
		for (i = 0 ; i < dim ; i ++ )
			w->pixel_size[i] *= 2;
	}
	return 0;
}


void
open_h_layering(MATRIX_TOKEN * t)
{
int err;
MATRIX * m;
MATRIX_PARAM p;
int er;
MATRIX_CHANNEL_INFO inf;
HL_OPEN_WORK * ow;
RS_BUF b;
int i;
INTEGER64 int64_zero = 0;
double double_zero = 0;

int err_code;

	err_code = 1;

	m = t->wait_matrix;
	ow = m->open_work;

	/* FILE Name on the cache */

	if ( ow->r->h.cache_file_id == 0 ) {
		memset(&b,0,sizeof(b));
		b.r = ow->r;
		set_rs_resource_cache(&b);
	}

	/* FILE Lock Operation */

	if ( ow->r->h.cache_file_id ) {

		m->filename = get_filepath_from_file_id(ow->r->h.cache_file_id,".mtx");
		err = load_matrix_header(m,O_RDWR,0644);


ss_printf("old_matrix err %i\n",err);
		if ( err == 0 ) {
			memset(&p,0,sizeof(p));
			p.write_file = matrix_standard_write_file;
			p.read_file = matrix_standard_read_file;
			p.read_net = 0;
			p.close_file = hl_close_matrix_file;
			p.new_token = hl_new_token;
			p.free_token = hl_free_token;
			p.trigger = hl_trigger;

	err_code = 2;
			er = set_matrix_param(m,&p);
			if ( er < 0 )
				goto error;
			goto exist_file;
		}
	}


	memset(&p,0,sizeof(p));
	p.channel_nos = HL_CH_MAX;


	p.modify_time = ow->r->h.modify;
	p.flags = 0;
	p.dim = 2;
	p.pri_area[MI_FETCH_1_TP] = PR_UP + 0x2000000;
	p.pri_area[MI_FETCH_1_MD] = PR_UP + 0x2000000;
	p.pri_area[MI_FETCH_1_BT] = PR_UP + 0x2000000;
	p.pri_area[MI_FETCH_2_TP] = PR_UP + 0x2000000;
	p.pri_area[MI_FETCH_2_MD] = PR_UP + 0x2000000;
	p.pri_area[MI_FETCH_2_BT] = PR_UP + 0x2000000;
	p.pri_area[MI_VISU_1_TP] = PR_UP + 0x2000000;
	p.pri_area[MI_VISU_1_MD] = PR_UP + 0x2000000;
	p.pri_area[MI_VISU_1_BT] = PR_UP + 0x2000000;
	p.pri_area[MI_SAVE_TP] = PR_STRT + 0x2000000;
	p.pri_area[MI_SAVE_MD] = PR_STRT + 0x2000000;
	p.pri_area[MI_SAVE_BT] = PR_STRT + 0x2000000;


	p.write_file = matrix_standard_write_file;
	p.read_file = matrix_standard_read_file;
	p.read_net = 0;
	p.close_file = hl_close_matrix_file;
	p.new_token = hl_new_token;
	p.free_token = hl_free_token;
	p.trigger = hl_trigger;

	er = set_matrix_param(m,&p);
	err_code = 3;
	if ( er < 0 )
		goto error;
	er = set_matrix_dim_divide(m,0,1);		// divide 2^1
	err_code = 4;
	if ( er < 0 )
		goto error;
	er = set_matrix_block_size(m,0,8);		// blocksize = 2^8
	err_code = 5;
	if ( er < 0 )
		goto error;
	er = set_matrix_dim_divide(m,1,1);
	err_code = 6;
	if ( er < 0 )
		goto error;
	er = set_matrix_block_size(m,1,8);
	err_code = 7;
	if ( er < 0 )
		goto error;
		
	for ( i = 0 ; i < m->p.dim ; i ++ ) {
		er = set_matrix_pixel_size(m,0,ow->pixel_size[i]);
		err_code = 8;
		if ( er < 0 )
			goto error;
	}

	
	inf.data_type = &mx_type_int64_v;
	inf.flags = MF_FILE|MF_VISU;
	inf.default_data = &int64_zero;
	er = set_matrix_channel_info(m,HL_CH_PATTERN_INT,&inf);
	err_code = 10;
	if ( er < 0 )
		goto error;

	inf.data_type = &mx_type_double_v;
	inf.flags = MF_FILE|MF_VISU;
	inf.default_data = &double_zero;
	er = set_matrix_channel_info(m,HL_CH_PATTERN_DBL,&inf);
	err_code = 10;
	if ( er < 0 )
		goto error;


	inf.data_type = &mx_type_block;
	inf.flags = MF_FILE|MF_VISU;
	inf.default_data = 0;
	er = set_matrix_channel_info(m,HL_CH_DATA,&inf);
	err_code = 11;
	if ( er < 0 )
		goto error;

exist_file:

/*
	set_matrix_cal(m,MI_VISU_1_BT,
		sexp_commands(
			std_cm,
	"<mxFinish",
	"	normal-jump-status=\"MS_OK\"",
	"	err-jump-status=\"MS_PROCESS_ERR_1\"/>",
	"(mxSet 1 ([mxUncompressOldFormat format=\"R64\"] (mxCH 2)))",
		0));
	set_matrix_cal_equ(m,MI_VISU_1_MD,MI_VISU_1_BT);
	set_matrix_cal(m,MI_VISU_1_TP,
		sexp_commands(
			std_cm,
		"(mxSet 1 ([mxUncompressOldFormat format=\"R64\"] (mxCH 2)))",
		0));
*/

	set_matrix_cal(m,MI_SAVE_TP,
		sexp_commands(
			std_cm,
			"<mxSaveSB/>",
			"<mxSave/>",
		0));
	set_matrix_cal_equ(m,MI_SAVE_MD,MI_SAVE_TP);
	set_matrix_cal_equ(m,MI_SAVE_BT,MI_SAVE_TP);
	
	set_matrix_cal(m,MI_FETCH_1_TP,
		sexp_commands(
			std_cm,
			"<If>",
			"    (= <IfDef>^create-node enable</IfDef> \"enable\")",
			"    <Then>",
			"     <mxFinish normal-jump-status=\"MS_OK\" err-jump-status=\"MS_OK\"/>",
			"    </Then>",
			"    <Else>",
			"     <mxFinish normal-jump-status=\"MS_OK\" err-jump-status=\"MS_PROCESS_ERR_1\"/>",
			"    </Else>",
			"   </If>",
			"<mxLoad target=\"file\"/>",
			"<mxSetSB channel=\"", sxc_int(HL_CH_DATA), "\">",
			"	<sbHeuristicLayering/>"
			"</mxSetSB>",
			"<mxLoadSB/>",
		0));
	set_matrix_cal_equ(m,MI_FETCH_1_MD,MI_FETCH_1_TP);
	set_matrix_cal_equ(m,MI_FETCH_1_BT,MI_FETCH_1_TP);
	
	if ( err == ME_DESTROY_FILE || err == ME_CANNOT_OPEN_FILE )
		save_matrix_header(m);


	set_matrix_mode(m,MM_STANBY);
	;
	d_f_ree(t);
	return;
error:
	er_panic("ERROR");
}


void hl_close_matrix_file(MATRIX* m)
{
HL_OPEN_WORK * ow;
	ow = (HL_OPEN_WORK*)m->open_work;
	if ( ow->org )
		d_f_ree(ow->org);
	if ( ow->pixel_size )
		d_f_ree(ow->pixel_size);
	d_f_ree(ow);
}


void hl_new_token(MATRIX_TOKEN* t,int u)
{
}

void hl_free_token(MATRIX_TOKEN* t,int u)
{
}

void hl_trigger(int type,void* t)
{
}

HL_PATTERN *
sorting_pattern(HL_PATTERN * p)
{
HL_PATTERN * a,*b,*t;
HL_PATTERN ** pp;
int ret;
	if ( p == 0 )
		return 0;
	if ( p->next == 0 )
		return p;
	a = b = 0;
	for ( ; p ; p = p->next ) {
		t = p;
		p = t->next;
		t->next = a;
		a = t;
		
		if ( p == 0 )
			break;
		
		t = p;
		p = t->next;
		
		t->next = b;
		b = t;
	}
	a = sorting_pattern(a);
	b = sorting_pattern(b);
	
	p = 0;
	pp = &p;
	for ( ; a && b ; ) {
		ret = l_strcmp(a->url,b->url);
		if ( ret == 0 ) {
			t = a;
			a = t->next;
			t->next = 0;
			*pp = t;
			pp = &t->next;
			
			t = b;
			b = t->next;
			t->next = 0;
			*pp = t;
			pp = &t->next;
		}
		else if ( ret < 0 ) {
			t = a;
			a = t->next;
			t->next = 0;
			*pp = t;
			pp = &t->next;
		}
		else {
			t = b;
			b = t->next;
			t->next = 0;
			*pp = t;
			pp = &t->next;
		}
	}
	if ( a )
		*pp = a;
	else	*pp = b;
	return p;
}


void
hl_pattern_endian()
{
}

int
hl_pattern_to_sft(HL_PATTERN_SFT * sft,unsigned char * hdr)
{
unsigned char * p1;
int dim;
	p1 = get_uncompressed_code64(&sft->next,hdr);
	p1 = get_uncompressed_code64(&sft->id,p1);
	dim = get_dim_code_dimension(p1);
	if ( dim <= 0 )
		return -1;
	sft->dim = dim;
	sft->top_dc = d_alloc(sizeof(INTEGER64)*(dim+1));
	p1 = get_uncompressed_code64(sft->top_dc,p1);
	sft->url = nl_copy_str(&utf8_cm,(char*)p1);
	return 0;
}

unsigned char *
hl_pattern_to_hdr(HL_PATTERN_SFT * sft)
{
RECORD_LIST64 * rl;
unsigned char * buffer,*ptr,*ret;
INTEGER64 dummy_dc[1];

	rl = new_recordlist64(0,0);
	
	buffer = d_alloc(sizeof(INTEGER64)*2+1);
	ptr = get_compressed_code64(buffer,sft->next);
	set_recordlist_chain64(rl,buffer,ptr-buffer,1);

	buffer = d_alloc(sizeof(INTEGER64)*2+1);
	ptr = get_compressed_code64(buffer,sft->id);
	set_recordlist_chain64(rl,buffer,ptr-buffer,1);

	if ( sft->top_dc ) {
		buffer = d_alloc(sizeof(INTEGER64)*2*(sft->dim+1)+2);
		ptr = get_compressed_dim_code(buffer,sft->top_dc,sft->dim);
		set_recordlist_chain64(rl,buffer,ptr-buffer,1);
	}
	else {
		buffer = d_alloc(sizeof(INTEGER64)*2*(sft->dim+1)+2);
		dummy_dc[0] = 0;
		ptr = get_compressed_dim_code(buffer,dummy_dc,0);
		set_recordlist_chain64(rl,buffer,ptr-buffer,1);
	}
	
	ptr = (unsigned char*)ln_copy_str(&utf8_cm,sft->url);
	set_recordlist_chain64(rl,(char*)ptr,strlen((char*)ptr)+1,1);

	sft->hdr_size = setup_recordlist64(rl);
	
	ret = rl->data;
	rl->data = 0;
	free_recordlist64(rl);
	return ret;
}

int
hl_pattern_cmp(unsigned char * d1,unsigned char * d2,void * p)
{
HL_PATTERN_SFT sft1,sft2;
int ret;
	hl_pattern_to_sft(&sft1,d1);
	hl_pattern_to_sft(&sft2,d2);
	ret = l_strcmp(sft1.url,sft2.url);
	d_f_ree(sft1.url);
	d_f_ree(sft1.top_dc);
	d_f_ree(sft2.url);
	d_f_ree(sft2.top_dc);
	
	return ret;
}

HL_PATTERN_LIST *
_convert_pattern_to_loop(MATRIX * m,INTEGER64 fofs,HL_PATTERN * p,HL_PATTERN * target)
{
HL_PATTERN_LIST * ret, * r, ** rp;

	if ( fofs == 0 )
		return 0;
	ret = 0;
	rp = &ret;
	for ( ; p ; p = p->next ) {
		r = _convert_pattern_to_id_2(m,fofs,p,target);
		if ( r == 0 )
			continue;
		*rp = r;
		for ( ; *rp ; rp = &(*rp)->next );
	}
	return ret;
}

void
free_pattern_sft(HL_PATTERN_SFT * sft)
{
	if ( sft->top_dc )
		d_f_ree(sft->top_dc);
	if ( sft->url )
		d_f_ree(sft->url);
}

HL_PATTERN *
hl_push_pattern(L_CHAR * url,HL_PATTERN * target)
{
HL_PATTERN * p;
	p = d_alloc(sizeof(*p));
	p->url = ll_copy_str(url);
	p->next = target;
	return p;
}

void
hl_pop_pattern(HL_PATTERN * target)
{
	d_f_ree(target->url);
	d_f_ree(target);
}

HL_PATTERN *
hl_copy_pattern(HL_PATTERN * p)
{
HL_PATTERN * ret,* p1;
	ret =0;
	for ( ; p ; p = p->next ) {
		p1 = d_alloc(sizeof(*p1));
		p1->url = ll_copy_str(p->url);
		p1->next = ret;
		ret = p1;
	}
	return ret;
}

void
hl_free_pattern(HL_PATTERN * p)
{
HL_PATTERN * pp;
	for ( ; p ; ) {
		pp = p->next;
		hl_pop_pattern(p);
		p = pp;
	}
}


int
_convert_pattern_to_id_hit(HL_PATTERN_SFT * sft_ret,MATRIX * m,INTEGER64 fofs,HL_PATTERN * p,int insert_flag)
{
FAVT64_ROOT * rt;
HL_OPEN_WORK * ow;
FAVT64_NODE * fn;
int err;
unsigned char * hdr;
HL_PATTERN_SFT sft;
int ret;
PDB64 * pfd;

	if ( fofs == 0 || p == 0 )
		return -1;
	pfd = get_mx_file_pfd(m);
	ow = m->open_work;

	rt = get_root64(&err,
		pfd,
		fofs,
		hl_pattern_endian);
	memset(&sft,0,sizeof(sft));
	sft.url = p->url;
	hdr = hl_pattern_to_hdr(&sft);
	fn = favt64_search(&err,
			rt,
			root_node64(&err,rt),
			hdr,hl_pattern_cmp,0);
	if ( fn == 0 ) {
		if ( insert_flag == 0 ) {
			d_f_ree(hdr);
			return -1;
		}
		fn = favt64_alloc_node(rt,hdr,sft.hdr_size);
		favt64_insert(&err,rt,&rt->node,fn,hl_pattern_cmp,0);
	}
	hdr = fn->data;
	hl_pattern_to_sft(&sft,hdr);
	sft.fn = fn;
	
	if ( p->next ) {
		ret = _convert_pattern_to_id_hit(sft_ret,m,sft.next,p->next,insert_flag);
		free_pattern_sft(&sft);
	}
	else {
		if ( insert_flag ) {
			fn = favt64_delete(&err,
					rt,
					&rt->node,
					hdr,
					hl_pattern_cmp,0);
			favt64_free_node(fn);
			d_f_ree(hdr);
			hdr = hl_pattern_to_hdr(sft_ret);
			fn = favt64_alloc_node(rt,hdr,sft_ret->hdr_size);
			favt64_insert(&err,rt,&rt->node,fn,hl_pattern_cmp,0);
			ret = 0;
		}
		else {
			ret = 0;
			*sft_ret = sft;
		}
	}
	d_f_ree(hdr);
	
	return ret;
}


int
convert_pattern_to_id_hit(HL_PATTERN_SFT * sft_ret,MATRIX * m,HL_PATTERN * p,int insert_flag)
{
HL_OPEN_WORK * ow;
int ret;
int er;
	if ( (er=start_file_access(m)) < 0 )
		return 0;
	ow = m->open_work;
	if ( p == 0 ) {
		if ( insert_flag ) {
			ow->param_i[PTN_TOP_ID] = sft_ret->id;
			memcpy(&ow->param_i[PTN_TOP_DC],sft_ret->top_dc,sizeof(INTEGER64)*(m->p.dim+1));
			ow->dirty_flag = NF_DIRTY;
		}
		else {
			if ( ow->param_i[PTN_TOP_ID] == 0 ) {
				ret = -1;
			}
			else {
				ret = 0;
				sft_ret->id = ow->param_i[PTN_TOP_ID];
				sft_ret->top_dc = copy_dim_code(m,&ow->param_i[PTN_TOP_DC]);
				sft_ret->next = ow->param_i[PTN_PATTERN];
				sft_ret->dim = m->p.dim;
				sft_ret->url = 0;
			}
		}
	}
	else
		ret = _convert_pattern_to_id_hit(sft_ret,m,ow->param_i[PTN_PATTERN],p,insert_flag);

	finish_file_access(m,0);

	return ret;
}


HL_PATTERN_LIST *
_convert_pattern_to_id_2(MATRIX * m,INTEGER64 fofs,HL_PATTERN * p,HL_PATTERN * target)
{
FAVT64_ROOT * rt;
HL_OPEN_WORK * ow;
FAVT64_NODE * fn;
int err;
unsigned char * hdr;
HL_PATTERN_SFT sft;
HL_PATTERN_LIST * ret;
PDB64 * pfd;

	if ( fofs == 0 || p == 0 )
		return 0;
	pfd = get_mx_file_pfd(m);
	ow = m->open_work;

	rt = get_root64(&err,
		pfd,
		fofs,
		hl_pattern_endian);
	memset(&sft,0,sizeof(sft));
	sft.url = p->url;
	hdr = hl_pattern_to_hdr(&sft);
	fn = favt64_search(&err,
			rt,
			root_node64(&err,rt),
			hdr,hl_pattern_cmp,0);
	d_f_ree(hdr);
	if ( fn == 0 )
		return 0;
	hdr = fn->data;
	hl_pattern_to_sft(&sft,hdr);
	target = hl_push_pattern(p->url,target);
	
	if ( sft.id ) {
		ret = d_alloc(sizeof(*ret));
		ret->p = hl_copy_pattern(target);
		ret->id = sft.id;
		ret->top_dc = sft.top_dc;
		sft.top_dc = 0;
		ret->next = _convert_pattern_to_loop(m,sft.next,p->next,target);
	}
	else	ret = _convert_pattern_to_loop(m,sft.next,p->next,target);
	
	hl_pop_pattern(target);
	free_pattern_sft(&sft);
	
	return ret;
}

HL_PATTERN_LIST *
convert_pattern_to_id(MATRIX * m,HL_PATTERN * p)
{
HL_OPEN_WORK * ow;
HL_PATTERN_LIST * ret;
int er;
	if ( (er=start_file_access(m)) < 0 )
		return 0;
	ow = m->open_work;
	if ( ow->param_i[PTN_TOP_ID] ) {
		ret = d_alloc(sizeof(*ret));
		ret->p = 0;
		ret->id = ow->param_i[PTN_TOP_ID];
		ret->top_dc = copy_dim_code(m,&ow->param_i[PTN_TOP_DC]);
		ret->next = _convert_pattern_to_id_2(m,ow->param_i[PTN_PATTERN],p,0);
	}
	else	ret = _convert_pattern_to_id_2(m,ow->param_i[PTN_PATTERN],p,0);

	finish_file_access(m,0);

	return ret;
}



INTEGER64 *
convert_dc(double * reso_p,MATRIX * m,REAL1 * point)
{
HL_OPEN_WORK * ow;
INTEGER64 * dc;
REAL1 sq,rs;
int i;
	ow = m->open_work;
	dc = d_alloc(sizeof(INTEGER64)*m->p.dim);
	sq = sqrt(2);
	if ( point[0] > ow->base_reso )
		dc[0] = 0;
	else {
		dc[0] = 0;
		rs = ow->base_reso;
		for ( ; ; dc[0] ++ , rs = rs/2 ) {
			if ( point[0] > rs/sq )
				break;
		}
	}
	for ( i = 0 ; i < m->p.dim ; i ++ )
		dc[i+1] = (point[i] - ow->org[i])*ow->base_reso;
	*reso_p = ow->base_reso/point[0] * m->block_size[0];
	return dc;
}

MATRIX *
open_hl_matrix(RESOURCE * r)
{
HL_OPEN_WORK * w;
MATRIX_NODE * n;
MATRIX * m;
INTEGER64 * dc;
void * data;
MATRIX_ALLOC_VECTOR_PARAM vp;
int i;
int err;
int ix;
INTEGER64 int64_zero = 0;
double double_zero = 0;
MATRIX_DH_SET dh;
PDB64 * pfd;
int er;
FAVT64_ROOT * rt;
	m = 0;
	w = d_alloc(sizeof(*w));
	memset(w,0,sizeof(*w));
	w->r = r;
	if ( get_hl_size(w,2) < 0 )
		goto err1;
	r->h.mtx = m = open_matrix(get_url_str2(&r->h.target),0,0,
					open_h_layering,w);
	dc = d_alloc(sizeof(INTEGER64)*(m->p.dim+1));
	dc[0] = 0;
	dc[1] = (HL_DC_PARAM<<m->block_size[0]);
	for ( i = 2 ; i < m->p.dim+1 ; i ++ )
		dc[i] = 0;
	ix = PTN_TOP_DC + m->p.dim + 1;
	vp.dim = 1;
	vp.ix_size = &ix;
	vp.default_data = &int64_zero;
	data = get_matrix_node_channel(&err,&n,m,dc,HL_CH_PATTERN_INT,GN_NODE,GN_NODE_CREATE,&vp);
	if ( n == 0 )
		er_panic("open_hl_matrix(1)");
	w->param_node = n;
	get_matrix_dh_set(&dh,data);
	w->param_i = (INTEGER64*)dh.offset;
	if ( w->param_i[PTN_ID_MAX] <= 0 ) {
		w->param_i[PTN_ID_MAX] = 2;
		w->param_i[PTN_TOP_ID] = 1;
		pfd = get_mx_file_pfd(m);


		if ( (er=start_file_access(m)) < 0 )
			er_panic("open_hl_amtrix(2)");
		rt = favt64_alloc_root(pfd,FAT_MX_NODE,hl_pattern_endian);
		w->param_i[PTN_PATTERN] = rt->h.fofs;

		finish_file_access(m,0);

		w->dirty_flag = NF_DIRTY;
	}
/*
	matrix_node_channel_unlock(n,flags);
	unlock_node(n,0);
*/
	vp.dim = PTN_ORG + m->p.dim;
	vp.ix_size = &ix;
	vp.default_data = &double_zero;
	data = get_matrix_node_channel(&err,&n,m,dc,HL_CH_PATTERN_DBL,GN_NODE,GN_NODE_CREATE,&vp);
	get_matrix_dh_set(&dh,data);
	w->param_d = (double*)dh.offset;
	if ( w->param_d[PTN_BASE_RESO] == 0 ) {
		w->param_d[PTN_BASE_RESO] = w->base_reso;
		for ( i = 0 ; i < m->p.dim ; i ++ )
			w->param_d[PTN_ORG+i] = w->org[i];
		w->dirty_flag = NF_DIRTY;
	}

err1:
	d_f_ree(w);
	return m;
}


void
close_hl_matrix(MATRIX * m)
{
HL_OPEN_WORK * w;
	w = (HL_OPEN_WORK*)m->open_work;
	matrix_node_channel_unlock(w->param_node,w->dirty_flag);
	matrix_node_channel_unlock(w->param_node,w->dirty_flag);
	unlock_node(w->param_node,0);
	unlock_node(w->param_node,0);
	close_matrix(m);
}

HL_PATTERN_LIST * 
hl_sorting_by_length_2(HL_PATTERN_LIST * lst)
{
HL_PATTERN_LIST * l1,*l2,*l3,**pp;
	if ( lst == 0 )
		return 0;
	if ( lst->next == 0 )
		return 0;
	l1 = l2 = 0;
	for ( ; lst ; ) {
		l3 = lst;
		lst = l3->next;
		l3->next = l1;
		l1 = l3;
		
		if ( lst == 0 )
			break;
		
		l3 = lst;
		lst = l3->next;
		l3->next = l2;
		l2 = l3;
	}
	l1 = hl_sorting_by_length_2(l1);
	l2 = hl_sorting_by_length_2(l2);
	lst = 0;
	pp = &lst;
	for ( ; l1 && l2 ; ) {
		if ( l1->len < l2->len ) {
			l3 = l2;
			l2 = l3->next;
			*pp = l3;
			l3->next = 0;
			pp = &l3->next;
		}
		else {
			l3 = l1;
			l1 = l3->next;
			*pp = l3;
			l3->next = 0;
			pp = &l3->next;
		}
	}
	if ( l1 )
		*pp = l1;
	else	*pp = l2;
	return lst;
}


HL_PATTERN_LIST * 
hl_sorting_by_length(HL_PATTERN_LIST * lst)
{
int len;
HL_PATTERN_LIST * l_1;
HL_PATTERN * p;
	for ( l_1 = lst ; l_1 ; l_1 = l_1->next ) {
		for ( p = l_1->p , len = 0 ;
			p;
			len ++ , p = p->next );
		l_1->len = len;
	}
	return hl_sorting_by_length_2(lst);
}

void
hl_free_pattern_list(HL_PATTERN_LIST * lst)
{
HL_PATTERN_LIST * l;
	for ( ; lst ; ) {
		l = lst->next;
		d_f_ree(lst->top_dc);
		hl_free_pattern(lst->p);
		lst = l;
	}
}

HL_ESTIMATE *
hl_new_estimate(MATRIX * m,INTEGER64 * new_dc,INTEGER64 * dc,REAL1 reso)
{
HL_ESTIMATE *ret;
INTEGER64 * ndc;
int i;
INTEGER64 s,mask;
HL_OPEN_WORK * ow;
double ndc_reso,d,d1;
	ret = d_alloc(sizeof(*ret));
	memset(ret,0,sizeof(*ret));
	ret->dc = copy_dim_code(m,new_dc);
	ndc = copy_dim_code(m,new_dc);
	for ( i = 0 ; i < m->p.dim ; i ++ ) {
		s = ((INTEGER64)1)<<(ndc[0]*m->dim_divide[i] + m->block_size[i]);
		mask = -s;
		ndc[i+1] &= mask;
		ret->dc[i+1] &= mask;
		ndc[i+1] += s/2;
	}
	ow = m->open_work;
	ndc_reso = ow->base_reso;
	for ( i = 0 ; i < ndc[0] ; i ++ )
		ndc_reso /= 2;
	d = (ndc_reso - reso);
	d = d*d;
	for ( i = 0 ; i < m->p.dim ; i ++ ) {
		d1 = dc[i+1] - ndc[i+1];
		d += d1*d1;
	}
	ret->distance = sqrt(d);
	ret->m = m;
	return ret;
}

int
hl_cmp_estimate(HL_ESTIMATE * e1,HL_ESTIMATE * e2)
{
int i;
MATRIX * m;
	m = e1->m;
	for ( i = 0 ; i <= m->p.dim ; i ++ ) {
		if ( e1->dc[i] < e2->dc[i] )
			return -1;
		if ( e1->dc[i] > e2->dc[i] )
			return 1;
	}
	return 0;
}


HL_ESTIMATE * 
hl_search_finish_estimate(HL_ESTIMATE_WORK * w,INTEGER64 * dc)
{
HL_ESTIMATE e;
AVT_NODE * target;
	e.dc = dc;
	target = avt_search(w->finish_dc,&e,hl_cmp_estimate);
	if ( target == 0 )
		return 0;
	return target->data;
}

void
hl_insert_finish_estimate(HL_ESTIMATE_WORK * w,HL_ESTIMATE * e)
{
AVT_NODE * target,*a;
	target = d_alloc(sizeof(*target));
	target->data = e;
	a = avt_insert(&w->finish_dc,target,hl_cmp_estimate);
	if ( a != target )
		er_panic("insert_finish_estimate");
}

void
hl_free_estimate(HL_ESTIMATE * e)
{
	d_f_ree(e->dc);
	d_f_ree(e);
}

void
hl_free_finish_estimate(HL_ESTIMATE_WORK * w)
{
AVT_NODE * a;
	for ( ; w->finish_dc ; ) {
		a = avt_delete(&w->finish_dc,w->finish_dc->data,hl_cmp_estimate);
		hl_free_estimate(a->data);
		d_f_ree(a);
	}
}

void
hl_insert_estimate_queue(HL_ESTIMATE_WORK * w,HL_PATTERN_LIST * lst,HL_ESTIMATE * e)
{
HL_ESTIMATE ** pp, * p;
INTEGER64 mask;
int i;
	if ( lst->top_dc[0] < e->dc[0] ) {
		hl_free_estimate(e);
		return;
	}
	for ( i = 0 ; i < e->m->p.dim ; i ++ ) {
		mask = -(((INTEGER64)1)<<(lst->top_dc[0]*e->m->dim_divide[i] + e->m->block_size[i]));
		if ( (lst->top_dc[i+1]&mask) != (e->dc[i+1]&mask) ) {
			hl_free_estimate(e);
			return;
		}
	}
	if ( hl_search_finish_estimate(w,e->dc) ) {
		hl_free_estimate(e);
		return;
	}
	pp = &w->queue;
	for ( ; *pp ; ) {
		p = *pp;
		if ( p->distance > e->distance )
			break;
	}
	e->next = *pp;
	*pp = e;
}

HL_ESTIMATE *
hl_delete_estimate_queue(HL_ESTIMATE_WORK * w)
{
HL_ESTIMATE * ret;
	ret = w->queue;
	if ( ret == 0 )
		return 0;
	w->queue = ret->next;
	return ret;
}
int
hl_estimate_list_2(MATRIX * m,HL_PATTERN_LIST * lst,HL_ESTIMATE * e,INTEGER64 * dc,REAL1 reso)
{
MX_STRUCT_BLOCK * b;
MATRIX_NODE * n;
int er;
HL_OP op;
	n = get_matrix_node_wait(&er,m,e->dc,GN_NODE_CREATE);
	if ( n == 0 )
		return 0;
	b = get_channel_struct_block(n,HL_CH_DATA);
	if ( b == 0 )
		goto finish;
	memset(&op,0,sizeof(op));
	op.calc.id = lst->id;
	op.calc.ix = &dc[1];
	op.calc.reso = reso;
	op.calc.show = op.calc.hide = 0;
	(*b->tbl->operation)(SBOP_CALC,b,&op);
	if ( op.calc.show >= 0 ) {
		lst->show_distance += op.calc.show;
	}
	if ( op.calc.hide >= 0 ) {
		lst->hide_distance += op.calc.hide;
	}
	matrix_node_channel_unlock(n,0);
finish:
	unlock_node(n,0);
	return 1;
}

void
hl_estimate_list(MATRIX * m,HL_PATTERN_LIST * lst,INTEGER64 * dc,REAL1 reso)
{
HL_ESTIMATE * e;
HL_ESTIMATE_WORK w;
INTEGER64 * ndc;
INTEGER64 s,mask,backup;
INTEGER64 * start, * end, * step, * ix;
int i;
int exist;
	memset(&w,0,sizeof(w));
	e = hl_new_estimate(m,dc,dc,reso);
	hl_insert_estimate_queue(&w,lst,e);
	if ( w.queue == 0 ) {
		e = hl_new_estimate(m,lst->top_dc,dc,reso);
		hl_insert_estimate_queue(&w,lst,e);
	}
	lst->show_distance = lst->hide_distance = 0;
	for ( ; ; ) {
		e = hl_delete_estimate_queue(&w);
		if ( e == 0 )
			break;
		exist = hl_estimate_list_2(m,lst,e,dc,reso);

		hl_insert_finish_estimate(&w,e);
		
		ndc = copy_dim_code(m,e->dc);
		if ( exist == 0 )
			goto next3;
		for ( i = 0 ; i < m->p.dim ; i ++ ) {
			backup = ndc[i+1];
			s = ((INTEGER64)1)<<(ndc[0]*m->dim_divide[i] + m->block_size[i]);
			mask = -s;
			ndc[i+1] &= mask;
			ndc[i+1] -= s;
			if ( ndc[i+1] < 0 )
				goto next1;
			hl_insert_estimate_queue(&w,lst,hl_new_estimate(m,ndc,dc,reso));
		next1:
			ndc[i+1] = backup;
			ndc[i+1] &= mask;
			ndc[i+1] += s;
			if ( ndc[i+1] > m->pixel_size[i] )
				goto next2;
			hl_insert_estimate_queue(&w,lst,hl_new_estimate(m,ndc,dc,reso));
		next2:
			ndc[i+1] = backup;
		}
		if ( ndc[0] ) {
			start = copy_dim_code(m,ndc);
			end = copy_dim_code(m,ndc);
			step = copy_dim_code(m,ndc);
			for ( i = 0 ; i < m->p.dim ; i ++ ) {
				s = ((INTEGER64)1)<<(ndc[0]*m->dim_divide[i] + m->block_size[i]);
				mask = -s;
				start[i+1] &= mask;
				end[i+1] &= mask;
				end[i+1] += s;
				step[i+1] = s>>m->dim_divide[i];
			}
			start[0] --;
			end[0] --;
			ix = copy_dim_code(m,start);
			for ( ; ; ) {
				hl_insert_estimate_queue(&w,lst,hl_new_estimate(m,ix,dc,reso));
				if ( inc_dim_code_ix(ix,start,step,end,m->p.dim) )
					break;
			}
			d_f_ree(start);
			d_f_ree(end);
			d_f_ree(step);
			d_f_ree(ix);
		}
	next3:
		if ( ndc[0] < m->total_levels-1 ) {
			ndc[0] ++;
			hl_insert_estimate_queue(&w,lst,hl_new_estimate(m,ndc,dc,reso));
		}
		d_f_ree(ndc);
		
		if ( lst->show_distance < 0 )
			continue;
		if ( lst->hide_distance < 0 )
			continue;
		if ( w.queue == 0 )
			break;
		if ( w.queue->distance < 2 * lst->show_distance )
			continue;
		if ( w.queue->distance < 2 * lst->hide_distance )
			continue;
		break;
	}
	hl_free_finish_estimate(&w);
	for ( ; w.queue ; )
		hl_free_estimate(hl_delete_estimate_queue(&w));
}

int
hl_estimate(RESOURCE * r,REAL1 * point,HL_PATTERN * p)
{
HL_PATTERN_LIST * plst,*plst2;
HL_PATTERN * p1,**pp;
L_CHAR * url;
INTEGER64 * dc;
int show,hide;
int ret;
double reso;
	if ( r->h.mtx == 0 )
		return HL_ESTIMATE_NUTRAL;
	p = hl_copy_pattern(p);
	url = get_url_str2(&r->h.entry);
	for ( pp = &p ; *pp ; pp = &(*pp)->next ) {
		p1 = *pp;
		if ( l_strcmp(p1->url,url) == 0 ) {
			*pp = p1->next;
			p1->next = 0;
			hl_free_pattern(p1);
			break;
		}
	}
	p = sorting_pattern(p);
	plst = convert_pattern_to_id(r->h.mtx,p);
	if ( plst == 0 ) {
		hl_free_pattern(p);
		return HL_ESTIMATE_NUTRAL;
	}
	dc = convert_dc(&reso,r->h.mtx,point);
	plst = hl_sorting_by_length(plst);
	show = hide = 0;

	set_matrix_env(r->h.mtx,"create-node","disable");
	for ( plst2 = plst ; plst2 ; plst2 = plst2->next ) {
		if ( plst->len != plst2->len && show != hide )
			break;
		hl_estimate_list(r->h.mtx,plst2,dc,reso);
		if ( plst2->show_distance == 0 )
			plst2->show_distance = _LLONG_MAX;
		else	plst2->show_distance = 1/plst2->show_distance;
		if ( plst2->hide_distance == 0 )
			plst2->hide_distance = _LLONG_MAX;
		else	plst2->hide_distance = 1/plst2->hide_distance;
		if ( plst2->show_distance > 2 * plst2->hide_distance ) {
			show ++;
			continue;
		}
		if ( plst2->show_distance * 2 < plst2->hide_distance ) {
			hide ++;
			continue;
		}
	}
	if ( show > hide )
		ret = HL_ESTIMATE_SHOW;
	else if ( show < hide )
		ret = HL_ESTIMATE_HIDE;
	else	ret = HL_ESTIMATE_NUTRAL;
	hl_free_pattern_list(plst);
	hl_free_pattern(p);
	d_f_ree(dc);
	return ret;
}


HL_PATTERN_LIST * 
hl_copy_pattern_list(MATRIX * m,HL_PATTERN_LIST * lst)
{
HL_PATTERN_LIST * ret,**pp,*b;
	ret = 0;
	pp = &ret;
	for ( ; lst ; lst = lst->next ) {
		b = d_alloc(sizeof(*b));
		*b = *lst;
		if ( lst->top_dc )
			b->top_dc = copy_dim_code(m,lst->top_dc);
		b->p = hl_copy_pattern(lst->p);
		
		*pp = b;
		pp = &b->next;
	}
	return ret;
}

HL_PATTERN_LIST * 
_get_whole_pattern(MATRIX * m,HL_PATTERN * _p)
{
HL_PATTERN_LIST * a, *b,* p,**pp;
	if ( _p == 0 )
		return 0;
	a = _get_whole_pattern(m,_p->next);
	b = hl_copy_pattern_list(m,a);
	for ( pp = &b ; *pp ; pp = &(*pp)->next ) {
		p = *pp;
		p->p = hl_push_pattern(_p->url,p->p);
	}
	*pp = a;
	return b;
}

HL_PATTERN_LIST * 
get_whole_pattern(MATRIX * m,HL_PATTERN * _p)
{
HL_PATTERN_LIST * ret,*p;
	ret = _get_whole_pattern(m,_p);
	for ( p = ret ; p ; p = p->next )
		p->p = sorting_pattern(p->p);
	return ret;
}


int
hl_add_point(RESOURCE * r,HL_PATTERN * p,REAL1 * point,int show,int hide)
{
HL_PATTERN_LIST * lst,*l1;
HL_PATTERN_SFT sft;
double reso;
INTEGER64 * dc;
INTEGER64 level,mask;
HL_OPEN_WORK * ow;
int i;
MATRIX_NODE * n;
MX_STRUCT_BLOCK * b;
HL_OP op;
int er;
	if ( r->h.mtx == 0 )
		return -1;
	set_matrix_env(r->h.mtx,"create-node","enable");
	dc = convert_dc(&reso,r->h.mtx,point);
	lst = get_whole_pattern(r->h.mtx,p);
	for ( l1 = lst ; l1 ; l1 = l1->next ) {
		if ( convert_pattern_to_id_hit(&sft,r->h.mtx,l1->p,0) >= 0 ) {
			level = dc[0] < sft.top_dc[0] ? sft.top_dc[0] : dc[0];
			for ( ; level < r->h.mtx->total_levels ; level ++ ) {
				for ( i = 0 ; i < r->h.mtx->p.dim ; i ++ ) {
					mask = -(((INTEGER64)1)<<(level*r->h.mtx->dim_divide[i] + r->h.mtx->block_size[i]));
					if ( (dc[i+1]&mask) != (sft.top_dc[i+1]&mask) )
						goto next;
				}
				break;
			next:	;
			}
			if ( level == r->h.mtx->total_levels ) {
				free_pattern_sft(&sft);
				goto err1;
			}
			if ( sft.top_dc[0] == level )
				goto next2;
			sft.top_dc[0] = level;
			for ( i = 0 ; i < r->h.mtx->p.dim ; i ++ ) {
				mask = -(((INTEGER64)1)<<(level*r->h.mtx->dim_divide[i] + r->h.mtx->block_size[i]));
				sft.top_dc[i+1] &= mask;
			}
		}
		else {
			ow = r->h.mtx->open_work;
			memset(&sft,0,sizeof(sft));
			ow->param_i[PTN_ID_MAX]++;
			ow->dirty_flag = NF_DIRTY;
			sft.id = ow->param_i[PTN_ID_MAX];
			sft.top_dc = copy_dim_code(r->h.mtx,dc);
			for ( i = 0 ; i < r->h.mtx->p.dim ; i ++ ) {
				mask = -(((INTEGER64)1)<<(sft.top_dc[0]*r->h.mtx->dim_divide[i] + r->h.mtx->block_size[i]));
				sft.top_dc[i+1] &= mask;
			}
			sft.dim = r->h.mtx->p.dim;
		}
		/***/
		/* save sft */
		convert_pattern_to_id_hit(&sft,r->h.mtx,l1->p,1);
	next2:
		/* save tree */

		get_matrix_node_channel(&er,&n,
				r->h.mtx,dc,HL_CH_DATA,GN_TREE,GN_LIST_CREATE,0);
		if ( n == 0 )
			er_panic("get_matrix_node_wait");
		b = get_channel_struct_block(n,HL_CH_DATA);
		if ( b == 0 )
			er_panic("get_matrix_node_wait(2)");
		op.insert.id = sft.id;
		op.insert.ix = &dc[1];
		op.insert.reso = reso;
		if ( show )
			op.insert.show = 1;
		if ( hide )
			op.insert.hide = 1;
		(*b->tbl->operation)(SBOP_INSERT,b,&op);
		matrix_node_channel_unlock(n,0);
		matrix_node_channel_unlock(n,0);
		unlock_node(n,0);
		;
		free_pattern_sft(&sft);
	}
	d_f_ree(dc);
	hl_free_pattern_list(lst);
	return 0;
err1:
	d_f_ree(dc);
	hl_free_pattern_list(lst);
	return -1;
}


GBVIEW_LAYER_STATUS *
sorting_layer_status(GBVIEW_LAYER_STATUS*ls)
{
GBVIEW_LAYER_STATUS * a,* b,*c,**pp;
REAL1 m1,m2;
	if ( ls == 0 )
		return 0;
	if ( ls->next == 0 )
		return ls;
	a = b = 0;
	for ( ; ls ; ) {
		c = ls;
		ls = ls->next;
		
		c->next = a;
		a = c;
		
		if ( ls == 0 )
			break;
		
		c = ls;
		ls = ls->next;
		
		c->next = b;
		b = c;
	}
	
	a = sorting_layer_status(a);
	b = sorting_layer_status(b);
	
	ls = 0;
	pp = &ls;
	
	for ( ; a && b ; ) {
		m1 = (a->small_rect.br.x - a->small_rect.tl.x) *
			(a->small_rect.br.y - a->small_rect.tl.y);
		m2 = (b->small_rect.br.x - b->small_rect.tl.x) *
			(b->small_rect.br.y - b->small_rect.tl.y);
		if ( m1 < m2 ) {
			c = a;
			a = a->next;
			
			*pp = c;
			c->next = 0;
			pp = &c->next;
		}
		else {
			c = b;
			b = b->next;
			
			*pp = c;
			c->next = 0;
			pp = &c->next;
		}
	}
	if ( a == 0 )
		*pp = b;
	else
		*pp = a;
	return ls;
}

int
hl_estimate_loop(GBVIEW_FLAME * gf)
{
GBVIEW_STATUS sts;
GBVIEW_LAYER_STATUS * ls,*ls2, ** pp;
int dummy;
int ofs;
int cnt;
int _dummy;
HL_PATTERN * p;
GBVIEW_LAYER_STATUS * target_r;
REAL1 point[3];
	wf_status_flags(gf,&sts,SF_LAYER_DETAILS);
	if ( (sts.flags & SF_LAYER_NOS) == 0 )
		return -1;
	pp = &sts.layers;
	for ( ; *pp ; ) {
		ls = *pp;
		if ( ls->distance < 0 )
			goto del;
		if ( ls->small_rect.br.x - ls->small_rect.tl.x < sts.width/3 &&
				ls->small_rect.br.y - ls->small_rect.tl.y < sts.height/3 )
			goto del;
		pp = &ls->next;
		continue;
	del:
		*pp = ls->next;
		ls->next = 0;
		wf_free_layer_status(ls);
	}
	sts.layers = sorting_layer_status(sts.layers);
	dummy = HL_TOTAL_PATTERN/2;
	ofs = HL_TOTAL_PATTERN/2+1;
	for ( ls = sts.layers ; ls ; ls = ls->next ) {
		p = 0;
		ls2 = ls;
		_dummy = dummy;
		target_r = 0;
		for ( cnt = 0 ; cnt < HL_TOTAL_PATTERN && ls2 ; cnt ++ ) {
			if ( cnt == ofs ) {
				target_r = ls2;
			}
			else {
				p = hl_push_pattern(ls2->entry_url,p);
			}
			if ( _dummy ) {
				_dummy --;
			}
			else {
				ls2 = ls2->next;
			}
		}
		if ( target_r == 0 )
			break;
		if ( get_xltime() - target_r->hide_time < invalid_hide_time )
			continue;
		point[0] = target_r->distance/distance(target_r->center[0],target_r->center[1]);
		point[1] = target_r->center[0].x;
		point[2] = target_r->center[0].y;
		switch ( hl_estimate(target_r->r,point,p) ) {
		case HL_ESTIMATE_NUTRAL:
			break;
		case HL_ESTIMATE_HIDE:
			wf_set_overlay(gf,target_r->entry_url,WFF_HIDE,WFF_HIDE);
			break;
		case HL_ESTIMATE_SHOW:
			wf_set_overlay(gf,target_r->entry_url,0,WFF_HIDE);
			break;
		default:
			er_panic("estimate");
		}

	}
	wf_free_status(&sts);
	return 0;
}


void
hl_point_estimate(GBVIEW_FLAME * gf)
{
HL_QUE_DATA * d;
char buffer[50];
	d = new_queue_node(sizeof(*d));
	sprintf(buffer,"%p-point-estimate",gf);
	d->h.key = nl_copy_str(std_cm,buffer);
	d->h.pri = 0;
	d->gf = gf;
	d->target = 0;
	d->layer = 0;
	insert_queue(&hl_que,d,0);
}

int
hl_eye_estimate(GBVIEW_FLAME * gf,RESOURCE * r)
{
HL_QUE_DATA * d;
char buffer[50];
GBVIEW_STATUS sts;
GBVIEW_LAYER_STATUS * ls,*ls2, ** pp, * target;
int cnt;
GBVIEW_LAYER_STATUS * window[HL_TOTAL_PATTERN];
int window_head,window_cnt;
	wf_status_flags(gf,&sts,SF_LAYER_DETAILS);
	if ( (sts.flags & SF_LAYER_NOS) == 0 )
		return -1;
	pp = &sts.layers;
	for ( ; *pp ; ) {
		ls = *pp;
		if ( ls->distance < 0 )
			goto del;
		if ( ls->small_rect.br.x - ls->small_rect.tl.x < sts.width/3 &&
				ls->small_rect.br.y - ls->small_rect.tl.y < sts.height/3 )
			goto del;
		pp = &ls->next;
		continue;
	del:
		*pp = ls->next;
		ls->next = 0;
		wf_free_layer_status(ls);
	}
	sts.layers = sorting_layer_status(sts.layers);
	memset(window,0,sizeof(window));
	window_head = window_cnt = 0;
	target = 0;
	for ( ls = sts.layers ; ls ; ls = ls->next ) {
		if ( ls->r == r ) {
			target = ls;
			break;
		}
		window[window_head++] = ls;
		window_head = window_head % HL_TOTAL_PATTERN;
		window_cnt ++;
	}
	if ( target == 0 ) {
		wf_free_status(&sts);
		return -1;
	}
	for ( cnt = HL_TOTAL_PATTERN/2 ; cnt > 0 || window_cnt < HL_TOTAL_PATTERN ; cnt -- ) {
		window[window_head++] = ls;
		window_head = window_head % HL_TOTAL_PATTERN;
		window_cnt ++;
	}
	ls = window[window_head];
	for ( cnt = HL_TOTAL_PATTERN , ls2 = ls ; ls2 && cnt > 1 ; ls2 = ls2->next );
	wf_free_layer_status(ls2->next);
	ls2->next = 0;
	for ( pp = &sts.layers ; *pp != ls ; pp = &(*pp)->next );
	*pp = 0;
	wf_free_status(&sts);
	d = new_queue_node(sizeof(*d));
	sprintf(buffer,"%p-point-estimate",gf);
	d->h.key = nl_copy_str(std_cm,buffer);
	d->gf = gf;
	d->h.pri = 1;
	d->target = target;
	d->layer = ls;
	return 0;
}

int
hl_purge_que(SYS_QUEUE * que,HL_QUE_DATA * d,GBVIEW_FLAME * gf)
{
	if ( d->gf != gf )
		return -1;
	if ( d->target || d->layer )
		return -1;
	return 0;
}

void
hl_task(TKEY d)
{
XL_INTERPRETER * xli;
L_CHAR * key;
SYS_QUEUE * que;

HL_QUE_DATA * d1,*d2;

RESOURCE * r;
REAL1 point[3];
HL_PATTERN * p;
int show,hide;
GBVIEW_LAYER_STATUS * ls;
	que = (SYS_QUEUE *)GET_TKEY(d);
	key = touch_qkey(que);
	if ( key == 0 )
		return;

	xli = new_xl_interpreter();
	xli->a_type = XLA_SELF;
	setup_i(xli);

	for ( ; ; ) {
		d1 = delete_queue(que,0,0,0);
		if ( d1 == 0 )
			break;
		if ( d1->target == 0 ) {
			for ( ; ; ) {
				d2 = delete_queue(que,hl_purge_que,d1->gf,0);
				if ( d2 == 0 )
					break;
				d_f_ree(d2->h.key);
				d_f_ree(d2);
			}
			hl_estimate_loop(d1->gf);
		}
		else {
			if ( d1->target->flags & WFF_HIDE ) {
				hide = 1;
				show = 0;
			}
			else {
				hide = 0;
				show = 1;
			}
			r = d1->target->r;
			point[0] = d1->target->distance/distance(d1->target->center[0],d1->target->center[1]);
			point[1] = d1->target->center[0].x;
			point[2] = d1->target->center[1].y;
			p = 0;
			for ( ls = d1->layer ; ls ; ls = ls->next ) {
				p = hl_push_pattern(ls->entry_url,p);
			}
			hl_add_point(r,p,point,show,hide);
			hl_free_pattern(p);
			wf_free_layer_status(d1->layer);
		}
		d_f_ree(d1->h.key);
		d_f_ree(d1);
	}

	release_qkey(que,key);
	d_f_ree(key);

	close_self_interpreter();

}

void
init_h_layering()
{

	memset(&hl_que,0,sizeof(SYS_QUEUE));
	hl_que.flags = QF_FIFO;
	hl_que.gc_func = 0;
	hl_que.gc_get = 0;
	hl_que.key_func = hl_task;
	hl_que.pri = PRI_FETCH;
	setup_queue(&hl_que);
}

