/** @file
 */
#if defined(HAVE_CONFIG_H)
#  include "../../config.h"
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <iconv.h>

#include <cerrno>
#include <sstream>
#include <queue>

#include <boost/format.hpp>

#include "../gettext.h"
#include <glibmm/i18n.h>
#include <glibmm/main.h>
#include <gtkmm/main.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-directory.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>

#include "hyperestraier/hyperestraier.hpp"
#include "gdestraier.hpp"
#include "model/preferences.hpp"
#include "builder.hpp"
#include "document_draft_filter.hpp"
#include "text_filter.hpp"
#include "html_filter.hpp"
#include "id3_filter.hpp"
#include "unknown_filter.hpp"

#define GATHER_VFS_FILEINFO_OPTIONS ::GnomeVFSFileInfoOptions(::GNOME_VFS_FILE_INFO_GET_MIME_TYPE)


#define DEBUG_GATHERER_BENCHMARK   0


namespace gdestraier {
  namespace builder {

    /** @brief Default constractor
     *
     *  Initialy, the builder skip all build processes.(call null-builder)
     *
     *  You must turn on the flaver flags before launch().
     */
    builder::builder() :
      target_index_(0),
      lock_timeout_(64),
      collect_overwrite_spaces_(false),
      collect_deleted_spaces_(false),
      gather_documents_(false),
      purge_nonexist_documents_(false),
      optimize_(false),
      generate_keywords_(false),
      cache_size_(64),
      canceled_(false)
    {
    }


    builder::~builder()
    {
    }


    /** @brief インデックスの構築を実行します
     *  @param index 対象のインデックス。NULLを指定した場合、すでに関連付いている
     *               インデックスが対象になります。
     */
    void
    builder::launch(gdestraier::model::index_type const* index)
    {
      if (index != 0) target_index_ = index;

      if (target_index_->document_path_.empty()) return; // 文書ファイルに関連付いていないので無視

      if (index->database_location_ != gdestraier::model::index_type::LOCAL_FILESYSTEM) {

        gdestraier::model::preferences const& pref = gdestraier::gui::get_preferences();

        hyperestraier::node_database db(index->database_path_.c_str());
        db.set_timeout(pref.network_.timeout_);
        db.set_auth(index->user_.c_str(), index->password_.c_str());

        std::string proxy_host;
        int         proxy_port;
        pref.get_proxy(&proxy_host, &proxy_port);
        if (! proxy_host.empty())
          db.set_proxy(proxy_host.c_str(), proxy_port);

        if (gather_documents_)         run_gather(db);
        if (purge_nonexist_documents_) run_purge(db);

        return ;
      } else {

        local_db_type db;


        if (gather_documents_ || purge_nonexist_documents_ || optimize_) {

          int ecode = db.open(target_index_->database_path_.c_str(),
                              ESTDBWRITER | ESTDBCREAT | (index->use_Ngram_for_all_languages_? ESTDBPERFNG : 0) );
          if (! db.is_opened()) {
            signal_stderr_.emit(db.get_error_message(ecode));
            signal_stderr_.emit("\n");
          } else {

            db.set_cache_size(cache_size_, -1, -1, -1);

            if (gather_documents_)         run_gather(db);
            if (purge_nonexist_documents_) run_purge(db);


            if (! canceled_ && optimize_) {
              signal_stdout_.emit(_("Optimizing database ..."));

              if (! db.optimize(collect_deleted_spaces_? 0 : ESTOPTNOPURGE) )
                signal_stderr_.emit(db.get_error_message(db.get_db_error()) );

              signal_stdout_.emit(_(" done.\n"));
            }

            db.close(); // タイミングが明確である必要があるので、明示的にクローズ。
          }
        }

        if (generate_keywords_)        run_extkeys();
      }
    }


    void
    builder::cancel()
    {
      canceled_ = true;
    }



    void builder::run_gather(hyperestraier::database& db)
    {
      struct visitor_type {

        ::gdestraier::builder::builder* builder_;
        ::hyperestraier::database* db_;
        ::GnomeVFSURI* base_uri_;


        struct uri_item_type {
          ::GnomeVFSFileInfo* info_;
          ::GnomeVFSURI*      uri_;
        };
        enum { URI_QUEUE_SIZE = 1024 };
        uri_item_type uri_queue_[URI_QUEUE_SIZE];
        uri_item_type* uri_queue_tail_;

        visitor_type() : uri_queue_tail_(uri_queue_) { }

        void flush_uri_queue() {
          for (uri_item_type* i = uri_queue_; i < uri_queue_tail_; i++) {
            builder_->put_doc(*db_, i->uri_, i->info_);
            ::gnome_vfs_uri_unref(i->uri_);
          }

          uri_queue_tail_ = uri_queue_;
        }

        static ::gboolean on_visit(::gchar const* rel_path,
                                   ::GnomeVFSFileInfo* info,
                                   ::gboolean recursing_will_loop,
                                   ::gpointer data,
                                   ::gboolean* recursive)
        {
          visitor_type* me = (visitor_type*)data;

          if (me->builder_->is_canceled()) return false; // キャンセルされていた

          // キューに積む
          me->uri_queue_tail_->info_ = ::gnome_vfs_file_info_dup(info); // 戻ると上書きされるのでコピー

          char* escaped_rel_path = ::gnome_vfs_escape_path_string(rel_path);
          me->uri_queue_tail_->uri_  = ::gnome_vfs_uri_resolve_relative(me->base_uri_,
                                                                        escaped_rel_path);
          ::g_free(escaped_rel_path);

          if (++(me->uri_queue_tail_) >= me->uri_queue_ + URI_QUEUE_SIZE)
            me->flush_uri_queue();

          *recursive = ! recursing_will_loop;
          return true;
        }
      };



      ::GnomeVFSFileInfo* info = ::gnome_vfs_file_info_new();
      if (::gnome_vfs_get_file_info(target_index_->document_path_.c_str(),
                                    info,
                                    ::GNOME_VFS_FILE_INFO_DEFAULT) != GNOME_VFS_OK) {
        ::g_warning("Couldn't get file info for document root.:%s",
                    target_index_->document_path_.c_str());
        ::gnome_vfs_file_info_unref(info);
        return;
      }
      bool root_is_dir = (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY);
      ::gnome_vfs_file_info_unref(info);

      if (root_is_dir) {
        // NOTE: 末尾に '/' が無いと、相対パス解決の時に末尾要素が消されてしまう。
        std::string root = target_index_->document_path_ + "/";

        visitor_type visitor;
        visitor.builder_ = this;
        visitor.db_ = &db;
        visitor.base_uri_ = ::gnome_vfs_uri_new(root.c_str());
        
        ::GnomeVFSResult res = ::gnome_vfs_directory_visit_uri(visitor.base_uri_,
                                                               GATHER_VFS_FILEINFO_OPTIONS,
                                                               ::GNOME_VFS_DIRECTORY_VISIT_LOOPCHECK,
                                                               visitor_type::on_visit,
                                                               ::gpointer(&visitor) );
        visitor.flush_uri_queue();
        if (res != GNOME_VFS_OK)
          ::g_warning("gnome_vfs_directory_visit_uri failed: %s",
                      ::gnome_vfs_result_to_string(res) );

        ::gnome_vfs_uri_unref(visitor.base_uri_);
      } else {
        ::g_warning("Document root is not directory.: %s",
                    target_index_->document_path_.c_str());
      }
    }




    /**
     * 文書を前処理してインデックスに登録します。
     *
     *  NOTE: このメソッドが呼ばれたという事はとにかくリソースは存在するので、
     *        エラー時にも処理を中断せずに取り合えず登録する事が望ましい。
     */
    void builder::put_doc(hyperestraier::database& db,
                          ::GnomeVFSURI* uri,
                          ::GnomeVFSFileInfo* info)
    {
      static boost::format logfmt(_("Processing: %1$s ... %2$s\n"));
      static std::string const log_stat_failed(_("stat failed"));
      static std::string const log_skip(_("Skip"));
      static std::string const log_unknown_file_type(_("Unknown file type"));
      static std::string const log_filter_failed(_("Filter failed"));
      static std::string const log_put_failed(_("Failed to put to database."));
      static std::string const log_succeded(_("Success."));


      // 呼出元からinfoをもらえなかった場合には、自分で検索する
      if (info == 0) {
        info = ::gnome_vfs_file_info_new();
        ::GnomeVFSResult res =
            ::gnome_vfs_get_file_info_uri(uri, info, GATHER_VFS_FILEINFO_OPTIONS);
        if (res == GNOME_VFS_OK) {
          if (info != 0) ::gnome_vfs_file_info_unref(info);
          return ;
        }
      }


      // URIの文字列表現を得る
      char* text_uri = ::gnome_vfs_uri_to_string(uri,
                                                 ::GnomeVFSURIHideOptions(::GNOME_VFS_URI_HIDE_USER_NAME |
                                                                          ::GNOME_VFS_URI_HIDE_PASSWORD) );

      // この文書の更新が必要であるか判断する。
      if (info != 0) {
        char const* attr_mdate = db.get_doc_attr(text_uri, ESTDATTRMDATE, 0);
        if (attr_mdate != 0) {
          std::time_t doc_mdate = ::cbstrmktime(attr_mdate);
          if (doc_mdate != std::time_t(-1) && doc_mdate >= info->mtime) {
            signal_stdout_.emit((logfmt % text_uri % log_skip).str().c_str());

            ::gnome_vfs_file_info_unref(info);
            ::g_free(text_uri);

            return ; // エラーが無く、かつ文書が更新されていなかったので再インデックス不要
          }
        }
      }



#if DEBUG_GATHERER_BENCHMARK
      double t1, t2, t3;
      t1 = ::est_gettimeofday();
#endif

      //
      // 更新が必要である事がハッキリしたので、文書ドラフトを生成する
      //
      hyperestraier::local_document doc;
      if (info->type == ::GNOME_VFS_FILE_TYPE_DIRECTORY) {
        //
        // ディレクトリだったので、ディレクトリ用の文書ドラフトを作る
        //
        doc.create();
        doc.set_attr(ESTDATTRTYPE, info->mime_type);
        doc.set_attr(ESTDATTRTITLE, info->name);
        // doc.set_attr(ESTDATTRAUTHOR, "");  // TODO: オーナのユーザ名
      } else {
        //
        // ファイルだったのでフィルタを適用して文書ドラフトを作る
        //

        // まず、ファイルタイプを決定し、フィルタを生成する
        gdestraier::builder::filter::abstract_filter* docfilter = 0;
        char const* mime_type = 0;

        if (target_index_->filetype_ == &gdestraier::model::filetype::document_draft_)
          docfilter = gdestraier::builder::filter::from_document_draft::create();
        else if (target_index_->filetype_ == &gdestraier::model::filetype::plain_text_)
          docfilter = gdestraier::builder::filter::from_text::create();
        else if (target_index_->filetype_ == &gdestraier::model::filetype::html_)
          docfilter = filter::from_html::create();
#if 0
        else if (target_index_->filetype_ == &gdestraier::model::filetype::mime_)
          docfilter = filter::from_mime::create();
#endif
#if defined(HAVE_LIBID3)
        else if (target_index_->filetype_ == &gdestraier::model::filetype::mp3_)
          docfilter = filter::from_id3::create();
#endif
        else if (target_index_->filetype_ == &gdestraier::model::filetype::auto_) {

          // 先に拡張子に対応するフィルタのファクトリを探す
          //   NOTE: 先にMIME型から探すと、テキストベースの不明なタイプのファイルが全てtext/plain
          //         になってしまい、Gnomeは知らないけどgdestraierは知っているタイプを処理できない為。

          // 拡張子を抽出する
          std::string ext;
          {
            char const* p = text_uri;
            char const* ext_first = 0;
            while (*p != '\0') {
              if (*p == '.') ext_first = p + 1;
              else if (*p == '/') ext_first = 0;
              p++;
            }
            if (ext_first != 0) ext.assign(ext_first, p);
          }

          // ファクトリから探す
          typedef gdestraier::builder::filter::available_factories factory_map_type;
          for (factory_map_type::const_iterator i = factory_map_type::instance().begin();
               i != factory_map_type::instance().end(); i++) {
            gdestraier::builder::filter::factory::extention_map_type const* exts = i->second->get_extentions();

            while (exts->ext_ != 0 && exts->mime_type_ != 0) {
              if (exts->ext_ != 0 && ext == exts->ext_) {
                docfilter = i->second->create_filter();
                mime_type = exts->mime_type_;
                break;
              }
              exts++;
            }
          }

          // 拡張子基準で見付からなかったので、GnomeにMIME型を問い合わせて、MIME型を元に
          // フィルタを決定する。
          if (docfilter == 0) {
            mime_type = info->mime_type;

            for (factory_map_type::const_iterator i = factory_map_type::instance().begin();
                 i != factory_map_type::instance().end(); i++) {
              gdestraier::builder::filter::factory::extention_map_type const* exts = i->second->get_extentions();

              while (exts->ext_ != 0 && exts->mime_type_ != 0) {
                if (exts->mime_type_ != 0 && std::strcmp(mime_type, exts->mime_type_) == 0) {
                  docfilter = i->second->create_filter();
                  break;
                }
                exts++;
              }
            }
          }
        }

        // 適切なフィルタが見付からなかったので、unknown として処理する
        if (docfilter == 0)
          docfilter = filter::from_unknown::create();




        //
        // フィルタを使って文書オブジェクトを構築
        //
        if (! (*docfilter)(&doc, *target_index_, uri, text_uri, info, mime_type)) {
          signal_stdout_.emit((logfmt % text_uri % log_filter_failed).str().c_str());

          ::gnome_vfs_file_info_unref(info);
          ::g_free(text_uri);

          return ;
        }
      }


      //
      // 構築した文書をインデックスに登録する
      //
      doc.set_attr(ESTDATTRURI, text_uri);
      doc.set_attr(ESTDATTRSIZE, (boost::format("%d") % info->size).str().c_str());
      char const* title = doc.get_attr(ESTDATTRTITLE, 0);
      if (! title)
        doc.set_attr(ESTDATTRTITLE, (title = info->name));

      // ファイル名とタイトルを隠しテキストとして追加しておく
      if (target_index_->include_title_to_body_)
        doc.add_text(title, true);

      if (target_index_->include_uri_to_body_) {
        char* unescaped_uri = ::gnome_vfs_unescape_string(text_uri, "/");
        if (unescaped_uri != 0) {
          doc.add_text(unescaped_uri, true);
          ::g_free(unescaped_uri);
        } else
          doc.add_text(text_uri, true);
      }

      // タイムスタンプ属性を付ける
      {
        char* p = ::cbdatestrwww(info->ctime, 0);
        doc.set_attr(ESTDATTRCDATE, p);
        ::free(p);

        p = ::cbdatestrwww(info->mtime, 0);
        doc.set_attr(ESTDATTRMDATE, p);
        ::free(p);

        p = ::cbdatestrwww(info->atime, 0);
        doc.set_attr(ESTDATTRADATE, p);
        ::free(p);
      }

#if DEBUG_GATHERER_BENCHMARK
      t3 = ::est_gettimeofday();
#endif
#if 1
      logfmt % text_uri;
      if (db.put_doc(doc, (collect_overwrite_spaces_? ESTPDCLEAN : 0) ))
        logfmt % log_succeded;
      else
        logfmt % log_put_failed;
      signal_stdout_.emit(logfmt.str().c_str());
#else
      db.put_doc(doc, 0) ;
#endif

#if DEBUG_GATHERER_BENCHMARK
      t2 = ::est_gettimeofday();
      ::g_print("%d\t%d\t%d\t%d\t%s\n", int(t2 - t1), int(t2-t3), int(t3 - t1), int(info->size), text_uri);
#endif

      ::gnome_vfs_file_info_unref(info);
      ::g_free(text_uri);
    }





    /**
     * リソースの実体が存在しない文書をインデックスから除去します。
     */
    void builder::run_purge(hyperestraier::database& db)
    {
      if (canceled_) return ;

      signal_stdout_.emit(_("Purging nonexist documents ...\n"));

      int opts = collect_deleted_spaces_? ESTODCLEAN : 0;

      if (! db.iter_init()) {
        signal_stdout_.emit(_("Failed to iter_init.\n"));
        return ;
      }


      while (1) {
        hyperestraier::local_document doc(db.iter_next());
        if (! doc) break;

        char const* uri = doc.get_attr(ESTDATTRURI, 0);
        if (uri != 0) {
          bool uri_exists = true;  // 検査に失敗した場合には、安全の為に存在していた事にする。


          // URIに該当するリソースが存在するか検査する
          ::GnomeVFSURI* vfsuri = ::gnome_vfs_uri_new(uri);
          if (vfsuri == 0)
            signal_stdout_.emit("Failed to parse uri to GnomeVFSURI\n");
          else {
            uri_exists = (::gnome_vfs_uri_exists(vfsuri) != FALSE);
            ::gnome_vfs_uri_unref(vfsuri);
          }

          // リソースが存在しなかったので、除去する
          if (! uri_exists) {
            signal_stderr_.emit(uri);
            signal_stderr_.emit("\n");
            if (! db.out_doc(uri, opts))
              signal_stderr_.emit(_("Failed to out_doc\n"));
          }
        }
      }


      signal_stdout_.emit(_(" done.\n"));
    }




    void builder::run_extkeys()
    {
      gdestraier::model::preferences const& pref = gdestraier::gui::get_preferences();

      std::vector<std::string> args;

      args.push_back(pref.estcmd_path_);
      args.push_back("extkeys");
      args.push_back(target_index_->database_path_);

      run_command(args);
    }





    namespace {
      class log_spooler {
      public:
        Glib::RefPtr<Glib::IOChannel> iochannel_;
        sigc::signal<void, char const*>& signal_;
        sigc::connection connection_;
        volatile bool eof_;

        log_spooler(int fd, sigc::signal<void, char const*>& signal) :
          signal_(signal),
          eof_(false)
        {
          iochannel_ = Glib::IOChannel::create_from_fd(fd);
          connection_ = Glib::signal_io().connect(sigc::mem_fun(this, &log_spooler::on_io_event),
                                                  iochannel_,
                                                  Glib::IO_IN | Glib::IO_HUP | Glib::IO_ERR);
        }

        ~log_spooler() {
          connection_.disconnect();
        }


        bool on_io_event(Glib::IOCondition condition) {
          if (condition == Glib::IO_HUP || condition == Glib::IO_ERR) {
            eof_ = true;
            return false;
          } else {
            Glib::ustring line;
            if (iochannel_->read_line(line) == Glib::IO_STATUS_NORMAL)
              signal_.emit(line.c_str());
          }
          return true;
        }
        
      };
    }

    void builder::run_command(std::vector<std::string> const& args)
    {
      {
        signal_stdout_.emit(_("Running "));
        std::stringstream buf;
        for (std::vector<std::string>::const_iterator i = args.begin(); i != args.end(); i++)
          buf << *i << " ";
        buf << std::endl;

        signal_stdout_.emit(buf.str().c_str());
      }


      Glib::Pid pid;
      int stdout_pipe, stderr_pipe;
      try {
        Glib::spawn_async_with_pipes("/", // @TODO エラーの時どうなるの？
                                     args,
                                     Glib::SpawnFlags(0),
                                     sigc::slot<void>(),
                                     &pid,
                                     0, &stdout_pipe, &stderr_pipe);
      }
      catch (Glib::SpawnError e) {
        g_warning("estcmd spawn error: %s", e.what().c_str());
        return ;
      }
      log_spooler stdout_spooler(stdout_pipe, signal_stdout_);
      log_spooler stderr_spooler(stderr_pipe, signal_stderr_);

      while (! canceled_ && (! stdout_spooler.eof_ || ! stderr_spooler.eof_))
        Gtk::Main::iteration(false);

      Glib::spawn_close_pid(pid);

      signal_stdout_.emit(_("=== Finished ===\n"));
    }

  }
}

