/** @file
 */
#if defined(HAVE_CONFIG_H)
#  include "../../config.h"
#endif
#include <cstring>
#include <cstdlib>
#include <boost/format.hpp>
#include <glib/gfileutils.h>
#include <glib/gmessages.h>
#include <libxml++/libxml++.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include "preferences.hpp"

namespace gdestraier {
  namespace model {

    preferences::preferences()
    {
      GUI_.mainwindow_.top_ = -1;
      GUI_.mainwindow_.left_ = -1;
      GUI_.mainwindow_.width_ = 450;
      GUI_.mainwindow_.height_ = 200;

      path_ = PKGDATADIR "/gdestraierrc";
      load();


      path_ = getenv("HOME");
      path_ += "/.gdestraierrc";
    }


    preferences::~preferences()
    {
    }



    /** @brief 設定をファイルへ保存します
     */
    void
    preferences::save() const
    {
      xmlpp::Document xmldoc;
      xmlpp::Element* root = xmldoc.create_root_node("gdestraier-config");


      //
      // GUIの状態を保存します
      //
      {
        xmlpp::Element* gui_elm = root->add_child("GUI");

        xmlpp::Element* elm = gui_elm->add_child("mainwindow");
        elm->set_attribute("top", (boost::format("%d") % GUI_.mainwindow_.top_).str());
        elm->set_attribute("left", (boost::format("%d") % GUI_.mainwindow_.left_).str());
        elm->set_attribute("width", (boost::format("%d") % GUI_.mainwindow_.width_).str());
        elm->set_attribute("height", (boost::format("%d") % GUI_.mainwindow_.height_).str());
      }

      //
      // 外部コマンドの設定を保存します
      //
      {
        xmlpp::Element* extcmd_elm = root->add_child("external_commands");
        xmlpp::Element* cmd_elm = extcmd_elm->add_child("estcmd");
        cmd_elm->set_attribute("path", estcmd_path_);
      }


      //
      // ネットワークの設定を保存します
      //
      {
        xmlpp::Element* elm = root->add_child("network");
        elm->set_attribute("timeout", (boost::format("%d") % network_.timeout_).str());
        elm->set_attribute("proxy_host", network_.proxy_host_);
        elm->set_attribute("proxy_port", (boost::format("%d") % network_.proxy_port_).str());
        elm->set_attribute("use_gnome_proxy", network_.use_gnome_proxy_settings_? "yes" : "no");
      }


      //
      // 登録されたインデックスを保存します
      //
      {
        xmlpp::Element* indexes_elm = root->add_child("indexes");
        for (indexes_type::const_iterator i = indexes_.begin(); i != indexes_.end(); i++) {
          xmlpp::Element* index_elm = indexes_elm->add_child("index");
          index_elm->set_attribute("description", i->description_);

          index_elm->set_attribute("location", (boost::format("%d") % i->database_location_).str());
          index_elm->set_attribute("user", i->user_);
          index_elm->set_attribute("password", i->password_);

          index_elm->set_attribute("path", i->database_path_);
          index_elm->set_attribute("document", i->document_path_);
          index_elm->set_attribute("active", i->active_? "yes" : "no");
          index_elm->set_attribute("document_encoding",
                                   ((i->document_encoding_ == 0)? "Auto" : i->document_encoding_->id_));
          index_elm->set_attribute("filetype", i->filetype_->get_name());
          index_elm->set_attribute("language", i->language_);
          index_elm->set_attribute("use_Ngram_for_all_languages",
                                   i->use_Ngram_for_all_languages_? "yes" : "no");
          index_elm->set_attribute("include_document_title_to_body",
                                   i->include_title_to_body_? "yes" : "no");
          index_elm->set_attribute("include_document_uri_to_body",
                                   i->include_uri_to_body_? "yes" : "no");
          index_elm->set_attribute("quickbuild",
                                   i->is_enable_quick_build_? "yes" : "no");

          {
            xmlpp::Element* replace_elm = index_elm->add_child("uri_replace");
            replace_elm->set_attribute("regex", i->get_uri_replace_regex());
            replace_elm->set_attribute("replace", i->get_uri_replace_to());

            replace_elm = index_elm->add_child("title_replace");
            replace_elm->set_attribute("regex", i->get_title_replace_regex());
            replace_elm->set_attribute("replace", i->get_title_replace_to());
          }

          index_elm->set_attribute("depth", (boost::format("%d") % i->get_depth()).str());
          index_elm->set_attribute("max_documents", (boost::format("%d") % i->get_max_documents()).str());
          index_elm->set_attribute("use_snippet", i->get_use_snippet()? "yes" : "no");
        }
      }


      //
      // フォントと色の設定を保存します
      //
      {
        xmlpp::Element* font_and_colors_elm = root->add_child("font-and-colors");

        static char const* const targets[] = {
          "background", "group-header", "title", "score", "file-type", "author",
          "last-modified", "language", "index",
          "keyword-1", "keyword-2", "keyword-3", 
          0
        };

        for (char const* const* p = targets; *p != 0; p++) {
          font_and_color const* fc = get_font_and_color(*p);
          if (fc != 0) {
            xmlpp::Element* elm = font_and_colors_elm->add_child(*p);
            elm->set_attribute("font", fc->font_name_);
            elm->set_attribute("normal-color", fc->normal_color_);
            elm->set_attribute("highlight-color", fc->highlight_color_);
          }
        }
      }


      //
      // アイコンとサムネイルの設定を保存します
      //
      {
        xmlpp::Element* thumbnail_elm = root->add_child("thumbnail");
        thumbnail_elm->set_attribute("size", (boost::format("%d") % thumbnail_size_).str());
      }


      //
      // その他の設定を保存します
      //
      {
        xmlpp::Element* elm = root->add_child("group-key");
        switch (group_key_) {
        case GROUP_BY_NONE:           elm->set_attribute("key", "none"); break;
        case GROUP_BY_INDEX:          elm->set_attribute("key", "index"); break;
        case GROUP_BY_MIME_TYPE:      elm->set_attribute("key", "mime-type"); break;
        case GROUP_BY_AUTHOR:         elm->set_attribute("key", "author"); break;
        case GROUP_BY_LANGUAGE:       elm->set_attribute("key", "language"); break;
        case GROUP_BY_LAST_MODIFIED:  elm->set_attribute("key", "last-modified"); break;
        }

        elm = root->add_child("sort-key");
        switch (sort_key_) {
        case SORT_BY_SCORE:          elm->set_attribute("key", "score"); break;
        case SORT_BY_LAST_MODIFIED:  elm->set_attribute("key", "last-modified"); break;
        case SORT_BY_TITLE:          elm->set_attribute("key", "title"); break;
        }
      }

      xmldoc.write_to_file(path_, "UTF-8");
    }




    /** @brief 設定をファイルから読み込みます
     */
    void
    preferences::load()
    {
      if (! ::g_file_test(path_.c_str(), ::G_FILE_TEST_EXISTS))
        return ; // ファイルが無いので無視


      struct myparser : public xmlpp::SaxParser {
        enum {
          OUTSIDE, GUI, INDEXES, INDEX, EXTCOMMANDS, FONT_AND_COLORS
        };
        unsigned int state;
        index_type* current_index;
        preferences& pref_;

        myparser(preferences& pref) :
          SaxParser(false),
          state(OUTSIDE), current_index(0), pref_(pref)
        {
        }

        void on_start_element(Glib::ustring const& name, 
                              AttributeList const& attributes) {
          switch (state) {
          case OUTSIDE:
            if (name == "GUI")                    { state = GUI; }
            else if (name == "indexes")           { state = INDEXES; }
            else if (name == "external_commands") { state = EXTCOMMANDS; }
            else if (name == "font-and-colors")   { state = FONT_AND_COLORS; }
            else if (name == "group-key") {
              for (myparser::AttributeList::const_iterator i = attributes.begin();
                   i != attributes.end(); i++) {
                if (i->name == "key") {
                  if (i->value == "none")           pref_.group_key_ = preferences::GROUP_BY_NONE;
                  else if (i->value == "index")     pref_.group_key_ = preferences::GROUP_BY_INDEX;
                  else if (i->value == "mime-type") pref_.group_key_ = preferences::GROUP_BY_MIME_TYPE;
                  else if (i->value == "author")    pref_.group_key_ = preferences::GROUP_BY_AUTHOR;
                  else if (i->value == "language")  pref_.group_key_ = preferences::GROUP_BY_LANGUAGE;
                  else if (i->value == "last-modified") pref_.group_key_ = preferences::GROUP_BY_LAST_MODIFIED;
                  else                              pref_.group_key_ = preferences::GROUP_BY_INDEX;
                }
              }
            } else if (name == "sort-key") {
              for (myparser::AttributeList::const_iterator i = attributes.begin();
                   i != attributes.end(); i++) {
                if (i->name == "key") {
                  if (i->value == "title")              pref_.sort_key_ = preferences::SORT_BY_TITLE;
                  else if (i->value == "last-modified") pref_.sort_key_ = preferences::SORT_BY_LAST_MODIFIED;
                  else                                  pref_.sort_key_ = preferences::SORT_BY_SCORE;
                }
              }
            } else if (name == "thumbnail") {
              for (myparser::AttributeList::const_iterator i = attributes.begin();
                   i != attributes.end(); i++) {
                if (i->name == "size") pref_.thumbnail_size_ = std::atoi(i->value.c_str());
              }
            } else if (name == "network") {
              for (myparser::AttributeList::const_iterator i = attributes.begin();
                   i != attributes.end(); i++) {
                if (i->name == "timeout")
                  pref_.network_.timeout_ = std::atoi(i->value.c_str());
                else if (i->name == "proxy_host")
                  pref_.network_.proxy_host_ = i->value;
                else if (i->name == "proxy_port")
                  pref_.network_.proxy_port_ = std::atoi(i->value.c_str());
                else if (i->name == "use_gnome_proxy")
                  pref_.network_.use_gnome_proxy_settings_ = (i->value == "yes");
              }
            }
            break;


          case GUI:
            if (name == "mainwindow") {
              for (myparser::AttributeList::const_iterator i = attributes.begin();
                   i != attributes.end(); i++) {
                if (i->name == "width")        pref_.GUI_.mainwindow_.width_ = std::atoi(i->value.c_str());
                else if (i->name == "height")  pref_.GUI_.mainwindow_.height_ = std::atoi(i->value.c_str());
                else if (i->name == "top")     pref_.GUI_.mainwindow_.top_ = std::atoi(i->value.c_str());
                else if (i->name == "left")    pref_.GUI_.mainwindow_.left_ = std::atoi(i->value.c_str());
              }
            }
            break;

          case INDEXES:
            if (name == "index") {
              index_type tmp;
              pref_.indexes_.push_back(tmp);
              current_index = &(pref_.indexes_.back());

              current_index->active_ = false;
              current_index->document_encoding_ = 0;
              current_index->filetype_ = filetypes_manager::instance().find("Auto");
              current_index->language_ = "en";
              current_index->use_Ngram_for_all_languages_ = false;
              current_index->is_enable_quick_build_ = false;

              for (myparser::AttributeList::const_iterator i = attributes.begin();
                   i != attributes.end(); i++) {
                if (i->name == "description")
                  current_index->description_ = i->value;
                else if (i->name == "location") {
                  int n = std::atoi(i->value.c_str());
                  if (n == gdestraier::model::index_type::LOCAL_FILESYSTEM ||
                      n == gdestraier::model::index_type::REMOTE_NODE)
                    current_index->database_location_ = n;
                } else if (i->name == "user")
                  current_index->user_ = i->value;
                else if (i->name == "password")
                  current_index->password_ = i->value;
                else if (i->name == "path")
                  current_index->database_path_ = i->value;
                else if (i->name == "document")
                  current_index->document_path_ = i->value;
                else if (i->name == "active")
                  current_index->active_ = (i->value == "yes")? true : false;
                else if (i->name == "filetype")
                  current_index->filetype_ = filetypes_manager::instance().find(i->value.c_str());
                else if (i->name == "document_encoding") {
                  current_index->document_encoding_ = (i->value == "Auto")? 0 : encoding::find(i->value.c_str());
                } else if (i->name == "language")
                  current_index->language_ = i->value;
                else if (i->name == "use_Ngram_for_all_languages")
                  current_index->use_Ngram_for_all_languages_ = (i->value == "yes")? true : false;
                else if (i->name == "include_document_title_to_body")
                  current_index->include_title_to_body_ = (i->value == "yes")? true : false;
                else if (i->name == "include_document_uri_to_body")
                  current_index->include_uri_to_body_ = (i->value == "yes")? true : false;
                else if (i->name == "quickbuild")
                  current_index->is_enable_quick_build_ =  (i->value == "yes")? true : false;
                else if (i->name == "depth")
                  current_index->set_depth(std::atoi(i->value.c_str()));
                else if (i->name == "max_documents")
                  current_index->set_max_documents(std::atoi(i->value.c_str()));
                else if (i->name == "use_snippet")
                  current_index->set_use_snippet((i->value == "yes")? true : false);
              }

              state = INDEX;
            }
            break;

          case INDEX:
            if (name == "uri_replace") {
              Glib::ustring regex;
              Glib::ustring to;
              for (myparser::AttributeList::const_iterator i = attributes.begin();
                   i != attributes.end(); i++) {
                if (i->name == "regex")        regex = i->value;
                else if (i->name == "replace") to = i->value;
              }
              current_index->set_uri_replace(regex.c_str(), to.c_str());
            }
            else if (name == "title_replace") {
              Glib::ustring regex;
              Glib::ustring to;
              for (myparser::AttributeList::const_iterator i = attributes.begin();
                   i != attributes.end(); i++) {
                if (i->name == "regex")        regex = i->value;
                else if (i->name == "replace") to = i->value;
              }
              current_index->set_title_replace(regex.c_str(), to.c_str());
            }
            break;

          case EXTCOMMANDS:
            if (name == "estcmd") {
              for (myparser::AttributeList::const_iterator i = attributes.begin();
                   i != attributes.end(); i++) {
                if (i->name == "path") pref_.estcmd_path_ = i->value;
              }
            }
            break;

          case FONT_AND_COLORS:
            {
              font_and_color* fc = pref_.get_font_and_color(name.c_str());
              if (fc != 0) {
                for (myparser::AttributeList::const_iterator i = attributes.begin();
                     i != attributes.end(); i++) {
                  if (i->name == "font")                 fc->font_name_ = i->value;
                  else if (i->name == "normal-color")    fc->normal_color_ = i->value;
                  else if (i->name == "highlight-color") fc->highlight_color_ = i->value;
                }
              }
            }
            break;
          }
        }

        void on_end_element(Glib::ustring const& name) {
          if (state == INDEX && name == "index") {
            state = INDEXES;
            current_index = 0;
          } else if (name == "indexes" ||
                     (state == EXTCOMMANDS && name == "external_commands") ||
                     (state == FONT_AND_COLORS && name == "font-and-colors") || 
                     (state == GUI && name == "GUI") )
            state = OUTSIDE;
        }

      } parser(*this);

      try {
        parser.parse_file(path_);
      }
      catch (xmlpp::parse_error e) {
        ::g_warning(e.what());
      }
    }





    void
    preferences::get_proxy(std::string* host, int* port) const
    {
      if (! network_.use_gnome_proxy_settings_) {
        *host = network_.proxy_host_;
        *port = network_.proxy_port_;
      } else {
        *host = "";
        *port = 0;


        bool use_proxy = false;
        ::GConfClient* client = ::gconf_client_get_default();
        ::GConfValue* v = ::gconf_client_get_without_default(client, "/system/http_proxy/use_http_proxy", NULL);
        if (v) {
          use_proxy = ::gconf_value_get_bool(v);
          ::gconf_value_free(v);
        }

        if (use_proxy) {
          v = ::gconf_client_get_without_default(client, "/system/http_proxy/host", NULL);
          if (v) {
            *host = ::gconf_value_get_string(v);
            ::gconf_value_free(v);
          }

          v = ::gconf_client_get_without_default(client, "/system/http_proxy/port", NULL);
          if (v) {
            *port = ::gconf_value_get_int(v);
            ::gconf_value_free(v);
          }
        }

      }

    }


    index_type*
    preferences::find_index_by_name(char const* name)
    {
      for (indexes_type::iterator i = indexes_.begin();
           i != indexes_.end(); i++) {
        if (i->description_ == name) return &*i;
      }
      return 0;
    }


    font_and_color*
    preferences::get_font_and_color(char const* name)
    {
      if (! std::strcmp(name, "background"))    return &font_and_colors_.background_;
      if (! std::strcmp(name, "group-header"))  return &font_and_colors_.group_header_;
      if (! std::strcmp(name, "title"))         return &font_and_colors_.title_;
      if (! std::strcmp(name, "score"))         return &font_and_colors_.score_;
      if (! std::strcmp(name, "file-type"))     return &font_and_colors_.file_type_;
      if (! std::strcmp(name, "author"))        return &font_and_colors_.author_;
      if (! std::strcmp(name, "last-modified")) return &font_and_colors_.last_modified_;
      if (! std::strcmp(name, "language"))      return &font_and_colors_.language_;
      if (! std::strcmp(name, "index"))         return &font_and_colors_.index_;
      if (! std::strcmp(name, "keyword-1"))     return &font_and_colors_.keyword_[0];
      if (! std::strcmp(name, "keyword-2"))     return &font_and_colors_.keyword_[1];
      if (! std::strcmp(name, "keyword-3"))     return &font_and_colors_.keyword_[2];
      return 0;
    }


    font_and_color const*
    preferences::get_font_and_color(char const* name) const
    {
      preferences* self = const_cast<preferences*>(this);
      return self->get_font_and_color(name);
    }



  }
}
