#!/usr/bin/env python
# -*- coding: ascii -*-
#
#  gbottler.py - a SSTP Bottle Client
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#  Copyright (C) 2004 by Takuya KAWAHARA <num@sann.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: gbottler.py,v 1.165 2004/10/28 12:04:09 atzm Exp $
#

import os, sys, string, re, gettext, getopt
from types import *

import pygtk
pygtk.require('2.0')
import gtk

from bottlelib      import BottleClientError
from client         import Bottler
from channeldialog  import ChannelDialog
from localmessenger import LocalMessenger
from logwindow      import LogWindow
from phrases        import PhraseWindow
from prefs          import PreferenceManager, PreferenceDialog
from textmanager    import TextManager
from scriptchecker  import ScriptChecker

from viewer.ghostmanager import GhostManager
from gvwindow            import GVWindow
from svgmanager          import SVGManagerWindow

from common import *

if os.name == 'posix':
	gettext.install('gbottler')
else:
	# FIXME /-P
	os.environ['LANG'] = 'ja'
	gettext.install('gbottler', 'locale')

class Application:
	DEFAULT_TARGET = unicode(_("default"), "utf-8")

	def __init__(self, host="localhost", port=9801):
# === Definition of Basic Const === #
		self.prefs         = PreferenceManager()
		self.scriptchecker = ScriptChecker()

		self.accel_group = gtk.AccelGroup()

# === Create Base Window === #
		main_vbox = gtk.VBox(gtk.FALSE, 0)

		self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
		self.window.connect("destroy", self.quit, "WM destroy")
		self.window.set_title(unicode(_(APP), "utf-8"))
		self.window.add_accel_group(self.accel_group)
		self.window.add(main_vbox)
		self.window.set_default_size(500, 200)

# === Create Menubar === #
		self.set_self_messages()
		menu_items = self.get_menu_items()
		self.item_factory = gtk.ItemFactory(gtk.MenuBar, "<main>", self.accel_group)
		self.item_factory.create_items(menu_items)

		self.menu_list = {}
		for m in menu_items:
			# keeping ref to each items
			key = re.sub("_", "", m[0])
			self.menu_list[key] = self.item_factory.get_item(key)

		for m in ["/%s/%s" % (self.msg["file"][1], self.msg["part"]),
				  "/%s" % self.msg["channel"][1],
				  "/%s/%s" % (self.msg["edit"][1], self.msg["send"])]:
			# part, channel and send are not sensitive at startup
			self.menu_list[m].set_sensitive(gtk.FALSE)

		main_menubar = self.item_factory.get_widget("<main>")

# === Create Main Editor === #
		self.textmanager = TextManager(self, self.prefs)
		self.textmanager.set_style_script()
		self.textmanager.connect("populate-popup", self.popup)
		self.textmanager.grab_focus()

		self.editor = self.textmanager.get_buffer()
		self.editor.connect('end-user-action', self.editor_modified)

		sw = gtk.ScrolledWindow()
		sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		sw.set_shadow_type(gtk.SHADOW_IN)
		sw.add(self.textmanager)

# === Create Buttons === #
		self.tooltips = gtk.Tooltips()

		checkButton       = get_icon_button(gtk.STOCK_SPELL_CHECK, gtk.ICON_SIZE_MENU)
		testButton        = get_icon_button(gtk.STOCK_REFRESH, gtk.ICON_SIZE_MENU)
		self.send_button  = get_icon_button(gtk.STOCK_OK, gtk.ICON_SIZE_MENU)
		self.clear_button = get_icon_button(gtk.STOCK_CLEAR, gtk.ICON_SIZE_MENU)
		for item in [
			[checkButton,       self.check,         unicode(_("Check"), "utf-8")],
			[testButton,        self.test,          unicode(_("Test"), "utf-8")],
			[self.send_button,  self.send_question, unicode(_("Send"), "utf-8")],
			[self.clear_button, self.edit_clear,    unicode(_("Clear"), "utf-8")],
			]:
			item[0].connect('clicked', item[1], None)
			item[0].connect('focus-in-event', lambda x, y: self.textmanager.grab_focus())
			self.tooltips.set_tip(item[0], item[2])

		self.send_button.set_sensitive(gtk.FALSE) # at startup

# === Create Labels, e.g: channels, ghosts and users === #
		self.channel = gtk.Label()
		self.channel_event_box = gtk.EventBox()
		self.channel_event_box.set_events(gtk.gdk.BUTTON_PRESS_MASK)
		self.channel_event_box.add(self.channel)

		self.channel_ghost = gtk.Label()
		self.ghost_event_box = gtk.EventBox()
		self.ghost_event_box.set_events(gtk.gdk.BUTTON_PRESS_MASK)
		self.ghost_event_box.add(self.channel_ghost)

		self.users = gtk.Label()

# === Bring Buttons and Labels together to HBox, Frame === #
		main_toolbar = gtk.HBox(gtk.FALSE, 0)
		toolframe = gtk.Frame()
		toolframe.set_border_width(0)
		toolframe.set_shadow_type(gtk.SHADOW_OUT)
		toolframe.add(main_toolbar)

		for item in [
			checkButton, testButton, self.send_button, self.clear_button,
			gtk.VSeparator(), gtk.Label(unicode(_("Channel") + ":", "utf-8")),
			self.channel_event_box, self.ghost_event_box, gtk.VSeparator(),
			gtk.Label(unicode(_("Users") + ":", "utf-8")), self.users,
			]:
			main_toolbar.pack_start(item, gtk.FALSE, gtk.TRUE, 2)

# === Create Progressbar === #
		self.pbar = gtk.ProgressBar()
		self.pbar.set_size_request(350, -1)

		progress_frame = gtk.Frame()
		progress_frame.set_shadow_type(gtk.SHADOW_IN)
		progress_frame.set_border_width(0)
		progress_frame.add(self.pbar)

# === Create Statusbar as window resize grip === #
		self.sbar = gtk.Statusbar()
		self.sbar.set_has_resize_grip(gtk.TRUE)

# === Bring Progressbar and Statusbar together to HBox === #
		statusbar_box = gtk.HBox()
		statusbar_box.pack_start(progress_frame)
		statusbar_box.pack_start(self.sbar)

# === Bring Main Editor and Progressbar together to VPaned === #
		main_paned = gtk.VPaned()
		main_paned.pack1(sw, gtk.TRUE)
		main_paned.pack2(statusbar_box, gtk.FALSE, gtk.FALSE)

# === Bring all items together to Main Window Finally === #
		main_vbox.pack_start(main_menubar, gtk.FALSE, gtk.TRUE, 0)
		main_vbox.pack_start(toolframe, gtk.FALSE, gtk.TRUE, 0)
		main_vbox.pack_start(main_paned, gtk.TRUE, gtk.TRUE, 0) # main editor and progressbar

# === Initialization of Application === #
		[str, pos] = self.get_initial_script()
		self.textmanager.insert_with_color_from(str, pos)
		self.editor_modified()

		self.client = Bottler(self, self.prefs.get(PreferenceManager.PREF_DOWNLOAD),
							  self.prefs.get_all(),
							  self.prefs.get(PreferenceManager.PREF_LOGGING_DOWNLOAD))
		self.client.debug = 2

		try:
			luid = open(os.path.join(open_bottlecase(), "luid")).read()
		except IOError:
			luid = None
		if luid and len(luid) == 82:
			self.client.luid = luid

		self.logwindow    = LogWindow(self, self.prefs)
		self.phrasewindow = PhraseWindow(self)

		ghostmanager = GhostManager(os.path.join(open_bottlecase(), 'gviewer', 'ghost.txt'),
									os.path.join(open_bottlecase(), 'gviewer', 'Ghost'))
		self.svgmanager = SVGManagerWindow(self, self.prefs, ghostmanager)
		self.gvwindow   = GVWindow(ghostmanager) # must watch same instance of GhostManager

		self.menu_list["/%s/%s..." % (self.msg["window"][1],
									  self.msg["logs"])].connect("activate", self.logwindow.open, None)

		self.menu_list["/%s/%s..." % (self.msg["window"][1],
									  self.msg["phrases"])].connect("activate", self.phrasewindow.open, None)

		self.menu_list["/%s/%s..." % (self.msg["window"][1],
									  self.msg["gviewer"])].connect("activate", self.open_gviewer, None)

		self.menu_list["/%s/%s..." % (self.msg["window"][1],
									  self.msg["svgmanager"])].connect("activate", self.svgmanager.open, None)

		self.localmessenger = LocalMessenger(host, port, self.logwindow)

		self.target_ghost = self.DEFAULT_TARGET

		self.local_ghost = []
		self.fmo_ghosts = ghostmanager.get_fmo_ghostnames()

		self.channel_event_id = 1 # channel label eventbox handler id
		self.ghost_event_id = 1
		self.disable_channel()

		self.local_ghost = self.localmessenger.send_ghost_names()
		self.rebuild_ghostmenu()

		self.timeout_id = None
		self.localghost_timeout_id = gtk.timeout_add(WAIT_NEXT_LOCALGHOST_POLL, self.handle_localghost)

# === Initialization from Preferences === #
		if self.prefs.get(PreferenceManager.PREF_STARTUP_OPEN_LOG):
			self.logwindow.open(None, None)

		if self.prefs.get(PreferenceManager.PREF_STARTUP_AUTO_JOIN):
			self.join(None, None)

		self.notify_preferences_changed(self.prefs.get_all())

# === Setting Selections === #
		for target in ["CLIPBOARD", "PRIMARY"]:
			for type in ["UTF8_STRING", "STRING", "TEXT", "COMPOUND_TEXT"]:
				self.window.selection_add_target(target, type, 0)
		self.window.connect("selection-get", self.selection_handle)
		self.clipboard = ''

# === Finish building window === #
		self.window.show_all()

	def main(self):
		gtk.gdk.threads_init()
		gtk.main()

	def quit(self, data, widget):
		self.part(data, widget)
		gtk.main_quit()

	def open_gviewer(self, widget=None, data=None):
		self.gvwindow.open()
		if self.prefs.get(PreferenceManager.PREF_GVIEWER_STICKY):
			self.gvwindow.stick()
		else:
			self.gvwindow.unstick()

	def open_preference(self, widget=None, data=None):
		PreferenceDialog(self, self.prefs, self.client.luid)

	def get_initial_script(self):
		scr = self.prefs.get(PreferenceManager.PREF_INITIALSCRIPT)
		n = string.find(scr, '|')

		# | found
		if n != -1:
			n -= len(scr)

			# | found at rightmost
			if n != -1:
				scr = scr[:n] + scr[n+1:]
			else:
				scr = scr[:n]

		return [scr, n]

	def notify_log_current_tab_closed(self):
		if self.prefs.get(PreferenceManager.PREF_LOGGING_CURRENT_CLOSED):
			self.logwindow.autologging(self.prefs.get(PreferenceManager.PREF_LOGGING_PATH),
									   self.prefs.get(PreferenceManager.PREF_LOGGING_COMPRESS))

	def notify_preferences_changed(self, prefs):
		# notify BottleClient
		self.client.receive_preferences_changed(prefs)

		# notify LogWindow
		self.logwindow.receive_preferences_changed(prefs)

		# notify GVWindow
		self.gvwindow.receive_preferences_changed(prefs)

		# notify SVGManagerWindow
		self.svgmanager.browser_changed(prefs[PreferenceManager.PREF_BROWSER_COMMAND])

		# notify TextManager
		self.textmanager.receive_preferences_changed(prefs)

	def notify_svg_changed(self):
		self.fmo_ghosts = self.svgmanager.get_fmo_ghostnames()
		self.rebuild_ghostmenu()

	def notify_play_cancel(self, mid):
		return self.localmessenger.cancel_message_as(mid)

	def notify_play_cancel_page_of(self, list):
		self.localmessenger.notify_play_cancel_page_of(list)

	def notify_play_cancel_all(self):
		self.localmessenger.cancel_message_all()
		self.gvwindow.notify_cancel_all()

	def notify_log_update(self, mid, channel, ifghost, script, receive_time):
		self.logwindow.update(mid, channel, ifghost, script, receive_time)

	def dispose_selection(self):
		for target in ["CLIPBOARD", "PRIMARY"]:
			self.window.selection_owner_set(target)

	def selection_handle(self, widget, selection_data, info, time_stamp):
		selection_data.set_text(self.clipboard, -1)

	def copy_clipboard(self, text):
		if not text:
			return
		self.clipboard = text

	def editor_modified(self, widget=None, event=None):
		self.textmanager.coloring(widget, event)
		[si, ei] = self.editor.get_bounds()
		script = self.editor.get_text(si, ei)
		self.sbar.pop(0)
		try:
			self.sbar.push(0, str(len(script.encode('sjis'))) + unicode(_('bytes'), 'utf-8'))
		except:
			self.sbar.push(0, str(len(script)) + unicode(_('bytes'), 'utf-8'))

	def popup(self, textview, menu):
		popup_menu_items = self.get_popup_menu_items()
		popup_item_factory = gtk.ItemFactory(gtk.Menu, "<main>", self.accel_group)
		popup_item_factory.create_items(popup_menu_items)
		popup_menu = popup_item_factory.get_widget("<main>")

		orig_menu = []
		for child in menu.get_children():
			orig_menu.append(child)
			menu.remove(child)

		children = popup_menu.get_children()
		children.reverse()
		for child in children:
			popup_menu.remove(child)
			menu.prepend(child)

		for item in orig_menu:
			menu.get_children()[-1].get_submenu().append(item)

		# check channel permission
		send_item = popup_item_factory.get_item("/%s/%s" % (self.msg["edit"][1], self.msg["send"]))
		if not self.client.connected:
			send_item.set_sensitive(gtk.FALSE)
		else:
			if self.client.channels[self.get_channel_id()]["nopost"]:
				send_item.set_sensitive(gtk.FALSE)
			else:
				send_item.set_sensitive(gtk.TRUE)

	def join(self, widget=None, data=None):
		if not self.client.initialized:
			try:
				self.client.init()
			except BottleClientError, e:
				return
			filename = os.path.join(open_bottlecase(), "luid")
			try:
				open(filename, "w").write(self.client.luid)
			except IOError:
				self.monitor_insert("\n" + unicode(_("Warning"), "utf-8") + \
									": cannot write " + filename)

		if self.prefs.get(PreferenceManager.PREF_AUTO_JOIN_ALL):
			channels = range(len(self.client.channels))
		else:
			d = ChannelDialog(self.window, self.client.channels)
			d.run()
			channels = d.selection
			d.destroy()

		self.pbar.set_fraction(0.0)
		self.monitor_clear()

		if not channels:
			return
		if self.client.join(channels):
			return

		self.pbar.set_fraction(0.0)
		self.monitor_clear()
		self.enable_channel()
		self.timeout_id = gtk.timeout_add(WAIT_NEXT_REQUEST, self.handle_event)

	def part(self, data, widget):
		if not self.client.connected:
			return
		if self.timeout_id:
			gtk.timeout_remove(self.timeout_id)
			self.timeout_id = None
		self.client.close()
		self.monitor_clear()
		self.monitor_insert(unicode(_("Connection closed."), "utf8"))
		self.disable_channel()

		if self.prefs.get(PreferenceManager.PREF_LOGGING_DISCONNECTED):
			self.logwindow.autologging(self.prefs.get(PreferenceManager.PREF_LOGGING_PATH),
									   self.prefs.get(PreferenceManager.PREF_LOGGING_COMPRESS))

	def enable_channel(self):
		channel_item = self.menu_list["/%s" % self.msg["channel"][1]]
		channel_item.deselect()
		channel_item.set_sensitive(gtk.FALSE)
		channel_item.remove_submenu()

		channels = []
		for n in self.client.in_channels:
			name = self.client.channels[n]["name"]
			channels.append(name)
			if not self.channel.get_text():
				self.channel.set_text(name)

		channel_menu = []
		for n in range(len(channels)):
			if n == 0:
				channel_menu.append(("/"+channels[n], None, None, 0, "<RadioItem>"))
			else:
				channel_menu.append(("/"+channels[n], None, None, 0, "/"+channels[n-1]))
		channel_menu = tuple(channel_menu)

		self.channel_factory = gtk.ItemFactory(gtk.Menu, "<main>")
		self.channel_factory.create_items(channel_menu)

		for n in range(len(channels)):
			self.channel_factory.get_item(channel_menu[n][0]).connect("toggled",
																	  self.update_channel,
																	  channels[n])

		channel_item.set_submenu(self.channel_factory.get_widget("<main>"))
		channel_item.set_sensitive(gtk.TRUE)

		if not self.channel_event_box.handler_is_connected(self.channel_event_id):
			self.channel_event_id = self.channel_event_box.connect("button-press-event", self.button_popup,
											   self.channel_factory.get_widget("<main>"))

		self.rebuild_ghostmenu()
		self.menu_list["/%s/%s" % (self.msg["file"][1], self.msg["part"])].set_sensitive(gtk.TRUE)
		self.menu_list["/%s/%s" % (self.msg["file"][1], self.msg["join"])].set_sensitive(gtk.FALSE)
		self.logwindow.join()
		self.show_users()

	def disable_channel(self):
		self.menu_list["/%s/%s" % (self.msg["file"][1], self.msg["part"])].set_sensitive(gtk.FALSE)
		self.menu_list["/%s/%s" % (self.msg["file"][1], self.msg["join"])].set_sensitive(gtk.TRUE)
		self.logwindow.part()

		if self.channel_event_box.handler_is_connected(self.channel_event_id):
			self.channel_event_box.disconnect(self.channel_event_id)
			self.channel_event_id = 1
		if self.ghost_event_box.handler_is_connected(self.ghost_event_id):
			self.ghost_event_box.disconnect(self.ghost_event_id)
			self.ghost_event_id = 1

		self.send_button.set_sensitive(gtk.FALSE)
		self.menu_list["/%s/%s" % (self.msg["edit"][1], self.msg["send"])].set_sensitive(gtk.FALSE)
		self.channel.set_text("")

		self.users.set_text("")

		for m in ["/%s" % self.msg["channel"][1]]:
			menu = self.menu_list[m]
			menu.deselect()
			menu.set_sensitive(gtk.FALSE)
			menu.remove_submenu()

	def update_channel(self, widget=None, data=None):
		self.channel.set_text(data)
		self.show_users_of(self.get_channel_id())
		self.check_channel_permission()

	def close_channel(self, name):
		if self.channel.get_text() == name:
			self.channel.set_text("")
		self.enable_channel()
		self.monitor_clear()
		self.monitor_insert(name + unicode(_('channel was closed.'), 'utf-8'))

	def check_channel_permission(self):
		channel = self.get_channel_id()
		if self.client.channels[channel]["nopost"]:
			self.send_button.set_sensitive(gtk.FALSE)
			self.menu_list["/%s/%s" % (self.msg["edit"][1], self.msg["send"])].set_sensitive(gtk.FALSE)
		else:
			self.send_button.set_sensitive(gtk.TRUE)
			self.menu_list["/%s/%s" % (self.msg["edit"][1], self.msg["send"])].set_sensitive(gtk.TRUE)

	def get_channel_id(self):
		name = self.channel.get_text()
		for id in self.client.in_channels:
			if self.client.channels[id]["name"] == name:
				return id
		raise RuntimeError, "unknown channel name (%s)" % name

	def uniq_ghosts(self):
		ghosts = self.local_ghost + self.fmo_ghosts
		for ghostname in ghosts:
			while ghosts.count(ghostname) - 1:
				ghosts.remove(ghostname)
		return ghosts

	def rebuild_ghostmenu(self):
		ghost_item = self.menu_list["/%s" % self.msg["ghost"][1]]
		ghost_item.deselect()
		ghost_item.set_sensitive(gtk.FALSE)

		if self.ghost_event_box.handler_is_connected(self.ghost_event_id):
			self.ghost_event_box.disconnect(self.ghost_event_id)

		current = self.target_ghost
		ghosts = [self.DEFAULT_TARGET] + self.uniq_ghosts()
		name = ""

		if self.client.connected:
			for n in range(len(self.client.channels)):
				name = self.client.channels[n]["ghost"]
				if name and name not in ghosts:
					ghosts.append(name)

		ghost_menu = []
		ghost_menu.append(("/"+ghosts[0], None, None, 0, "<RadioItem>"))
		for n in range(1, len(ghosts)):
			ghost_menu.append(("/"+ghosts[n], None, None, 0, "/"+ghosts[n-1]))
		ghost_menu = tuple(ghost_menu)

		self.g_ifact = gtk.ItemFactory(gtk.Menu, "<main>")
		self.g_ifact.create_items(ghost_menu)

		all_ghost_items = {}
		for n in range(len(ghosts)):
			all_ghost_items[ghosts[n]] = self.g_ifact.get_item(ghost_menu[n][0])
			all_ghost_items[ghosts[n]].connect("toggled", self.update_ghost, ghosts[n])

		# reselect ghost it's selected by user
		if current in ghosts:
			all_ghost_items[current].set_active(gtk.TRUE)
			all_ghost_items[current].toggled()
		else:
			all_ghost_items[self.DEFAULT_TARGET].set_active(gtk.TRUE)
			all_ghost_items[self.DEFAULT_TARGET].toggled()

		ghost_item.set_submenu(self.g_ifact.get_widget("<main>"))
		ghost_item.set_sensitive(gtk.TRUE)

		if not self.ghost_event_box.handler_is_connected(self.ghost_event_id):
			self.ghost_event_id = self.ghost_event_box.connect("button-press-event",
											   self.button_popup, self.g_ifact.get_widget("<main>"))
		if self.client.connected:
			self.check_channel_permission()

	def update_ghost(self, widget=None, data=None):
		self.target_ghost = data
		self.channel_ghost.set_text("(" + self.target_ghost + ")")

	def handle_localghost(self):
		self.local_ghost = self.localmessenger.send_ghost_names()
		self.rebuild_ghostmenu()
		self.localghost_timeout_id = gtk.timeout_add(WAIT_NEXT_LOCALGHOST_POLL, self.handle_localghost)

	def handle_event(self):
		if not self.client.connected:
			return
		handled = 0
		try:
			handled = self.client.handle_event()
		except BottleClientError, e:
			self.client.close()
			self.disable_channel()
			return

		self.timeout_id = gtk.timeout_add(WAIT_NEXT_REQUEST, self.handle_event)

	def button_popup(self, widget, event, sender):
		if event.button in [1, 3]:
			sender.popup(None, widget, None, event.button, event.time)

	def save_phrases(self, data, widget):
		regexp = re.compile('\r?\n')

		title = regexp.sub('', open_simple_entry_dialog(unicode(_('Please input title'), 'utf-8'),
														unicode(_('Please input title'), 'utf-8'),
														parent=self.window))
		script = regexp.sub('', self.textmanager.get_script())

		if title and script:
			pm = PhraseManager()
			pm.append("%s\t%s" % (title, script))
			pm.save_database()

	def send_question(self, widget=None, data=None):
		self.textmanager.grab_focus()
		if self.check(widget, data):
			return

		d = get_simple_yes_or_no_dialog(unicode(_("Really?"), 'utf-8'),
										unicode(_("Send to "), "utf8") + \
										self.channel.get_text() + " / " + \
										self.target_ghost + unicode(_(", OK?"), "utf8"),
										self.window)
		d.show_all()
		res = d.run()
		d.destroy()

		if not res or res != gtk.RESPONSE_YES:
			return

		channel = self.get_channel_id()
		script = self.get_script()
		ghost = self.target_ghost
		if ghost == self.DEFAULT_TARGET:
			ghost = None

		self.client.send_message(self.prefs.get(PreferenceManager.PREF_AUTO_EDITOR_CLEAR),
								 self.prefs.get(PreferenceManager.PREF_LOGGING_SENT),
								 channel, script, ghost)

	def send_local_message(self, channel, ifghost, script, sender=APP, charset='Shift_JIS', mid=None, list=None):
		if list is None:
			list = self.logwindow.logbook.get_current_loglist()

		if self.prefs.get(PreferenceManager.PREF_FORWARD):
			if self.prefs.get(PreferenceManager.PREF_FORWARD_ACCEPT_ALL) or \
				   (self.prefs.get(PreferenceManager.PREF_FORWARD_ONLY_LISTED) and \
					ifghost in self.local_ghost):
				try:
					self.localmessenger.send_local_message(channel, ifghost, script, sender, charset, mid, list)
				except BottleClientError, e:
					open_error_dialog(str(e), self.window)
					return

		if self.prefs.get(PreferenceManager.PREF_GVIEWER_USE):
			if self.prefs.get(PreferenceManager.PREF_GVIEWER_OPEN_PLAY):
				self.open_gviewer()
			self.gvwindow.enqueue(channel, ifghost, script, sender)

	def log_votes(self, mid, type, num):
		self.logwindow.log_votes(mid, type, num)

	def vote_message(self, mid, type):
		self.client.send_vote_message(self.logwindow, mid, type)

	def test(self, widget=None, data=None):
		self.textmanager.grab_focus()
		if self.check(widget, data):
			return
		self.send_local_message('[' + unicode(_("Test"), "utf-8") + ']',
								self.target_ghost, self.get_script())

	def check(self, widget=None, data=None):
		self.textmanager.grab_focus()
		self.monitor_clear()
		self.monitor_insert(unicode(_("Now checking script..."), "utf8"))
		script = self.get_script()

		if self.scriptchecker.check(script, self.window):
			self.monitor_insert(unicode(_("Error"), "utf8"))
			return 1
		else:
			self.monitor_insert(unicode(_("No error."), "utf8"))
			return 0

	def get_script(self):
		[si, ei] = self.editor.get_bounds()
		script = string.replace(self.editor.get_text(si, ei), "\n", "")

		if string.rstrip(script)[-2:] != r"\e":
			script = script + r"\e"
		return unicode(script, "utf-8")

	def show_users(self):
		channel = self.get_channel_id()
		count = self.client.channels[channel]["count"]
		self.users.set_text("%d / %d " % (count, self.client.users))

	def show_users_of(self, channel):
		if channel == self.get_channel_id():
			count = self.client.channels[channel]["count"]
			self.users.set_text("%d / %d " % (count, self.client.users))

	def monitor_clear(self):
		self.pbar.set_text("")

	def monitor_insert(self, text):
		setted = ""
		if self.pbar.get_text():
			setted = self.pbar.get_text()
		self.pbar.set_text(setted + text)

	def edit_cut(self, widget, data=None):
		self.textmanager.emit("cut-clipboard")

	def edit_copy(self, widget, data=None):
		self.textmanager.emit("copy-clipboard")

	def edit_paste(self, widget, data=None):
		self.textmanager.emit("paste-clipboard")

	def edit_delete(self, widget, data=None):
		self.textmanager.emit("delete-from-cursor", gtk.DELETE_CHARS, 1)

	def edit_clear(self, widget=None, data=None):
		[script, pos] = self.get_initial_script()

		self.edit_clear_all(widget, data)
		self.textmanager.insert_with_color_from(script, pos)
		self.sbar.pop(0)
		self.sbar.push(0, str(len(script.encode('sjis'))) + unicode(_('bytes'), 'utf-8'))

	def edit_clear_all(self, widget=None, data=None):
		self.monitor_clear()
		self.editor.set_text('')
		self.sbar.pop(0)
		self.sbar.push(0, '0' + unicode(_('bytes'), 'utf-8'))

	def about(self, data, widget):
		about_dialog = gtk.Dialog(unicode(_("About"), "utf-8"), self.window,
								  gtk.DIALOG_DESTROY_WITH_PARENT|gtk.DIALOG_MODAL|gtk.DIALOG_NO_SEPARATOR,
								  (gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
		about_dialog.connect("destroy", lambda w: w.destroy)
		about_dialog.set_border_width(10)

		h = gtk.HBox()
		i = gtk.Image()
		i.set_from_stock(gtk.STOCK_DIALOG_INFO, gtk.ICON_SIZE_LARGE_TOOLBAR)
		h.pack_start(i, gtk.FALSE, gtk.TRUE, 0)
		l = gtk.Label(unicode(_("About"), 'utf-8'))
		h.pack_start(l, gtk.FALSE, gtk.TRUE, 0)
		about_dialog.vbox.add(h)
		s = gtk.HSeparator()
		about_dialog.vbox.add(s)

		frame = gtk.Frame(label=" %s %s " % (APP, VER))
		l = gtk.Label("\n"+
					  " Copyright(C) 2001-'04, Tamito KAJIYAMA \n"+
					  " Copyright(C) 2003-'04, Atzm WATANABE \n" +
					  " Copyright(C) 2004,     Takuya KAWAHARA \n")
		l.set_justify(gtk.JUSTIFY_LEFT)
		frame.add(l)
		about_dialog.vbox.add(frame)

		frame = gtk.Frame(label=unicode(_(" License "), "utf-8"))
		l = gtk.Label("\n"+
					  " GNU General Public License version 2 or later \n")
		l.set_justify(gtk.JUSTIFY_RIGHT)
		frame.add(l)
		about_dialog.vbox.add(frame)
		about_dialog.show_all()

		about_dialog.run()
		about_dialog.destroy()

	def insert(self, pre, post=""):
		self.textmanager.emit("insert-at-cursor", pre)
		if post:
			self.textmanager.emit("insert-at-cursor", post)
			self.textmanager.emit("move-cursor", gtk.MOVEMENT_LOGICAL_POSITIONS,
								  -1 * len(post), gtk.FALSE)

	def insert_h(self, data, widget): self.insert("\n\\h")
	def insert_u(self, data, widget): self.insert("\n\\u")
	def insert_s0(self, data, widget): self.insert(r"\s[0]")
	def insert_s1(self, data, widget): self.insert(r"\s[1]")
	def insert_s2(self, data, widget): self.insert(r"\s[2]")
	def insert_s3(self, data, widget): self.insert(r"\s[3]")
	def insert_s4(self, data, widget): self.insert(r"\s[4]")
	def insert_s5(self, data, widget): self.insert(r"\s[5]")
	def insert_s6(self, data, widget): self.insert(r"\s[6]")
	def insert_s7(self, data, widget): self.insert(r"\s[7]")
	def insert_s8(self, data, widget): self.insert(r"\s[8]")
	def insert_s20(self, data, widget): self.insert(r"\s[20]")
	def insert_s25(self, data, widget): self.insert(r"\s[25]")
	def insert_s10(self, data, widget): self.insert(r"\s[10]")
	def insert_s11(self, data, widget): self.insert(r"\s[11]")
	def insert_s19(self, data, widget): self.insert(r"\s[19]")
	def insert_username(self, data, widget): self.insert("%username")
	def insert_selfname(self, data, widget): self.insert("%selfname")
	def insert_selfname2(self, data, widget): self.insert("%selfname2")
	def insert_keroname(self, data, widget): self.insert("%keroname")
	def insert_url(self, data, widget): self.insert(r"\URL[http://", "]")
	def insert_c(self, data, widget): self.insert(r"\c")
	def insert_n(self, data, widget): self.insert("\\n\n")
	def insert_w2(self, data, widget): self.insert(r"\w2")
	def insert_w4(self, data, widget): self.insert(r"\w4")
	def insert_w6(self, data, widget): self.insert(r"\w6")
	def insert_w8(self, data, widget): self.insert(r"\w8")
	def insert_w10(self, data, widget): self.insert(r"\w8\w2")
	def insert_w12(self, data, widget): self.insert(r"\w8\w4")
	def insert_w18(self, data, widget): self.insert(r"\w9\w9")
	def insert_w20(self, data, widget): self.insert(r"\w8\w8\w4")
	def insert_w24(self, data, widget): self.insert(r"\w8\w8\w8")
	def insert_w30(self, data, widget): self.insert(r"\w9\w9\w9\w3")
	def insert_w40(self, data, widget): self.insert(r"\w9\w9\w9\w9\w4")
	def insert_w50(self, data, widget): self.insert(r"\w9\w9\w9\w9\w9\w5")
	def insert_w60(self, data, widget): self.insert(r"\w9\w9\w9\w9\w9\w9\w6")
	def insert_t(self, data, widget): self.insert(r"\t")
	def insert_q(self, data, widget): self.insert(r"\_q")
	def insert_s(self, data, widget): self.insert(r"\_s")
	def insert_e(self, data, widget): self.insert(r"\e")

	def set_self_messages(self):
		self.msg = {
			"file":   [unicode(_("File(_F)"), "utf-8"),
					   unicode(_("File(F)"), "utf-8")],
			"join":   unicode(_("Join"), "utf-8"),
			"part":   unicode(_("Part"), "utf-8"),
			"quit":   unicode(_("Quit"), "utf-8"),
			"edit":   [unicode(_("Edit(_E)"), "utf-8"),
					   unicode(_("Edit(E)"), "utf-8")],

			"check":  unicode(_("Check"), 'utf-8'),
			"test":   unicode(_("Test"), 'utf-8'),
			"send":   unicode(_("Send"), 'utf-8'),

			"cut":	  unicode(_("Cut"), "utf-8"),
			"copy":   unicode(_("Copy"), "utf-8"),
			"paste":  unicode(_("Paste"), "utf-8"),
			"delete": unicode(_("Delete"), "utf-8"),
			"save phrases": unicode(_("Save phrases"), 'utf-8'),
			"clear editor": unicode(_("Clear Editor"), "utf-8"),
			"clear editor all": unicode(_("Clear Editor All"), "utf-8"),
			"pref":   [unicode(_("Preferences(_P)"), "utf-8"),
					   unicode(_("Preferences(P)"), "utf-8")],
			"channel":[unicode(_("Channel(_C)"), "utf-8"),
					   unicode(_("Channel(C)"), "utf-8")],
			"ghost":  [unicode(_("Ghost(_G)"), "utf-8"),
					   unicode(_("Ghost(G)"), "utf-8")],
			"window": [unicode(_("Window(_W)"), "utf-8"),
					   unicode(_("Window(W)"), "utf-8")],
			"logs":    unicode(_("Logs"), "utf-8"),
			"phrases": unicode(_("Phrases"), "utf-8"),
			"gviewer": unicode(_("GViewer"), "utf-8"),
			"svgmanager": unicode(_("SVGManager"), "utf-8"),
			"help":   [unicode(_("Help(_H)"), "utf-8"),
					   unicode(_("Help(H)"), "utf-8")],
			"about":  unicode(_("About"), "utf-8"),
		}
	
		self.pmsg = {
     		"sessions":	    unicode(_("Sessions"), "utf-8"),
     		"timecritical": unicode(_("TimeCritical"), "utf-8"),
     		"quick":		unicode(_("Quick"), "utf-8"),
     		"syncronized":  unicode(_("Syncronized"), "utf-8"),
     
     		"sakura":       unicode(_("Sakura"), "utf-8"),
     		"countenance":  unicode(_("Countenance"), "utf-8"),
     		"ordinary":	    unicode(_("Ordinary"), "utf-8"),
     		"shy":		    unicode(_("Shy"), "utf-8"),
     		"surprised":	unicode(_("Surprised"), "utf-8"),
     		"angst":		unicode(_("Angst"), "utf-8"),
     		"ease off":	    unicode(_("Ease off"), "utf-8"),
     		"smile":		unicode(_("Smile"), "utf-8"),
     		"closed eyes":  unicode(_("Closed eyes"), "utf-8"),
     		"angry":		unicode(_("Angry"), "utf-8"),
     		"scorned":	    unicode(_("Scorned"), "utf-8"),
     		"signboard":	unicode(_("Signboard"), "utf-8"),
     		"singing":	    unicode(_("Singing"), "utf-8"),
     
     		"unyu":		    unicode(_("Unyu"), "utf-8"),
     		"opened eyes":  unicode(_("Opened Eyes"), "utf-8"),
     
     		"wait":		    unicode(_("Wait"), "utf-8"),
     		"sec":		    unicode(_(" sec"), "utf-8"),
     
     		"meta string":  unicode(_("Meta String"), "utf-8"),
     		"user name":	unicode(_("User name"), "utf-8"),
     		"self name":	unicode(_("Self name"), "utf-8"),
     		"true":		    unicode(_("True"), "utf-8"),
     		"false":		unicode(_("False"), "utf-8"),
     		"kero name":	unicode(_("Kero name"), "utf-8"),
     
     		"new line":	    unicode(_("New line"), "utf-8"),
     		"clear":		unicode(_("Clear"), "utf-8"),
     
     		"terminate":	unicode(_("Terminate"), "utf-8"),
     		}

	def get_popup_menu_items(self):     
		popup_menu_items = (
     		( "/%s" % self.pmsg["sessions"], None, None, 0, "<Branch>" ),
     		( "/%s/%s" % (self.pmsg["sessions"], self.pmsg["timecritical"]), None, self.insert_t, 0, None ),
     		( "/%s/%s" % (self.pmsg["sessions"], self.pmsg["quick"]), None, self.insert_q, 0, None ),
     		( "/%s/%s" % (self.pmsg["sessions"], self.pmsg["syncronized"]), None, self.insert_s, 0, None ),
     		( "/sep1", None, None, 0, "<Separator>" ),
     
     		( "/%s" % self.pmsg["sakura"], None, self.insert_h, 0, None ),
     		( "/%s" % self.pmsg["countenance"], None, None, 0, "<Branch>" ),
     		( "/%s/[0]  %s" % (self.pmsg["countenance"], self.pmsg["ordinary"]),
			  "<control><alt>0",self.insert_s0,  0, None, ),
     		( "/%s/[1]  %s" % (self.pmsg["countenance"], self.pmsg["shy"]),
			  "<control><alt>1", self.insert_s1, 0, None, ),
     		( "/%s/[2]  %s" % (self.pmsg["countenance"], self.pmsg["surprised"]),
			  "<control><alt>2", self.insert_s2, 0, None, ),
     		( "/%s/[3]  %s" % (self.pmsg["countenance"], self.pmsg["angst"]),
			  "<control><alt>3", self.insert_s3, 0, None, ),
     		( "/%s/[4]  %s" % (self.pmsg["countenance"], self.pmsg["ease off"]),
			  "<control><alt>4", self.insert_s4, 0, None, ),
     		( "/%s/[5]  %s" % (self.pmsg["countenance"], self.pmsg["smile"]),
			  "<control><alt>5", self.insert_s5, 0, None, ),
     		( "/%s/[6]  %s" % (self.pmsg["countenance"], self.pmsg["closed eyes"]),
			  "<control><alt>6", self.insert_s6, 0, None, ),
     		( "/%s/[7]  %s" % (self.pmsg["countenance"], self.pmsg["angry"]),
			  "<control><alt>7", self.insert_s7, 0, None, ),
     		( "/%s/[8]  %s" % (self.pmsg["countenance"], self.pmsg["scorned"]),
			  "<control><alt>8", self.insert_s8, 0, None, ),
     		( "/%s/[20] %s" % (self.pmsg["countenance"], self.pmsg["signboard"]), None, self.insert_s20, 0, None, ),
     		( "/%s/[25] %s" % (self.pmsg["countenance"], self.pmsg["singing"]), None, self.insert_s25, 0, None, ),
     		( "/sep2", None, None, 0, "<Separator>" ),
     
     		( "/%s" % self.pmsg["unyu"], None, self.insert_u, 0, None ),
     		( "/%s" % self.pmsg["countenance"], None, None, 0, "<Branch>" ),
     		( "/%s/[10] %s" % (self.pmsg["countenance"], self.pmsg["ordinary"]),
			  "<control><alt><shift>g", self.insert_s10, 0, None ),
     		( "/%s/[11] %s" % (self.pmsg["countenance"], self.pmsg["opened eyes"]), None, self.insert_s11, 0, None ),
     		( "/%s/[19] %s" % (self.pmsg["countenance"], self.pmsg["singing"]), None, self.insert_s19, 0, None ),
     		( "/sep3", None, None, 0, "<Separator>" ),
     
     		( "/%s" % self.pmsg["wait"], None, None, 0, "<Branch>" ),
     		( "/%s/0.1%s" % (self.pmsg["wait"], self.pmsg["sec"]), None, self.insert_w2, 0, None ),
     		( "/%s/0.2%s" % (self.pmsg["wait"], self.pmsg["sec"]), None, self.insert_w4, 0, None ),
     		( "/%s/0.3%s" % (self.pmsg["wait"], self.pmsg["sec"]), None, self.insert_w6, 0, None ),
     		( "/%s/0.4%s" % (self.pmsg["wait"], self.pmsg["sec"]), None, self.insert_w8, 0, None ),
     		( "/%s/0.5%s" % (self.pmsg["wait"], self.pmsg["sec"]), None, self.insert_w10, 0, None ),
     		( "/%s/0.6%s" % (self.pmsg["wait"], self.pmsg["sec"]), None, self.insert_w12, 0, None ),
     		( "/%s/0.9%s" % (self.pmsg["wait"], self.pmsg["sec"]), "<control><alt>w", self.insert_w18, 0, None ),
     		( "/%s/1.0%s" % (self.pmsg["wait"], self.pmsg["sec"]), None, self.insert_w20, 0, None ),
     		( "/%s/1.2%s" % (self.pmsg["wait"], self.pmsg["sec"]), None, self.insert_w24, 0, None ),
     		( "/%s/1.5%s" % (self.pmsg["wait"], self.pmsg["sec"]), None, self.insert_w30, 0, None ),
     		( "/%s/2.0%s" % (self.pmsg["wait"], self.pmsg["sec"]), None, self.insert_w40, 0, None ),
     		( "/%s/2.5%s" % (self.pmsg["wait"], self.pmsg["sec"]), None, self.insert_w50, 0, None ),
     		( "/%s/3.0%s" % (self.pmsg["wait"], self.pmsg["sec"]), None, self.insert_w60, 0, None ),
     
     		( "/%s" % self.pmsg["meta string"], None, None, 0, "<Branch>" ),
     		( "/%s/%s" % (self.pmsg["meta string"], self.pmsg["user name"]), None, self.insert_username, 0, None ),
     		( "/%s/%s (%s)" % (self.pmsg["meta string"], self.pmsg["self name"], self.pmsg["true"]),
			  None, self.insert_selfname, 0, None ),
     		( "/%s/%s (%s)" % (self.pmsg["meta string"], self.pmsg["self name"], self.pmsg["false"]),
			  None, self.insert_selfname2, 0, None),
     		( "/%s/%s" % (self.pmsg["meta string"], self.pmsg["kero name"]), None, self.insert_keroname, 0, None ),
     		( "/sep4", None, None, 0, "<Separator>" ),
     
     		( "/%s" % self.pmsg["new line"], "<control><alt>n", self.insert_n, 0, None ),
     		( "/%s" % self.pmsg["clear"], None, self.insert_c, 0, None ),
     		( "/URL", None, self.insert_url, 0, None ),
     
			( "/%s" % self.pmsg["terminate"], "<control><alt>e", self.insert_e, 0, None ),
     		( "/sep5", None, None, 0, "<Separator>" ),
			( "/%s" % self.msg["edit"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["check"]), None, self.check, 0, "<StockItem>", gtk.STOCK_SPELL_CHECK ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["test"]), None, self.test, 0, "<StockItem>", gtk.STOCK_REFRESH ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["send"]), None, self.send_question, 0, "<StockItem>", gtk.STOCK_OK ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["clear editor"]), None, self.edit_clear, 0, "<StockItem>", gtk.STOCK_CLEAR ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["clear editor all"]), None, self.edit_clear_all, 0, "<StockItem>", gtk.STOCK_CLEAR ),
     		( "/%s/sep6" % self.msg["edit"][1], None, None, 0, "<Separator>" ),
			( "/%s/%s..." % (self.msg["edit"][1], self.msg["save phrases"]), None, self.save_phrases, 0, "<StockItem>", gtk.STOCK_SAVE_AS ),
     		( "/%s/sep7" % self.msg["edit"][1], None, None, 0, "<Separator>" ),
     		)
		return popup_menu_items
	
	def get_menu_items(self):
		if gtk.gtk_version[1] >= 4:
			join_item = ( "/%s/%s" % (self.msg["file"][1], self.msg["join"]), "<control>J", self.join, 0, "<StockItem>", gtk.STOCK_NETWORK)
		else:
			join_item = ( "/%s/%s" % (self.msg["file"][1], self.msg["join"]), "<control>J", self.join, 0, None)

		menu_items = (
			( "/%s" % self.msg["file"][0], None, None, 0, "<Branch>" ),
			join_item,
			( "/%s/%s" % (self.msg["file"][1], self.msg["part"]), "<control>P", self.part, 0, "<StockItem>", gtk.STOCK_STOP ),
			( "/%s/sep1" % self.msg["file"][1], None, None, 0, "<Separator>" ),
			( "/%s/%s" % (self.msg["file"][1], self.msg["quit"]), "<control>Q", self.quit, 0, "<StockItem>", gtk.STOCK_QUIT ),

			( "/%s" % self.msg["edit"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["check"]), None, self.check, 0, "<StockItem>", gtk.STOCK_SPELL_CHECK ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["test"]), None, self.test, 0, "<StockItem>", gtk.STOCK_REFRESH ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["send"]), None, self.send_question, 0, "<StockItem>", gtk.STOCK_OK ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["clear editor"]), None, self.edit_clear, 0, "<StockItem>", gtk.STOCK_CLEAR ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["clear editor all"]), None, self.edit_clear_all, 0, "<StockItem>", gtk.STOCK_CLEAR ),
			( "/%s/sep2" % self.msg["edit"][1], None, None, 0, "<Separator>" ),
			( "/%s/%s..." % (self.msg["edit"][1], self.msg["save phrases"]), None, self.save_phrases, 0, "<StockItem>", gtk.STOCK_SAVE_AS ),

			( "/%s/sep3" % self.msg["edit"][1], None, None, 0, "<Separator>" ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["cut"]), "<control>X", self.edit_cut, 0, "<StockItem>", gtk.STOCK_CUT ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["copy"]), "<control>C", self.edit_copy, 0, "<StockItem>", gtk.STOCK_COPY ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["paste"]), "<control>V", self.edit_paste, 0, "<StockItem>", gtk.STOCK_PASTE ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["delete"]), "Delete", self.edit_delete,0, "<StockItem>", gtk.STOCK_DELETE ),
			( "/%s/sep4" % self.msg["edit"][1], None, None, 0, "<Separator>" ),
#			( "/%s/collect" % (self.msg["edit"][1]), None, lambda x, y: gc.collect(), 0, None ),
			( "/%s/%s..." % (self.msg["edit"][1], self.msg["pref"][0]), None, self.open_preference, 0, "<StockItem>", gtk.STOCK_PREFERENCES ),

			( "/%s" % self.msg["channel"][0], None, None, 0, "<Branch>" ),
			( "/%s" % self.msg["ghost"][0], None, None, 0, "<Branch>" ),

			( "/%s" % self.msg["window"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s..." % (self.msg["window"][1], self.msg["logs"]), "<control>L", None, 0, None ),
			( "/%s/%s..." % (self.msg["window"][1], self.msg["phrases"]), None, None, 0, None ),
			( "/%s/%s..." % (self.msg["window"][1], self.msg["gviewer"]), None, None, 0, None ),
			( "/%s/%s..." % (self.msg["window"][1], self.msg["svgmanager"]), None, None, 0, None ),

			( "/%s" % self.msg["help"][0], None, None, 0, "<LastBranch>" ),
			( "/%s/%s..." % (self.msg["help"][1], self.msg["about"]), None, self.about, 0, "<StockItem>", gtk.STOCK_DIALOG_INFO ),
			)
	
		return menu_items


def usage():
	port = unicode(_("alternative port for forwarding SSTP messages"), 'utf-8').encode('euc-jp')
	help = unicode(_("show this message"), 'utf-8').encode('euc-jp')
	host = unicode(_("alternative host for forwarding SSTP messages"), 'utf-8').encode('euc-jp')

	usage = """\
	-p, --port NUM  %s
	--host HOST     %s
	-h, --help      %s\n""" % (port, host, help)

	sys.stderr.write(unicode(_("Usage"), 'utf-8').encode('euc-jp') + ": " + \
					 AGENT_NAME + "\n" + \
					 unicode(_("Options"), 'utf-8').encode('euc-jp') + ":\n" + usage)
	sys.exit(1)

def main():
	# parse command line arguments
	try:
		options, rest = getopt.getopt(sys.argv[1:], "p:h",
									  ["port=", "help", "host="])
	except getopt.error, e:
		sys.stderr.write(unicode(_("Error"), 'utf-8').encode('euc-jp') + ": %s\n" % str(e))
		usage()
	if rest:
		usage()

	port = 9801
	host = "localhost" # default - localhost:9801

	# check environment variables
	val = os.environ.get("BOTTLER_PORT")
	if val:
		try:
			num = int(val)
		except ValueError:
			sys.stderr.write(unicode(_("Invalid BOTTLER_PORT number (ignored)") + \
									 "\n", 'utf-8').encode('euc-jp'))
		else:
			port = num

	# parse command line options
	for opt, val in options:
		if opt in ["-p", "--port"]:
			port = int(val)
		elif opt == "--host":
			host = val
		else:
			usage()
	# run
	app = Application(host, port)
	app.main()

if __name__ == "__main__":
	main()

