<?php
/*
 *metaフォルダへのアクセサ
*/

require_once (dirname(__FILE__) . '/../config.php');
require_once 'const.php';
require_once 'compatible.php';
require_once 'filehashing.php';

if(readConfig("USEIMGFINDER")){
  require_once 'imgfinder.php';
}

//-----------------------------------------
// DB関連の定数

define('USEFULLCACHE', 1); //一覧用キャッシュ利用フラグ
define('FULLCACHENAME', 'allinfo.cache'); //一覧用キャッシュ名

define('VERSIONFILE', 'version.dat');     //データバージョンファイル
define('CURRENTVERSION', '00.06.27' );    //現在のデータバージョン
define('HTMLMIGRATE', 0);                 //マイグレーション時にhtmlも作り直す
define('DBACONFNAME', 'dbaconf.dat');     //DBA設定情報

define('USEQDBM', 1);                     //QDBMを利用する場合 1
define('METAQDBMFILE', 'allinfo.qdb');
define('METASQLITEFILE', 'allinfo.sqlite');
define('METAFILEHASH', 'filehash.sqlite');


require_once 'dba/textfinder.php';
require_once 'dba/DBA_qdbm.php';
require_once 'dba/DBA_sqlite3.php';
require_once 'dba/DBA_chunkfile.php';

//------------------------------------------
//DB設定の読み取りと初期設定
//
function _readDBAConfig()
{
  $confpath = readConfig("METADIR") . DBACONFNAME;
  
  if(is_file($confpath)){
    return unserialize(file_get_contents($confpath));
  }else{
    $conf = array(
      "kvstype" => "chunkfile",
      "msgpack" => false
    );
    if( extension_loaded('msgpack') ){
      $conf["msgpack"] = true;
    }
    /*
    if(is_file(readConfig("METADIR") . METAQDBMFILE)){
      $conf["kvstype"] = "qdbm";
    }else if(is_file(readConfig("METADIR") . METASQLITEFILE)){
      $conf["kvstype"] = "sqlite3";
    }else if( USEQDBM && enable_qdbm() ){
      $conf["kvstype"] = "qdbm";
    }else if( enable_sqlite3() ){
      $conf["kvstype"] = "sqlite3";
    }
    //*/
    
    file_put_contents( $confpath, serialize($conf) );
    return $conf;
  }
}
$DBACONF = _readDBAConfig();

function readDBAConfig($key,$default=false)
{
  global $DBACONF;
  if(array_key_exists($key,$DBACONF)){
    return $DBACONF[$key];
  }else{
    return $default;
  }
}

//------------------------------------------
//DBAハンドラ生成

function makeDBA($mode)
{
  $kvstype = readDBAConfig("kvstype");
  if($kvstype == "qdbm"){
    return new DBA_qdbm($mode, readConfig("METADIR") . METAQDBMFILE);
  }else if($kvstype == "sqlite3"){
    return new DBA_sqlite3($mode, readConfig("METADIR") . METASQLITEFILE);
  }else{
    return new DBA_Chunkfile();
  }
}

//------------------------------------------
//シリアライザ(dbaでも共用)
function ftb_pack($value)
{
  if(readDBAConfig("msgpack")){
    return msgpack_pack($value);
  }else{
    return serialize($value);
  }
}

function ftb_unpack($value)
{
  if(readDBAConfig("msgpack")){
    return msgpack_unpack($value);
  }else{
    return unserialize($value);
  }
}

//------------------------------------------
//メタデータバージョンが古い場合
//一括でマイグレーション
$verpath = readConfig("METADIR") . VERSIONFILE;
if(!is_file($verpath)
 || strcmp( file_get_contents($verpath), CURRENTVERSION ) < 0)
{
  removeJsonApiCache();

  $dba = makeDBA('w');
  foreach($dba->allKeys() as $key){
    $info = $dba->get($key);

    if(HTMLMIGRATE){
      $basenames   = urlToBasename($key);
      $indexgzpath = readConfig("CONTDIR") . $basenames["CONT"] . _PGET_OTHER_DIR_ . _PGET_INDEX_PAGE_SRCGZ_;
      //$indexpath = readConfig("CONTDIR") . $basenames["CONT"] . _PGET_INDEX_PAGE_;

      $html = "";
      // 加工前のhtml(gz版)を優先的に読み込む
      if( is_file($indexgzpath)  )
      {
        //htmlの再変換
        $dldir = readConfig("CONTDIR") . $basenames["CONT"];
        $html = cust_pget($key,$dldir,'pgetDmyGetImageList','pgetDmyGetImage',true);
      }
    }

    $dba->set($key, meta_verifyInfo($info));
  }
  $dba->optimize();
  $dba->close();

  file_put_contents( $verpath, CURRENTVERSION );
}

//------------------------------------------

/**
 *urlで指定したinfoを読み出す
 *読み出せなかった場合はfalse
 */
function readUrlInfo($url)
{
  $dba = makeDBA('r');
  $info = $dba->get($url);
  $dba->close();

  return $info;
}

/**
 *infoを保存する
 */
function saveInfo($info)
{
  $url = $info['url'];
  $dba = makeDBA('w');
  if(!$dba->exists($url)){
    removeJsonApiCache();
  }
  $dba->set($url,meta_verifyInfo($info));
  $dba->close();
}

/**
 *全てのinfoを読み出し
 *更新順にソートして返す
 */
function readAllInfo()
{
  $dba = makeDBA('r');
  $infos = $dba->allValues();
  $dba->close();

  return $infos;
}

/**
 * キャッシュ付きreadAllInfo
 * 一覧画面などで高速化目的で利用する
 * スレ追加,削除,タグ追加の時にしか更新されないため
 * date,life_time などが重要な場合は利用できない
 */
function readAllInfoCached()
{
  $allinfo = array();
  
  if(USEFULLCACHE){
    $lockpath = readConfig("LOCKDIR") . FULLCACHENAME . ".lock";
    if($lock = fopen($lockpath, 'w')){
      if (flock($lock, LOCK_EX)){
        $cachepath = readConfig("METADIR") . FULLCACHENAME;
        if(is_file($cachepath)){
          $data = file_get_contents($cachepath);
          $allinfo = ftb_unpack($data);
        }else{
          $allinfo = readAllInfo();
          file_put_contents($cachepath, ftb_pack($allinfo) );
        }
        flock($lock, LOCK_UN);
      }
      fclose($lock);
    }
  }else{
    $allinfo = readAllInfo();
  }

  return $allinfo;
}

/**
 *新旧のinfoを統合する
 */
function mergeInfo($info, $oldInfo)
{
  //古いinfoの内容を引き継ぐ
  if($oldInfo){
    $info['first_date'] = $oldInfo['first_date'];
    if($info['user_update'] == 0) {
      $info['user_update'] = $oldInfo['user_update'];
    }

    $info['user_ip']    = $oldInfo['user_ip'];
    $info['user_host']  = $oldInfo['user_host'];
    $info['user_regid'] = $oldInfo['user_regid'];
    $info['tags']       = get_value($oldInfo,'tags', array());
    $info['alluserid']  = get_value($oldInfo,'alluserid', array());
    $info['samefiles']  = array_merge(
                            get_value($info,'samefiles', array()),
                            get_value($oldInfo,'samefiles', array())
                          );
  }
  return $info;
}

/**
 * 読み込んだデータに抜けているカラムがあった場合
 * 初期データを挿入する
**/
function meta_verifyInfo($src)
{
  global $ROOTDIR;
  $newinfo = newInfo();
  $keys = array_keys($newinfo);

  foreach($keys as $key) {
    if(array_key_exists($key, $src)) {
      $newinfo[$key] = $src[$key];
    }
  }
  if($newinfo["first_date"] == 0){
    $newinfo["first_date"] = $newinfo["date"];
  }
  if($newinfo["user_update"] == 0){
    $newinfo["user_update"] = $newinfo["date"];
  }
  //tagの構造変更対応
  $tags = array();
  foreach($newinfo["tags"] as $tag) {
    if(!is_array($tag)) {
      $tags[] = array($tag,"");
    }else{
      $tags[] = $tag;
    }
  }
  $newinfo["tags"] = $tags;

  $basenames = urlToBasename($newinfo["url"]);

  //画像の存在チェック
  $newinfo["has_thumb"]   = is_file($ROOTDIR . $newinfo["thumb"]);
  $newinfo["has_catalog"] = is_file(readConfig("CONTDIR") . $basenames["CONT"] . _PGET_CATALOG_);
  $newinfo["catalog_path"] = readConfig("CONTURL") . $basenames["CONT"] . _PGET_CATALOG_;

  //other_filesがない場合再計算する
  if(!$newinfo["other_files"]) {

    $otherdir  = readConfig("CONTDIR") . $basenames["CONT"] . _PGET_OTHER_DIR_;
    $imgdir    = readConfig("CONTDIR") . $basenames["CONT"] . _PGET_IMG_DIR_;
    $other_files = array();
    $sys_files = array();
    if(is_dir($otherdir)) {
      $other_files = find_files($otherdir);
      $sys_files = array_filter($other_files, 'is_sysfiles');
    }
    $img_files = array();
    if(is_dir($imgdir)) {
      $img_files = find_files($imgdir);
    }
    $newinfo["other_files"]  = count($other_files) + count($img_files) - count($sys_files);

  }

  //lostをbooleanから回数に変更
  if($newinfo["lost"] === true){
    $newinfo["lost"] = 3;
  }
  
  //filehashの計算
  if(get_value($newinfo, "htmlhash", "") == ""){
    $gzfilename = readConfig("CONTDIR") . $basenames["CONT"] . (_PGET_OTHER_DIR_._PGET_INDEX_PAGE_SRCGZ_);
    if(is_file($gzfilename)){
      $newinfo["htmlhash"] = sha1_file($gzfilename);
    }
  }

  return $newinfo;
}

/**
 * 新規のinfoオブジェクトを作成
 */
function newInfo()
{
  return array(
    "date"      => 0,
    "title"     => "",
    "url"       => "",
    "store_url" => "",
    "thumb"     => "",
    "res"       => "",
    "lost"      => 0,
    "ng"        => false,
    "du"        => 0,
    "res_count" => 0,
    "first_date" => 0,
    "user_update" => 0,
    "user_ip"    => "",
    "user_host"  => "",
    "user_regid" => "",
    "tags" => array(),
    "samefiles" => array(),
    "contacts" => array(),
    "src1res_day" => "",
    "src1res_time" => "",
    "other_files" => false,
    "alluserid" => array(),
    "has_thumb" => false,
    "has_catalog" => false,
    "catalog_path" => "",
    "htmlhash" => "",
    );
}

function remove_directory($dir) {
  if ($handle = opendir("$dir")) {
   while (false !== ($item = readdir($handle))) {
     if ($item != "." && $item != "..") {
       if (is_dir("$dir/$item")) {
         remove_directory("$dir/$item");
       } else {
         unlink("$dir/$item");
       }
     }
   }
   closedir($handle);
   rmdir($dir);
  }
}

/**
 *urlで指定したinfoと関連する情報を削除する
 */
function removePage($url)
{
  $basenames = urlToBasename($url);

  $lockpath = readConfig("LOCKDIR") . $basenames["LOCK"];
  if($lock = fopen($lockpath, 'w')){
    if (flock($lock, LOCK_EX)){
      //付随ファイルの削除
      if($info = readUrlInfo($url)) {
        foreach( $info["samefiles"] as $target ) {
          if(file_exists($target)) { unlink($target);}
        }

        //全文検索エンジンからの削除
        if(readConfig("USESEARCHSV")) {
          $sv = makeFullTextDB();
          $sv->Del($info['url']);
          $sv->close();
        }

        //画像検索エンジンからの削除
        if(readConfig("USEIMGFINDER") ) {
          imgfinder_remove($url);
        }
        //ファイルハッシュを削除
        filehash_remove($url);
      }

      if(!readConfig("CONTDIRREADONLY")){
        $target = readConfig("CONTDIR") . $basenames["CONT"];
        if(strlen($basenames["CONT"]) > 0
           && file_exists($target))
        {
          remove_directory($target);
        }        
      }

      $target = readConfig("ZIPDIR") . $basenames["ZIP"];
      if(file_exists($target)){unlink($target);}

      $target = readConfig("ZIPDIR") . $basenames["MHT"];
      if(file_exists($target)){unlink($target);}

      $dba = makeDBA('w');
      $dba->del($url);
      $dba->close();
    }
    flock($lock, LOCK_UN);
    fclose($lock);
  }
  unlink($lockpath);

  //json_apiのキャッシュを更新
  removeJsonApiCache();
}

//----------------------------------------------------------
// データの最適化
function meta_optimize()
{
  $dba = makeDBA('w');
  $dba->optimize();
  if(readConfig("BAYESIANTAG")){
    require_once 'analyze/naivebayes_cache.php';
    $urls = $dba->allKeys();
    $bayes = new NaiveBayesCache();
    $bayes->optimize($urls);
    $bayes->close();
  }
  $dba->close();
  if(readConfig("USESEARCHSV") && mt_rand(1,20) == 1) {
    $sv = makeFullTextDB();
    $sv->optimize();
    $sv->close();
  }
  if(readConfig("USEFILEHASHING") && mt_rand(1,20) == 1) {
    $dbpath    = readConfig("METADIR") . METAFILEHASH;
    $sv = new FileHashDB($dbpath);
    $sv->optimize();
    $sv->close();
  }
}


//----------------------------------------------------------
//連絡機能関連

function _contact_host($contact){
  return $contact[1];
}

//指定スレに連絡を追加
function addcontact($url, $reason) {

  if($info = readUrlInfo($url)) {
    $host = gethostbyaddr(getClientIp());
    //同一ホストからの削除要望は受け付けない
    if(!in_array($host, array_map('_contact_host', $info["contacts"]))){
      $contacts = $info['contacts'];
      $contacts[] = array($reason,$host);
      $info['contacts'] = $contacts;

      saveInfo($info);
    }
  }
}

//----------------------------------------------------------
//ディスク使用量

function getDiskUsageAll($infos = false)
{
  $diskUsageAll = loadJsonStore("du");
  $tick = saveJsonStoreTick("du");
  if( !$diskUsageAll || ($tick == -1 || $tick > 300) && mt_rand(1,20) == 1 )
  {
    if (!$infos) {
      $infos = readAllInfoCached();
    }
    //calc du
    //calc disk usage
    $du = 0;
    foreach ($infos as $meta){
      if(array_key_exists ("du",$meta)) {
        $du += $meta["du"];

        $basenames = urlToBasename($meta["url"]);
        //zipsize
        $mhtpath = readConfig("ZIPDIR") . $basenames["MHT"];  //mhtのサイズ
        if(file_exists($mhtpath)) {
          $du += filesize($mhtpath);
        }
        $zippath = readConfig("ZIPDIR") . $basenames["ZIP"];  //zipのサイズ
        if(file_exists($zippath)) {
          $du += filesize($zippath);
        }
      }
    }
    $diskUsageAll = array("du" => $du);
    saveJsonStore($diskUsageAll,"du");
  }

  return $diskUsageAll["du"];
}


//----------------------------------------------------------
// 生存時間の再計算
function getLifetime($info)
{
  $basenames = urlToBasename($info["url"]);

  //残り時間
  $lifetime = 0;
  if(readConfig("LIFETIME") > 0) {
    if(readConfig("LIFETIME_TYPE") == 0){
      $lifetime = readConfig("LIFETIME")-(time()-$info["date"]);
    }else if(readConfig("LIFETIME_TYPE") == 1){
      $lifetime = readConfig("LIFETIME")-(time()-$info["user_update"]);
    }else{
      $lifetime = readConfig("LIFETIME")-(time()-$info["first_date"]);
    }
  }

  return $lifetime;
}


//----------------------------------------------------------
function _tag_value($tag){
  return $tag[0];
}

//指定スレにタグの追加
function addtag($url, $tag, $host=FALSE) {
  $tags = array();

  $result = false;
  if($info = readUrlInfo($url)) {
    $tags = $info['tags'];
    if($tag !== "" && !in_array($tag,array_map("_tag_value", $tags)) ){
      if($host === FALSE){
        $host = gethostbyaddr(getClientIp());
      }

      $tags[] = array($tag,$host);
      $info['tags'] = $tags;

      saveInfo($info);
      $result = true;
      removeJsonApiCache();
    }
  }
  return $result;
}

//タグの削除
//登録元と同じホストの場合のみタグ削除を行う
function deltag_owner($url, $tag) {
  if($meta = readUrlInfo($url)){
    $pos = array_search( array( $tag, gethostbyaddr(getClientIp())), $meta['tags'] );
    if($pos !== FALSE){
      array_splice( $meta['tags'], $pos, 1);

      saveInfo($meta);
      removeJsonApiCache();
    }
  }
}

//全体タグリストの読み込み
function readalltag( $infos = false, $withCount = false, $forceRecalc = false)
{
  $tags = loadJsonStore("tags");
  $tick = saveJsonStoreTick("tags");
  if( !$tags || $forceRecalc
   || ($tick == -1 || $tick > 300) && mt_rand(1,20) == 1 )
  {
    if (!$infos) {
      $infos = readAllInfoCached();
    }
    $tags = _readalltag_recalc($infos);
    saveJsonStore($tags,"tags");
  }

  if($withCount == false) {
    $tags = array_map("_tag_name", $tags);
  }

  return $tags;
}

function _tag_name($item) {
  return $item['name'];
}

//全体タグリストの再読み込み
function _readalltag_recalc( $infos )
{
  $tags = array();
  foreach($infos as $info) {
    foreach($info['tags'] as $item){
      $tags[] = $item[0];
    }
  }
  $kvs = array();
  foreach($tags as $key) {
    $kvs[$key] = get_value($kvs,$key, 0) + 1;
  }
  $tags = array();
  $countlist = array();
  foreach($kvs as $key => $value) {
    $tags[]    = array("name" => $key, "count" => $value);
    $countlist[] = $value;
  }
  array_multisort($countlist, SORT_DESC, $tags);

  return $tags;
}

//--------------------------------------------
// json形式を使った簡易データストア
//

function _jsonStoreCachePath($name, $orign=""){
  global $ROOTDIR;
  if(strlen($orign)>0){
    if(!preg_match("/\/$/",$orign)){
      $orign .= "/";
    }
    mkdir_r($ROOTDIR . "json/". $orign );
  }
  return $ROOTDIR . "json/". $orign . $name . ".json";
}

function _jsonStoreLockPath($name, $orign=""){
  $lockpath  = readConfig("LOCKDIR") . $name . ".lock" ;
  if(strlen($orign)>0){
    $lockpath  = readConfig("LOCKDIR") . rtrim($orign,"/") . ".lock" ;
  }
  return $lockpath ;
}

function saveJsonStoreRaw($data, $name, $orign="")
{
  $cachepath = _jsonStoreCachePath($name, $orign);

  file_put_contents($cachepath, $data);
}

function saveJsonStore($data, $name, $orign="")
{
  $lockpath  = _jsonStoreLockPath($name, $orign);

  if($lock = fopen($lockpath, 'w')){
    if (flock($lock, LOCK_EX)){
      saveJsonStoreRaw(json_encode($data), $name, $orign);

      flock($lock, LOCK_UN);
    }
    fclose($lock);
  }
}

function loadJsonStoreRaw($name, $orign="")
{
  $cachepath = _jsonStoreCachePath($name, $orign);
  if( !is_file ($cachepath) ){
    return false;
  }else{
    return file_get_contents($cachepath);
  }
}

function loadJsonStore($name, $orign="")
{
  $result = loadJsonStoreRaw($name, $orign);
  if($result) {
    return json_decode($result,true);
  }
  return false;
}

function saveJsonStoreTick($name, $orign="")
{
  $cachepath = _jsonStoreCachePath($name, $orign);
  if( !is_file ($cachepath) ){
    return -1;
  }else{
    return time() - @filemtime($cachepath);
  }
}

function removeJsonStore($name, $orign="")
{
  $cachepath = _jsonStoreCachePath($name, $orign);
  $lockpath  = _jsonStoreLockPath($name, $orign);
  if(is_file($cachepath)) unlink($cachepath);
  if(is_file($lockpath)) unlink($lockpath);
}

function removeJsonApiCache(){
  //各種json_apiのキャッシュ削除
  removeJsonStore('readallinfo');
  removeJsonStore('du');

  if(USEFULLCACHE){
    $lockpath = readConfig("LOCKDIR") . FULLCACHENAME . ".lock";
    if($lock = fopen($lockpath, 'w')){
      if (flock($lock, LOCK_EX)){
        $cachepath = readConfig("METADIR") . FULLCACHENAME;
        if(is_file($cachepath)) unlink($cachepath);
        flock($lock, LOCK_UN);
      }
      fclose($lock);
    }
  }
}
