<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.pages.scaffold.organizer
 */
/**
 * @file Smarty.php
 * @brief Smarty.
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: Smarty.php 30 2007-08-09 03:57:28Z ishitoya $
 *
 */

require_once("ficus/lang/Assert.php");

/**
 * @class Ficus_Smarty
 */
class Ficus_Smarty{
    const TEMPLATE_SUFFIX = ".tpl";

    /**
     * smarty
     */
    protected $smarty = null;

    /**
     * system paths
     */
    protected $paths = array();

    /**
     * user paths
     */
    protected $userPaths = array();

    /**
     * constructor
     * @param $templateDir string template Dir
     * @param $cacheDir string cache Dir
     */
    public function __construct($cacheDir = null){
        $this->smarty = Ficus_SmartyComponentFactory::getSmarty();
        $this->setCacheDir($cacheDir);
    }

    /**
     * set template dir
     * default load path for smarty include
     */
    public function setTemplateDir($dir){
        $this->smarty->template_dir = $dir;
    }
    
    /**
     * add path
     */
    public function addPath($name, $path){
        if(is_dir($path) == false){
            throw new Ficus_IllegalArgumentException("$path is not readable");
        }
        $this->paths[$name] = new Ficus_ClassPathElement($path);
    }

    /**
     * add user dir
     */
    public function addUserPath($name, $path){
        if(is_dir($path) == false){
            throw new Ficus_IllegalArgumentException("$path is not a directory");
        }
        $this->userPaths[$name] = new Ficus_ClassPathElement($path);
    }

    /**
     * set cache dir
     */
    public function setCacheDir($cacheDir){
        $this->smarty->compile_dir = $cacheDir;
    }

    /**
     * set id
     */
    public function setCompileId($id){
        $this->smarty->compile_id = md5($id);
    }
    
    /**
     * search for template
     */
    protected function search($template){
        $template = new Ficus_File($template);
        if(is_null($template->getExtension())){
            $template->add(self::TEMPLATE_SUFFIX);
        }
        $template = $template->normalize();
        $template = $template->getPath();
        $paths = array();
        foreach($this->userPaths as $path){
            $paths = array_merge($path->search($template), $paths);
        }
        if(empty($paths)){
            foreach($this->paths as $path){
                $paths = array_merge($path->search($template), $paths);
            }
        }
        if(empty($paths)){
            throw new Ficus_FileNotFoundException("template $template is not found in " . implode(",", $this->paths) . "," . implode(",", $this->userPaths));
        }
        if(count($paths) >= 2){
            throw new Ficus_MultipleFileFoundException("multiple template named $template found in " . explode(",", $paths));
        }

        $path = $paths[0];
        $this->assign("smarty_AbsolutePath", dirname($path));
        $this->setCompileId($path);
        return "file:" . $path;
    }

    /**
     * is template exists
     */
    public function isTemplateExists($template){
        try{
            $this->search($template);
        }catch(Exception $e){
            return false;
        }
        return true;
    }
    
    /**
     * assign
     * @param $name name of variable
     * @param $var mixed variable
     */
    public function assign($name, $var){
        $this->smarty->assign($name, $var);
    }

    /**
     * assign by ref
     * @param $name name of variable
     * @param $var mixed variable
     */
    public function assignByReference($name, $var){
        $this->smarty->assign_by_ref($name, $var);
    }
    
    /**
     * fetch
     * @param $template string template
     * @return string result
     */
    public function fetch($template){
        return $this->smarty->fetch($this->search($template));
    }

    /**
     * display
     * @param $template string template
     */
    public function display($template){
        $this->smarty->display($this->search($template));
    }

    /**
     * clear
     * @param $name string name of property
     */
    public function clear($name){
        $this->smarty->clear_assign($name);
    }

    /**
     * get template vars
     */
    public function getAssigned($name = null){
        if(is_null($name)){
            return $this->smarty->get_template_vars();
        }else{
            return $this->smarty->get_template_vars($name);
        }
    }
}
?>
