<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.pages.scaffold.util
 */
/**
 * @file ScaffoldMediator.php
 * @brief ScaffoldMediator.
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: ScaffoldMediator.php 51 2007-09-24 13:39:45Z ishitoya $
 *
 */

require_once("ficus/lang/Assert.php");

/**
 * @class Ficus_ScaffoldMediator
 */
class Ficus_ScaffoldMediator implements Ficus_ScaffoldConstants{
    /**
     * target entity
     */
    protected $target = null;

    /**
     * configuration
     */
    protected $configuration = null;

    /**
     * s2daoManager
     */
    protected $daoManager = null;

    /**
     * entity
     */
    protected $entity = null;

    /**
     * request
     */
    protected $request = null;
    
    /**
     * smarty
     */
    protected $smarty = null;

    /**
     * scaffold bean
     */
    protected $scaffold = null;

    /**
     * dto
     */
    protected $condition = null;

    /**
     * construct
     * @param $target string target entity name
     */
    public function __construct(){
        $settings = Ficus_Registry::search(self::REGISTRY_SETTINGS);
        $this->configuration = new Ficus_ScaffoldConfiguration($settings);
        $this->daoManager = new Ficus_S2DaoManager();
        $this->request = new Ficus_PageRequest();
        $this->smarty = Ficus_ScaffoldComponentFactory::getSmarty();

        $templates =
            Ficus_Dir::normalize(Ficus_File::currentDir() . "/../templates");
        $this->smarty->addPath(self::SCAFFOLD_TEMPLATES, $templates);
        if(Ficus_Registry::search(self::REGISTRY_SCAFFOLD_TEMPLATES)){
            $dir =
                Ficus_Registry::search(self::REGISTRY_SCAFFOLD_TEMPLATES);
            $templates = Ficus_ClassPath::search($dir);
            if(empty($templates)){
                throw new Ficus_IllegalArgumentException("directory $dir is not found in class path");
            }
            $templates = $templates[0];
            $this->smarty->addUserPath(self::SCAFFOLD_TEMPLATES, $templates);
        }
    }

    /**
     * set target
     * @param $target string target
     */
    public function setTarget($target){
        $this->target = $target;
        $this->daoManager->setTarget($target);
    }

    /**
     * set entity
     */
    public function setEntity($entity){
        $this->entity = $entity;
    }
    
    /**
     * set scaffold bean
     */
    public function setScaffold($bean){
        $this->scaffold = $bean;
    }

    /**
     * set condition
     */
    public function setCondition($condition){
        $this->condition = $condition;
    }

    /**
     * get scaffold bean
     */
    public function scaffold(){
        return $this->scaffold;
    }
    
    /**
     * set request
     */
    public function setRequest($request){
        $this->request = $request;
    }

    /**
     * get request
     */
    public function request(){
        return $this->request;
    }

    /**
     * get entity
     */
    public function entity(){
        if(is_null($this->entity)){
            return $this->getEntityFromRequest();
        }
        return $this->entity;
    }

    /**
     * get condition
     */
    public function condition(){
        if(is_null($this->condition)){
            return $this->daoManager()->getPagerDto();
        }
        return $this->condition;
    }
    
    /**
     * get smarty
     */
    public function smarty(){
        return $this->smarty;
    }
    
    /**
     * configuration
     * @return Ficus_ScaffoldConfiguration
     */
    public function configuration(){
        $this->isInitialized();
        return $this->configuration;
    }

    /**
     * table
     */
    public function table($name = null){
        if(is_null($name)){
            $name = $this->target;
        }
        return $this->configuration->table($name);
    }

    /**
     * dao managert
     * @return Ficus_S2DaoManager
     */ 
    public function daoManager(){
        $this->isInitialized();
        return $this->daoManager;
    }

    /**
     * get Container
     */
    public function getScaffold($entity, $context){
        Ficus_ScaffoldEntity::setMediator($this);
        
        $config = $this->table();
        $config->setContext($context);
        $this->scaffold->setConfig($config);
        $builder = new Ficus_ConcreteScaffoldBuilder($this);
        $builder->build($entity);
        return $this->scaffold;
    }

    /**
     * deserializeEntity
     */
    public function getEntityFromRequest(){
        $entity = $this->daoManager->deserializeEntity(
                                                 $this->request->requests());
        $skips = $this->request->search('/skip_/');
        foreach($skips as $key => $value){
            if($value){
                $key = str_replace("__", ".", $key);
                preg_match('/skip_(.*?)$/', $key, $regs);
                $key = $regs[1];
                $result = $entity->search($key);
                $result->dispose();
            }
        }
        return $entity;
    }

    /**
     * check is initialized
     */
    protected final function isInitialized(){
        if(is_null($this->target)){
            throw new Ficus_NotReadyException("target must be initialized before runch any public functions");
        }
        return;
    }
}
?>