<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.pages.scaffold
 */
/**
 * @file ScaffoldManager.php
 * @brief index page of administrator site
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: ScaffoldManager.php 51 2007-09-24 13:39:45Z ishitoya $
 */

require_once("ficus/config/Registry.php");

/**
 * @class Ficus_ScaffoldManager
 */
class Ficus_ScaffoldManager extends Ficus_Bean
implements Ficus_ScaffoldConstants{
    /**
     * foreperson
     */
    protected $foreperson = null;

    /**
     * target
     */
    protected $target = null;
    
    /**
     * page
     */
    protected $page = null;

    /**
     * action
     */
    protected $action = null;

    /**
     * mediator
     */
    protected $mediator = null;

    /**
     * fuzzy
     */
    protected $fuzzy = false;

    /**
     * construct
     */
    public function __construct($page){
        $this->mediator = new Ficus_ScaffoldMediator();
        $this->guess($page);
    }

    /**
     * is guess able
     * @return boolean true if guessable
     */
    public function isGuessable(){
        if($this->page instanceof Ficus_ScaffoldPage &&
           $this->page->isEmptyTarget() == false){
            return true;
        }
            
        return $this->page->request()->has(self::KEY_TARGET);
    }
    
    /**
     * guess action
     * @param $args array of request
     */
    public function guess($page){
        $this->setPage($page);
        $request = $page->request();

        $scaffold = new Ficus_Scaffold();
        $scaffold->setAction($page->nextAction());
        $scaffold->setPage($page->pagename());
        $this->setScaffoldFromRequest($scaffold, "target", self::KEY_TARGET);
        $this->setScaffoldFromRequest($scaffold, "do",     self::KEY_DO);
        $this->setScaffoldFromRequest($scaffold, "method", self::KEY_METHOD);
        $this->setScaffoldFromRequest($scaffold, "message", self::KEY_MESSAGE);
        $this->setScaffoldFromRequest($scaffold, "transition",
                                      self::KEY_TRANSITION);
        $this->setScaffoldFromRequest($scaffold, "submitTitle",
                                  self::KEY_SUBMIT_TITLE);
        $this->setScaffold($scaffold);

        if($this->isGuessable()){
            if($page instanceof Ficus_ScaffoldPage){
                $target = $page->target();
                $foreperson = $page->foreperson();
            }else{
                $target = $request->extractAsString(self::KEY_TARGET);
                $foreperson = $request->extractAsString(self::KEY_FOREPERSON);
            }
            $this->setTarget($target);
            $this->setAction($foreperson);
        }
    }

    /**
     * set bean data from request
     */
    private function setScaffoldFromRequest($scaffold, $property, $key){
        $value = $this->page->request()->extractAsString($key);
        if(is_null($value) == false){
            $scaffold->set($property, $value);
        }
    }                       

    /**
     * set request
     */
    public function setPage($page){
        $this->page = $page;
    }
    
    /**
     * set Target
     * @param $target string target
     */
    public function setTarget($target){
        $this->initialize($target);
        $this->scaffold()->setTarget($target);
    }

    /**
     * set Action
     */
    public function setAction($action){
        $this->isInitialized();
        $this->action = $action;
        $this->foreperson =
            $this->createForeperson($this->action);
        $this->scaffold()->setForeperson($action);
    }

    /**
     * set scaffold bean
     */
    public function setScaffold($bean){
        $this->mediator->setScaffold($bean);
    }

    /**
     * get scaffold bean
     */
    public function scaffold(){
        return $this->mediator->scaffold();
    }

    /**
     * mediator
     */
    public function mediator(){
        return $this->mediator;
    }

    /**
     * get foreperson
     */
    public function foreperson(){
        return $this->foreperson;
    }

    /**
     * organize a scaffold
     */
    public function organize(){
        $this->isInitialized();
        if(is_null($this->action)){
            throw new Ficus_NotReadyException("action is empty, you must run guess or setAction before runch organize method");
        }
        $smarty = Ficus_ScaffoldComponentFactory::getSmarty();
        $scaffold = $this->mediator->scaffold();
        if(is_null($scaffold) == false){
            $this->assignScaffoldValues($smarty, $scaffold);
        }
        $template = $this->template($scaffold);
        $html = $this->foreperson->organize();
        if(is_null($html) == false){
            $this->assignScaffoldValues($smarty, $scaffold);
        }
        
        $smarty->assign(self::KEY_SCAFFOLD, $html);

        $scaffold = $smarty->fetch($template);
        if($this->fuzzy){
            $header = $smarty->fetch(self::TEMPLATE_HEADER);
            $footer = $smarty->fetch(self::TEMPLATE_FOOTER);
            $scaffold = $header . $scaffold . $footer;
        }
        return $scaffold;
    }

    /**
     * assign scaffold values
     */
    protected function assignScaffoldValues($smarty, $scaffold){
        $smarty->assign(self::KEY_PAGE,    $scaffold->page());
        $smarty->assign(self::KEY_METHOD,  $scaffold->method());
        $smarty->assign(self::KEY_ACTION,  $scaffold->action());
        $smarty->assign(self::KEY_TARGET,  $scaffold->target());
        $smarty->assign(self::KEY_DO,      $scaffold->do());
        $smarty->assign(self::KEY_MESSAGE, $scaffold->message());
        $smarty->assign(self::KEY_TRANSITION, $scaffold->transition());
        $smarty->assign(self::KEY_FOREPERSON, $scaffold->foreperson());
        $smarty->assign(self::KEY_SUBMIT_TITLE,
                        $scaffold->submitTitle());
    }
    
    /**
     * set entity
     */
    public function setEntity($entity){
        $this->mediator->setEntity($entity);
    }

    /**
     * entity
     */
    public function entity(){
        return $this->mediator->entity();
    }

    /**
     * get deserialized entity
     */
    public function getEntityFromRequest(){
        return $this->mediator->getEntityFromRequest();
    }

    /**
     * fuzzy
     */
    public function setFuzzy($fuzzy){
        $this->fuzzy = $fuzzy;
    }
    
    /**
     * is initialized
     */
    protected final function isInitialized(){
        if(is_null($this->target)){
            throw new Ficus_NotReadyException("target is empty, you must run setTarget or guess before runch any public functions");
        }
    }

    /**
     * get template
     */
    protected function template($scaffold){
        $do = $scaffold->do();
        $action = $scaffold->action();
        if($do == "submit"){
            return "ScaffoldSubmit";
        }else if($do == "confirm"){
            return "ScaffoldConfirm";
        }
        return ucfirst($action);
    }
    
    /**
     * initialize
     */
    protected function initialize($target){
        $this->target = $target;
        $this->mediator->setTarget($this->target);
        $this->mediator->setRequest($this->page->request());
    }
    
    /**
     * create foreperson
     * @param $action string action
     * @param $mediator Ficus_ScaffoldMediator mediator
     * @return UTil_ScaffoldForeperson
     */
    protected function createForeperson($action){
        if($this->mediator->request()->has(self::KEY_DO)){
            $do = $this->mediator->request()->extractAsString(self::KEY_DO);
        }else{
            $do = self::DO_DEFAULT;
        }

        switch($action){
        case(self::ACTION_VIEW):
            return new Ficus_ViewScaffoldForeperson($this->mediator, $do); 
        case(self::ACTION_LIST):
            return new Ficus_ListScaffoldForeperson($this->mediator, $do);
        case(self::ACTION_EDIT):
            return new Ficus_EditScaffoldForeperson($this->mediator, $do);
        case(self::ACTION_NEW):
            return new Ficus_NewScaffoldForeperson($this->mediator, $do);
        case(self::ACTION_DELETE):
            return new Ficus_DeleteScaffoldForeperson($this->mediator, $do);
        default: throw new Ficus_NotSupportedException("scaffold foreperson $action is not available");
        }
    }
}
?>
