<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.pages
 */
/**
 * @file PageController.php
 * @brief controll page
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: PageController.php 37 2007-08-28 01:17:50Z ishitoya $
 */

require_once("smarty/Smarty.class.php");
require_once("ficus/pages/PageModeExtractor.php");
require_once("ficus/lang/ClassLoader.php");
require_once("ficus/exception/ClassNotFoundException.php");
require_once("ficus/exception/PageNotFoundException.php");
require_once("ficus/lang/Class.php");
require_once("ficus/lang/String.php");
require_once("ficus/pages/PageComponentFactory.php");
require_once("ficus/pages/PageConstants.php");

/**
 * @class Ficus_PageController
 */
class Ficus_PageController implements Ficus_PageConstants{
    /**
     * cache dir
     */
    protected $cacheDir = null;
    /**
     * base dir
     */
    protected $baseDir = null;
    /**
     * pages
     */
    protected $pages = array();
    /**
     * mode extractor
     */
    protected $extractor = null;
    /**
     * @var $chain Ficus_PageActionSet
     */
    protected $chain = null;

    /**
     * @var $listeners array of Ficus_PageListener
     */
    protected $listeners = null;
    
    /**
     * constructor
     */
    public function __construct(){
        $templates =
            Ficus_Dir::normalize(Ficus_File::currentDir() . "/templates");
        Ficus_PageComponentFactory::getSmarty()->addPath(self::TEMPLATE_NAME,
                                                         $templates);
        $this->chain = new Ficus_PageActionSet();
        $this->listeners = new Ficus_ConcreteBeanAggregator("listeners");
        $listeners = Ficus_Registry::search("pages.listeners");
        if($listeners == false){
            $listeners = array();
        }
        foreach($listeners as $listener => $value){
            $instance = Ficus_PageComponentFactory::getListener($listener);
            $this->listeners->add($instance);
        }
    }

    /**
     * set base dir
     */
    public function setBaseDir($base){
        $this->baseDir = Ficus_Dir::normalize($base);
        $template = $this->baseDir . "/" . self::TEMPLATES_DIR;
        $smarty = Ficus_PageComponentFactory::getSmarty();
        $smarty->addUserPath(self::TEMPLATE_NAME, $template);
        $smarty->setTemplateDir($template);

    }

    /**
     * call method
     * @param $name string name of page
     * @param $mode string name of mode
     * @param $buffering boolean buffering
     * @return mixed return value of method
     */
    public function execute($name = null, $mode = null, $buffering = false){
        $request = new Ficus_PageRequest(func_get_args());
        if(is_null($name)){
            $name =
                Ficus_PageComponentFactory::getPageLoader()->dispatch($request);
        }
        if($this->isPageExists($name) == false){
            $this->addPage($name);
        }

        if(is_null($mode)){
            $mode = $this->getMode($name);
        }

        $page = $this->pages[$name]["page"];

        if($page->authorization() == self::FORWARDED){
            return;
        }
        if($buffering == true){
            try{
                Ficus_Assert::isInstanceOf($page, "Ficus_BufferedPage");
            }catch(Ficus_TypeMismatchException $e){
                throw new Ficus_IllegalArgumentException("out buffering is on, but page has not ability to buffer.");
            }
            $page->enableOutputBuffering();
        }else{
            if($page instanceof Ficus_BufferedPage){
                $page->disableOutputBuffering();
            }
        }

        $this->chain->add($name, $mode);

        foreach($this->listeners as $listener){
            $listener->onDo($page, $mode, $request);
        }
        $result = call_user_func(array($page, $mode), $request);
        foreach($this->listeners as $listener){
            $listener->onDone($page, $mode, $request, $result);
        }
        return $result;
    }

    /**
     * get mode from extractor
     * @param $name String name of page
     * @return String of mode name
     */
    protected function getMode($name){
        $mode = $this->pages[$name]["extractor"]->getMode();
        return $mode;
    }
    
    /**
     * add a page to pages array
     * @param $name String name of page
     */
    protected function addPage($name){
        $page = Ficus_PageComponentFactory::getPageLoader()->load($name);
        $extractor = new Ficus_PageModeExtractor($page);
        $extractor->setDefaultMode($page->getDefaultMode());
        $this->pages[$name]["page"] = $page;
        $this->pages[$name]["extractor"] = $extractor;
    }

    /**
     * checks page exist in pages array
     * @param $name String name of page
     * @return boolean return true if page exists
     */
    protected function isPageExists($name){
        return array_key_exists($name, $this->pages);
    }

    /**
     * chain
     */
    public function chain(){
        return $this->chain;
    }
    
    /**
     * forward
     * @param $pagename string target pagename
     * @param $actionname string actionname
     */
    public function forward($from, $pagename, $actionname = null, $buffering = false){
        if($pagename == $from->pagename()){
            return $this->pass($from, $actionname);
        }
        try{
            return $this->execute($pagename, $actionname, $buffering);
        }catch(Ficus_PageNotFoundException $e){
        }

        $actionname = $pagename;
        return $this->pass($from, $actionname);
    }

    /**
     * pass action in the page
     * @param $action name string actionname
     */
    public function pass($from, $actionname, $args = null){
        $request = new Ficus_PageRequest($args);
        return $from->{$actionname}($request);
    }

    /**
     * move
     * @param $pagename string target pagename
     * @param $actionname string actionname
     */
    public function move($from,
                         $pagename = null, $actionname = null, $time = 0){
        $location = "Location: ";
        if(preg_match('/https?:\/\//', $pagename)){
            $location .= $pagename;
        }else{
            $self = $_SERVER["PHP_SELF"];
            if(preg_match('/(.*?)index.php$/', $self, $regs)){
                $self = $regs[1];
            }
            if(is_null($pagename) && is_null($actionname)){
                $location .= $self;
            }else if(is_null($pagename) == false && is_null($actionname)){
                $location .= "$self?$pagename";
            }else if(is_null($pagename) && is_null($actionname) == false){
                $location .= "$self?$actionname";
            }else{
                $location .= "$self?$pagename&$actionname";
            }
        }
        if($time > 0){
            sleep($time);
        }
        header($location);
    }

    /**
     * add listeners
     */
    public function listeners(){
        return $this->listeners;
    }
}
?>