<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */

/**
 * PHP Version 5
 *
 * LICENSE: This source file is subject to version 3.0 of the PHP license
 * that is available through the world-wide-web at the following URI:
 * http://www.php.net/license/3_0.txt.  If you did not receive a copy of
 * the PHP License and are unable to obtain it through the web, please
 * send a note to license@php.net so we can mail you a copy immediately.
 *
 * @category   PHP
 * @package      Ficus_ReflectionAnnotation
 * @author     Sebastian Bergmann <sb@sebastian-bergmann.de>
 * @copyright  2005 Sebastian Bergmann <sb@sebastian-bergmann.de>
 * @license    http://www.php.net/license/3_0.txt  PHP License 3.0
 * @version    CVS: $Id: ReflectionAnnotationParser.php 2 2007-07-11 10:37:48Z ishitoya $
 * @link       http://pear.php.net/package/  Ficus_ReflectionAnnotation
 * @since      File available since Release 1.0.0
 */

/**
 * Parser for Annotations.
 *
 * @category   PHP
 * @package      Ficus_ReflectionAnnotation
 * @author     Sebastian Bergmann <sb@sebastian-bergmann.de>
 * @copyright  2005 Sebastian Bergmann <sb@sebastian-bergmann.de>
 * @license    http://www.php.net/license/3_0.txt  PHP License 3.0
 * @version    Release: @package_version@
 * @link       http://pear.php.net/package/  Ficus_ReflectionAnnotation
 * @since      Class available since Release 1.0.0
 */
class   Ficus_ReflectionAnnotationParser {
    const ANNOTATION_PATTERN =
      '/@([^\s]+?(?=[\s\(]))\s*(\(\s*((?:(?!^@).)*)\s*\)|([^\n]+))/ms';
//      '/@([^\s]+?(?=[\s\(]))\s*\(\s*((?:(?!^@).)*)\s*\)/ms';
    const ARRAY_PATTERN =
      '/([^\s]*?)\s*=\s*\{(.*?)\}(?=\s*,\s*[^{]+?=\s*\{|$)/ms';
    const QUOTED_PATTERN =
      '/([^\s]*?)\s*=\s*"(.*?)((?<!\\\)")/ms';
    const VALUE_PATTERN =
      '/([^\s]*?)\s*=\s*([a-zA-Z0-9][a-zA-Z0-9._]*)\s?/ms';
    const JAVA_STYLE_PATTERN =
      '/^\s*([^\s]*?)\s*=/';
    const T_BRIEF = "brief";
    const T_COMMENT = "comment";

    /**
     * Parser for Annotations.
     *
     * @param  string $docComment
     * @return array
     * @access public
     */
    public static function parseAnnotations($docComment) {
        $annotations = array();
        //remove comment indent
        $docComment = preg_replace('/ *\* */', '', $docComment);

        preg_match_all(self::ANNOTATION_PATTERN, $docComment, $matches);
        foreach($matches[3] as $key => $match){
            if(empty($match) == false){
                $matches[2][$key] = $match;
            }
        }
        $annotations = array_combine($matches[1], $matches[2]);

        foreach($annotations as $name => $annotation){
            $annotations[$name] =
              self::parseAnnotationContents($name, $annotation);
        }

        $comments = preg_split('/\n/', $docComment);
        if(array_key_exists(self::T_BRIEF, $annotations) == false &&
           empty($comments[1]) == false){
            $annotations[self::T_BRIEF] = $comments[1];
        }
        if(array_key_exists(self::T_COMMENT, $annotations) == false){
            $comment = "";
            for($i = 2; $i < count($comments); $i++){
                if(empty($comments[$i]) == false){
                    $comment .= $comments[$i];
                }else{
                    $annotations[self::T_COMMENT] = $comment;
                    break;
                }
            }
        }
        return $annotations;
    }

    /**
     * Parser for Annotation Contents
     *
     * @param string $name name of annotation
     * @param string $annotations contents of annotation
     * @return array array of parsed contents
     * @access public
     */
    public static function parseAnnotationContents($name, $annotation){
        if(preg_match(self::JAVA_STYLE_PATTERN, $annotation)){
            preg_match_all(self::ARRAY_PATTERN, $annotation, $arrayMatches);
            $annotation = preg_replace(self::ARRAY_PATTERN, "", $annotation);
            $arrayContents = array();
            while(($name  = each($arrayMatches[1])) !== false &&
                  ($value = each($arrayMatches[2])) !== false){
                $arrayContents[$name["value"]] =
                  self::parseArrayAnnotationContents($value["value"]);
            }
            
            preg_match_all(self::QUOTED_PATTERN, $annotation, $quotedMatches);
            $annotation = preg_replace(self::QUOTED_PATTERN, "", $annotation);
            
            preg_match_all(self::VALUE_PATTERN, $annotation, $valueMatches);
            
            $keys =
              array_merge($quotedMatches[1], $valueMatches[1]);
            $values =
              array_merge($quotedMatches[2], $valueMatches[2]);
            $contents = array_combine($keys, $values);
            $contents = array_merge($contents, $arrayContents);
            
            return $contents;
        }else{
            return $annotation;
        }
    }

    /**
     * Parse Array Annotation Contents
     *
     * @param string $contents contents of annotation
     * @return array array of parsed contents
     * @access public
     */
    public static function parseArrayAnnotationContents($annotation){
        $contents = array();
        $arrayPattern = '/(([^\s]*?)\s*=\s*)?\{(.*?)\}(?=\s*,\s*{|$)/ms';
        $valuePattern = '/(([^\s]*?)\s*=\s*)?(\"(.*?)(?<!\\\)"|([a-zA-Z0-9][a-zA-Z0-9._]*)\s*,?\s*)/ms';
        if(preg_match_all($arrayPattern, $annotation, $matches)){
            while(($name  = each($matches[2])) !== false &&
                  ($value = each($matches[3])) !== false){
                if(empty($name["value"])){
                    $contents[] =
                        self::parseArrayAnnotationContents($value["value"]);
                }else{
                    $contents[$name["value"]] =
                        self::parseArrayAnnotationContents($value["value"]);
                }
            }
        }else if(preg_match_all($valuePattern, $annotation, $matches)){
            while(($name   = each($matches[2])) !== false &&
                  ($value1 = each($matches[4])) !== false &&
                  ($value2 = each($matches[5])) !== false){
                $n  = $name["value"];
                $v1 = $value1["value"];
                $v2 = $value2["value"];
                if(empty($n)){
                    $contents[] = (empty($v1)) ? $v2 : $v1;
                }else{
                    $contents[$n] = (empty($v1)) ? $v2 : $v1;
                }
            }
                
        }
        return $contents;
    }        
}

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * c-hanging-comment-ender-p: nil
 * End:
 */
?>
