<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2006 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.lang
 */
/**
 * @file Session.php
 * @brief class utility class
 * @author <a href="mailto:kent@guarana.cc">ISHTIOYA Kentaro</a>
 * @version $Id: Session.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * class utility class
 */

/**
 * @class Ficus_Session
 */
class Ficus_Session extends Ficus_Bean
{
    /**
     * @var string session id
     */
    protected $id = null;

    /**
     * @var string session name
     */
    protected $name = null;

    /**
     * @var string session root
     */
    protected $root = null;

    /**
     * @var string session save path
     */
    protected $path = null;

    /**
     * @var string cookie lifetime
     */
    protected $expire = 1800; //30 minuites

    /**
     * constructor
     */
    public function __construct(){
        if($this->isStarted() == false){
            ini_set("session.use_only_cookies", true);
            ini_set("session.use_trans_sid", false);
        }
        $this->setExpire($this->expire);
    }

    /**
     * start the session
     * @return Ficus_Session this
     */
    public function start(){
        if($this->isStarted() == false){
            session_start();
        }
        $this->id = session_id();
        $this->name = session_name();
        if($this->isInitialized() == false){
            $this->initialize();
        }
        return $this;
    }

    /**
     * destroy the session
     * @return Ficus_Session this
     */
    public function destroy(){
        $isStarted = $this->isStarted();
        $_SESSION = array();
        self::destroyTargetSession($this->name, $this->root);
        if($isStarted){
            session_destroy();
        }
        return $this;
    }

    /**
     * destroy target session
     */
    public static function destroyTargetSession($name, $root){
        setcookie($name, '', time() - 2560000, $root);
    }        

    /**
     * get session value
     * @param $name string name of value
     * @return mixed value
     */
    public function get($name = null){
        if($this->isExists() == false){
            $_SESSION = array();
        }
        if(is_null($name)){
            return $_SESSION;
        } else if(isset($_SESSION[$name])){
            return $_SESSION[$name];
        }
        return null;
    }

    /**
     * extend lifetime
     */
    public function extendLifeTime($time){
        if(isset($_COOKIE[$this->name])){
            $value = $_COOKIE[$this->name];
            setcookie($this->name, $value, time() + $time, $this->path);
        }
    }

    /**
     * set session value
     * @param $name string name of value
     * @param $value mixed value to set
     * @return Ficus_Session this
     */
    public function set($name, $value){
        $_SESSION[$name] = $value;
        return $this;
    }

    /**
     * @param $key string to check
     * @return boolean true if exists
     */
    public function hasValue($key){
        return (is_null($this->get($key)) == false);
    }

    /**
     * check is session started
     * @return boolean true if started
     */
    public function isStarted(){
        return isset($_SESSION);
    }

    /**
     * check is cookie exists
     * @return boolean true if exists
     */
    public function isExists(){
        return isset($_COOKIE[$this->name]);
    }

    /**
     * regenerate session id
     */
    protected function regenerate(){
        session_regenerate_id(true);
        $this->id = session_id();
    }

    /**
     * check is session initialized
     * @return boolean true if initialized
     */
    protected function isInitialized(){
        return isset($_SESSION["initialized"]);
    }

    /**
     * initialize session
     */
    protected function initialize(){
        $_SESSION["initialized"] = true;
    }

    
    public function setId($id){
        session_id($id);
        $this->id = $id;
    }
    public function setName($name){
        session_name($name);
        $this->name = $name;
    }
    public function setRoot($root){
        ini_set("session.cookie_path", $root);
        $this->root = $root;
    }
    public function setExpire($expire){
        ini_set("session.cookie_lifetime", $expire);
        $this->expire = $expire;
    }
    public function setPath($path){
        session_save_path($path);
        $this->path = $path;
    }
}
?>
