<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.db.s2dao.models
 */
/**
 * @file S2DaoEntity.php
 * @brief S2DaoEntity.
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: S2DaoEntity.php 56 2007-10-01 09:13:43Z ishitoya $
 *
 */

require_once("ficus/lang/Class.php");
require_once("ficus/beans/Bean.php");
require_once("ficus/db/s2dao/models/S2DaoEntity.php");
require_once("ficus/db/s2dao/models/S2DaoModelConstants.php");

/**
 * @class Ficus_S2DaoEntity
 */
abstract class Ficus_S2DaoEntity extends Ficus_Bean implements Ficus_S2DaoModelConstants{

    /**
     * hook function overload
     * @param $name string of name
     * @param $arguments array of argument
     * @return mixed result
     */
    public function __call($name, $arguments){
        if(preg_match('/^(set|get)(.*?)(Direct)?Hook$/',
                      $name, $regs)){
            $method = $regs[1];
            $target = $regs[2];
            $direct = empty($regs[3]);
            if($method == "set"){
                $value = $arguments[0];
                $this->setterHook($target, $value, $direct);
            }
            return;
        }
        return parent::__call($name, $arguments);
    }

    /**
     * setter hook
     * @param $target string name of target property
     * @param $value  mixed value to set
     * @param $direct boolean is type of direct
     */
    protected function setterHook($target, $value, $direct){
        $target = strtolower($target[0]) . substr($target, 1);
        if($value instanceof Ficus_S2DaoEntity){
            if($value->isBlank() == false){
                $this->blank = false;
            }
            return;
        }
        /* removed with view history
        }else if($target == "id" && is_null($value) == false){
            $this->blank = true;
        }else if($direct == true && is_null($value) == false){
            $this->blank = false;*/
        if($target != "id" && is_null($value) == false){
            $this->blank = false;
        }
    }

    /**
     * fetch 
     */
    public function fetch(){
        if($this->id != null){
            $manager = new Ficus_S2DaoManager($this->getEntityName());
            $entity =  $manager->getSingleEntity($this->id);
            if(is_null($entity)){
                return;
            }
            $reader = new Ficus_S2DaoEntityAnnotationReader($entity);
            $properties = $reader->properties();
            foreach($properties as $property){
                $orig = $entity->get($property);
                $current = $this->get($property);
                $type = $reader->type($property);

                if($type == self::TYPE_DIRECT){
                    continue;
                }else if($current instanceof Ficus_S2DaoEntity &&
                         $current->isBlank() == false){
                    if($type == self::TYPE_LIST && is_null($current->name())){
                        $names = $current->names();
                        foreach($names as $key => $name){
                            if($name["id"] == $current->id()){
                                $current->setName($key);
                            }
                        }
                    }
                    continue;
                }else if(empty($current) && $current !== 0){
                    $this->set($property, $orig);
                }else if($current instanceof Ficus_S2DaoEntity &&
                         $current->isBlank()){
                    $this->set($property, $orig);
                }
            }
        }
    }

    /**
     * fetch all entities
     */
    public function fetchAll(){
        $directs = $this->beanProperties("/IdDirect$/");
        foreach($directs as $key => $direct){
            $id = $this->get($direct);
            if(is_null($id)){
                continue;
            }
            if(preg_match('/^(.*?)Direct$/', $direct, $regs) === false){
                continue;
            }
            $name = $regs[1];
            $entity = $this->get($name);
            $entity->setId($id);
            $entity->fetch();
            $entity->disableBlank();
            $this->set($name, $entity);
        }
    }

    /**
     * get entity name
     */
    public function getEntityName(){
        $class = get_class($this);
        if(preg_match('/_(Auto)?(.*)Entity$/', $class, $regs)){
            return $regs[2];
        }
        return $class;
    }
    
    /**
     * cast class
     * @param $from Ficus_S2DaoEntity entity
     * @to $to string subject name
     */
    protected function convert($from, $to, $auto = false){
        if(is_null($from)){
            return null;
        }
        if(is_object($to)){
            $to = get_class($to);
        }
        $temp = Ficus_Dao::getEntityWithSubject($to, $auto);
        return Ficus_Class::cast($from, $temp);
    }

    /**
     * recursive update method
     */
    public function update(){
        try{
            $reader = new Ficus_S2DaoEntityAnnotationReader($this);
            $properties = $reader->properties();
            if(count($properties) == 1){
                return 0;
            }
            $dao = $this->dao();
            $dao->update($this);
            foreach($properties as $property){
                if($reader->type($property) == self::TYPE_FOREIGN){
                    $getter = "get" . ucfirst($property);
                    $value = $this->{$getter}();
                    $id = $value->id();
                    if(is_null($id) || $value->isBlank()){
                        continue;
                    }
                    $value->update();
                }
            }
            return $this;
        }catch(PDOException $e){
            throw new Ficus_SQLException($e->getMessage() . " At " . $this->getEntityName());
        }
    }

    /**
     * recursive update method
     */
    public function insert(){
        try{
            $reader = new Ficus_S2DaoEntityAnnotationReader($this);
            $properties = $reader->properties();
            foreach($properties as $property){
                if($reader->type($property) != self::TYPE_FOREIGN){
                    continue;
                }
                $getter = "get" . ucfirst($property);
                $setter = "set" . ucfirst($property) . "Direct";
                $value = $this->{$getter}();
                
                if($value->isBlank() == false &&
                   $value->isEmptyId()){
                    $value->insert();
                    $this->{$setter}($value->id());
                }
            }
            
            $this->dao()->insert($this);
            return $this;
        }catch(PDOException $e){
            throw new Ficus_SQLException($e->getMessage() . " At " . $this->getEntityName());
        }
    }

    /**
     * delete
     */
    public function delete(){
        $this->dao()->delete($this);
    }

    /**
     * copy and insert
     */
    public function copyAsNew(){
        $entity = $this->createClone();
        $this->resetId($entity);
        return $entity;
    }

    /**
     * get name from id
     */
    public function getNameFromId($id = null){
        if(is_null($id)){
            $id = $this->id;
        }
        $names = $this->names();
        foreach($names as $name){
            if($name["id"] == $id){
                return $name["name"];
            }
        }
        return null;
    }

    /**
     * get name from id
     */
    public function getKeyFromId($id = null){
        if(is_null($id)){
            $id = $this->id;
        }
        $names = $this->names();
        foreach($names as $key => $name){
            if($name["id"] == $id){
                return $key;
            }
        }
        return null;
    }

    /**
     * reset id
     */
    private function resetId($entity){
        $reader = new Ficus_S2DaoEntityAnnotationReader($entity);
        $properties = $reader->properties();
        foreach($properties as $property){
            if($reader->type($property) == self::TYPE_FOREIGN){
                $getter = "get" . ucfirst($property);
                $setter = "set" . ucfirst($property) . "Direct";
                $value = $this->{$getter}();
                
                if($value->isBlank() == false){
                    $this->resetId($value);
                }
            }
        }
        $entity->setId(null);
    }

    /**
     * get dao
     * @return Dao
     */
    abstract public function dao();

    /**
     * search
     */
    public function search($name){
        if(is_string($name)){
            $name = explode('.', $name);
        }
        $first = array_shift($name);
        if($first != $this->getEntityName()){
            return null;
        }
        if(isset($name[0]) == false){
            return $this;
        }
        
        $second = $name[0];
        $properties = $this->beanProperties();
        foreach($properties as $property){
            $entity = $this->get($property);
            if($property == $second){
                return $entity;
            }
            if($entity instanceof Ficus_S2DaoEntity){
                $result = $entity->search($name);
                if(is_null($result) == false){
                    return $result;
                }
            }
        }
        return null;
    }

    /**
     * public function dispose
     */
    public function dispose(){
        $properties = $this->beanProperties();
        foreach($properties as $property){
            $this->set($property, null);
        }
        $this->enableBlank();
        return $this;
    }

    /**
     * tostring
     */
    public function __toString(){
        $properties = $this->beanProperties();
        $string = "";
        foreach($properties as $property){
            $child = $this->get($property);
            $string .= "$property : ";
            if($child instanceof Ficus_S2DaoEntity){
                $string .= $child->__toString();
            }else{
                $string .= $child;
            }
        }
        return $string;
    }
}
?>
