<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.beans
 */
/**
 * @file ConcreteBeanAggregator.php
 * @brief This interface is ConcreteBeanAggregator.
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: ConcreteBeanAggregator.php 24 2007-08-01 08:33:15Z ishitoya $
 *
 * Abstract ConcreteBeanAggregator.
 */

/**
 * @class Ficus_ConcreteBeanAggregator
 */
class Ficus_ConcreteBeanAggregator extends Ficus_Bean
implements Ficus_BeanAggregator{
    const DEFAULT_PROPERTY_NAME = "beans";
    
    protected $targetPropertyName = null;
    protected $targetBeanName = null;
    protected $targetKey = null;
    protected $comparator = null;

    /**
     * constructor with target property name
     */
    public function __construct($targetPropertyName = null,
                                $beans = array(),
                                $beanClass = null){
        Ficus_Assert::isPrimitiveTypeAllowNull($targetPropertyName, "string");
        if(is_null($targetPropertyName)){
            $targetPropertyName = self::DEFAULT_PROPERTY_NAME;
        }
        $this->targetPropertyName = $targetPropertyName;
        $this->{$this->targetPropertyName} = $beans;
        $this->targetBeanName = $beanClass;
    }

    /**
     * get target property
     */
    protected final function getTargetProperty(){
        return $this->get($this->targetPropertyName);
    }

    /**
     * set target property
     */
    protected final function setTargetProperty($values){
        $this->set($this->targetPropertyName, $values);
        return $this;
    }
    
    /**
     * create instance
     */
    protected function createInstance($values){
        $class = get_class($this);
        return new $class($this->targetPropertyName, $values);
    }

    /**
     * unique
     * @param string $with property name
     */
    public function unique(){
        $results = $this->getTargetProperty();
        $ret = array();
        foreach($results as $result){
            $key = $result->get($this->targetKey);
            if(isset($ret[$key])){
                continue;
            }
            $ret[$key] = $result;
        }

        return $this->createInstance($ret);
    }

    /**
     * sort with
     */
    public function sort(){
        if(is_null($this->comparator)){
            throw new Ficus_IllegalArgumentException("comparator must be specified before sort");
        }
        $ret = array_values($ret);
        usort($ret, array($this->comparator, "compare"));
        return $this->createInstance($ret);
    }

    /**
     * without
     */
    public function without($target){
        $ret = array();
        $beans = $this->getTargetProperty();
        foreach($beans as $key => $bean){
            if($bean->get($this->targetKey) == $target){
                continue;
            }
            $ret[] = $bean;
        }
        return $this->createInstance($ret);
    }

    /**
     * index of
     */
    public function indexOf($key){
        if(isset($this->{$this->targetPropertyName}[$key])){
            return $this->{$this->targetPropertyName}[$key];
        }
        if(is_int($key) && $key >= 0 && $key < $this->count()){
            $i = 0;
            $this->rewind();
            foreach($this as $value){
                if($i == $key){
                    return $value;
                }
                $i++;
            }
        }
        throw new Ficus_IllegalArgumentException("indexOf " . $key . " is not found");   
    }

    /**
     * normalize
     */
    public function normalize(){
        throw new Ficus_NotImplementedException("normalize is not implemented");
    }

    /**
     * getRange
     */
    public function range($limit, $offset = 0){
        $ret = array_slice($this->getTargetProperty(), $offset, $limit);
        return $this->createInstance($ret);
    }

    /**
     * search
     */
    public function search($target){
        $beans = $this->getTargetProperty();
        foreach($beans as $bean){
            if($bean->get($this->targetKey) == $target){
                return $bean;
            }
        }
        return null;
    }

    /**
     * remove
     */
    public function remove($target){
        $beans = $this->getTargetProperty();
        foreach($beans as $key => $bean){
            if($bean->get($this->targetKey) == $target){
                unset($beans[$key]);
            }
        }
        return $this->createInstance($beans);
    }

    /**
     * count
     */
    public function count(){
        return parent::count($this->targetPropertyName);
    }

    /**
     * pop
     */
    public function pop(){
        return parent::pop($this->targetPropertyName);
    }

    /**
     * shift
     */
    public function shift(){
        return parent::shift($this->targetPropertyName);
    }

    /**
     * add
     */
    public function add(){
        $args = func_get_args();
        if(func_num_args() == 1){
            $value = $args[0];
        }else{
            $key = $args[0];
            $value = $args[1];
        }
            
        if(is_null($this->targetBeanName) == false &&
           ($value instanceof $this->targetBeanName) == false){
            throw new Ficus_IllegalArgumentException("passed value must be instance of " . $this->targetBeanName . ".");
        }

        if(isset($key)){
            $this->{$this->targetPropertyName}[$key] = $value;
            return $this;
        }else{
            return parent::push($this->targetPropertyName, $value);
        }
    }

    /**
     * isEmpty
     */
    public function isEmpty(){
        return parent::isEmpty($this->targetPropertyName);
    }
        
    /**
     * is exists
     */
    public function isExists($target){
        return (is_null($this->search($target)) != false);
    }


    public function rewind() {
        reset($this->{$this->targetPropertyName});
    }

    public function current() {
        return current($this->{$this->targetPropertyName});
    }

    public function key() {
        return key($this->{$this->targetPropertyName});
        echo "key: $var\n";
        return $var;
    }

    public function next() {
        return next($this->{$this->targetPropertyName});
    }

    public function valid() {
        return $this->current() !== false;
    }
}
