
#==============================================================================#
# $Id: icon.rb,v 1.3 2003/11/23 10:54:22 yuya Exp $
#==============================================================================#

require 'exerb/error'
require 'exerb/win32/icon_file'
require 'exerb/resource/base'

#==============================================================================#

module Exerb

  class Resource

    class Icon < Base

      def initialize
        super()
        @width     = nil
        @height    = nil
        @bit_count = nil
        @value     = nil
      end

      attr_accessor :width, :height, :bit_count, :value

      def self.new_from_file(filepath, width, height, bit_count)
        icon_file  = File.open(filepath, 'rb') { |file| Win32::IconFile.read(file) }
        icon_entry = icon_file.entries.find { |entry| entry.width == width && entry.height == height && entry.bit_count == bit_count }
        raise(ExerbError, "no such icon #{width}x#{height}x#{bit_count} in #{filepath}") unless icon_entry

        icon = self.new
        icon.width     = icon_entry.width
        icon.height    = icon_entry.height
        icon.bit_count = icon_entry.bit_count
        icon.value     = icon_entry.value

        return icon
      end

      def pack
        return @value
      end

    end # Icon

    class DefaultIcon16x16x16 < Icon

      def initialize
        super()
        @width     = 16
        @height    = 16
        @bit_count = 4
        @value     = 
          "\x28\x00\x00\x00\x10\x00\x00\x00\x20\x00\x00\x00\x01\x00\x04\x00" +
          "\x00\x00\x00\x00\xC0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" +
          "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00" +
          "\x00\x80\x00\x00\x00\x80\x80\x00\x80\x00\x00\x00\x80\x00\x80\x00" +
          "\x80\x80\x00\x00\xC0\xC0\xC0\x00\x80\x80\x80\x00\x00\x00\xFF\x00" +
          "\x00\xFF\x00\x00\x00\xFF\xFF\x00\xFF\x00\x00\x00\xFF\x00\xFF\x00" +
          "\xFF\xFF\x00\x00\xFF\xFF\xFF\x00\x00\x00\x00\x11\x11\x00\x00\x00" +
          "\x00\x00\x01\x11\x11\x10\x00\x00\x00\x01\x11\x11\x11\x11\x10\x00" +
          "\x00\x01\x11\x11\x11\x11\x10\x00\x00\x01\x11\x11\x11\x11\x11\x00" +
          "\x00\x11\x11\x11\x11\x11\x11\x00\x00\x11\x11\x11\x11\x19\x11\x00" +
          "\x00\x11\x11\x11\x11\x19\x17\x00\x00\x11\x11\x11\x19\x11\x17\x00" +
          "\x00\x11\x79\x11\x19\x11\x11\x00\x00\x09\x71\x11\x11\x11\x11\x00" +
          "\x00\x0F\x71\x11\x11\x11\x11\x00\x00\x07\x01\x11\x11\x11\x01\x00" +
          "\x00\x00\x01\x11\x11\x11\x00\x00\x00\x00\x01\x19\x11\x11\x00\x00" +
          "\x00\x00\x00\x00\x00\x00\x00\x00\xFC\x3F\x00\x00\xF8\x1F\x00\x00" +
          "\xE0\x07\x00\x00\xE0\x07\x00\x00\xE0\x03\x00\x00\xC0\x03\x00\x00" +
          "\xC0\x03\x00\x00\xC0\x03\x00\x00\xC0\x03\x00\x00\xC0\x03\x00\x00" +
          "\xE0\x03\x00\x00\xE0\x03\x00\x00\xE0\x03\x00\x00\xF8\x07\x00\x00" +
          "\xF8\x0F\x00\x00\xFF\xFF\x00\x00"
      end

    end # DefaultIcon16x16x16

    class DefaultIcon16x16x256 < Icon

      def initialize
        super()
        @width     = 16
        @height    = 16
        @bit_count = 8
        @value     = 
          "\x28\x00\x00\x00\x10\x00\x00\x00\x20\x00\x00\x00\x01\x00\x08\x00" +
          "\x00\x00\x00\x00\x40\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" +
          "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00" +
          "\x00\x80\x00\x00\x00\x80\x80\x00\x80\x00\x00\x00\x80\x00\x80\x00" +
          "\x80\x80\x00\x00\xC0\xC0\xC0\x00\xC0\xDC\xC0\x00\xF0\xCA\xA6\x00" +
          "\x04\x04\x04\x00\x08\x08\x08\x00\x0C\x0C\x0C\x00\x11\x11\x11\x00" +
          "\x16\x16\x16\x00\x1C\x1C\x1C\x00\x22\x22\x22\x00\x29\x29\x29\x00" +
          "\x55\x55\x55\x00\x4D\x4D\x4D\x00\x42\x42\x42\x00\x39\x39\x39\x00" +
          "\x80\x7C\xFF\x00\x50\x50\xFF\x00\x93\x00\xD6\x00\xFF\xEC\xCC\x00" +
          "\xC6\xD6\xEF\x00\xD6\xE7\xE7\x00\x90\xA9\xAD\x00\x00\x00\x33\x00" +
          "\x00\x00\x66\x00\x00\x00\x99\x00\x00\x00\xCC\x00\x00\x33\x00\x00" +
          "\x00\x33\x33\x00\x00\x33\x66\x00\x00\x33\x99\x00\x00\x33\xCC\x00" +
          "\x00\x33\xFF\x00\x00\x66\x00\x00\x00\x66\x33\x00\x00\x66\x66\x00" +
          "\x00\x66\x99\x00\x00\x66\xCC\x00\x00\x66\xFF\x00\x00\x99\x00\x00" +
          "\x00\x99\x33\x00\x00\x99\x66\x00\x00\x99\x99\x00\x00\x99\xCC\x00" +
          "\x00\x99\xFF\x00\x00\xCC\x00\x00\x00\xCC\x33\x00\x00\xCC\x66\x00" +
          "\x00\xCC\x99\x00\x00\xCC\xCC\x00\x00\xCC\xFF\x00\x00\xFF\x66\x00" +
          "\x00\xFF\x99\x00\x00\xFF\xCC\x00\x33\x00\x00\x00\x33\x00\x33\x00" +
          "\x33\x00\x66\x00\x33\x00\x99\x00\x33\x00\xCC\x00\x33\x00\xFF\x00" +
          "\x33\x33\x00\x00\x33\x33\x33\x00\x33\x33\x66\x00\x33\x33\x99\x00" +
          "\x33\x33\xCC\x00\x33\x33\xFF\x00\x33\x66\x00\x00\x33\x66\x33\x00" +
          "\x33\x66\x66\x00\x33\x66\x99\x00\x33\x66\xCC\x00\x33\x66\xFF\x00" +
          "\x33\x99\x00\x00\x33\x99\x33\x00\x33\x99\x66\x00\x33\x99\x99\x00" +
          "\x33\x99\xCC\x00\x33\x99\xFF\x00\x33\xCC\x00\x00\x33\xCC\x33\x00" +
          "\x33\xCC\x66\x00\x33\xCC\x99\x00\x33\xCC\xCC\x00\x33\xCC\xFF\x00" +
          "\x33\xFF\x33\x00\x33\xFF\x66\x00\x33\xFF\x99\x00\x33\xFF\xCC\x00" +
          "\x33\xFF\xFF\x00\x66\x00\x00\x00\x66\x00\x33\x00\x66\x00\x66\x00" +
          "\x66\x00\x99\x00\x66\x00\xCC\x00\x66\x00\xFF\x00\x66\x33\x00\x00" +
          "\x66\x33\x33\x00\x66\x33\x66\x00\x66\x33\x99\x00\x66\x33\xCC\x00" +
          "\x66\x33\xFF\x00\x66\x66\x00\x00\x66\x66\x33\x00\x66\x66\x66\x00" +
          "\x66\x66\x99\x00\x66\x66\xCC\x00\x66\x99\x00\x00\x66\x99\x33\x00" +
          "\x66\x99\x66\x00\x66\x99\x99\x00\x66\x99\xCC\x00\x66\x99\xFF\x00" +
          "\x66\xCC\x00\x00\x66\xCC\x33\x00\x66\xCC\x99\x00\x66\xCC\xCC\x00" +
          "\x66\xCC\xFF\x00\x66\xFF\x00\x00\x66\xFF\x33\x00\x66\xFF\x99\x00" +
          "\x66\xFF\xCC\x00\xCC\x00\xFF\x00\xFF\x00\xCC\x00\x99\x99\x00\x00" +
          "\x99\x33\x99\x00\x99\x00\x99\x00\x99\x00\xCC\x00\x99\x00\x00\x00" +
          "\x99\x33\x33\x00\x99\x00\x66\x00\x99\x33\xCC\x00\x99\x00\xFF\x00" +
          "\x99\x66\x00\x00\x99\x66\x33\x00\x99\x33\x66\x00\x99\x66\x99\x00" +
          "\x99\x66\xCC\x00\x99\x33\xFF\x00\x99\x99\x33\x00\x99\x99\x66\x00" +
          "\x99\x99\x99\x00\x99\x99\xCC\x00\x99\x99\xFF\x00\x99\xCC\x00\x00" +
          "\x99\xCC\x33\x00\x66\xCC\x66\x00\x99\xCC\x99\x00\x99\xCC\xCC\x00" +
          "\x99\xCC\xFF\x00\x99\xFF\x00\x00\x99\xFF\x33\x00\x99\xCC\x66\x00" +
          "\x99\xFF\x99\x00\x99\xFF\xCC\x00\x99\xFF\xFF\x00\xCC\x00\x00\x00" +
          "\x99\x00\x33\x00\xCC\x00\x66\x00\xCC\x00\x99\x00\xCC\x00\xCC\x00" +
          "\x99\x33\x00\x00\xCC\x33\x33\x00\xCC\x33\x66\x00\xCC\x33\x99\x00" +
          "\xCC\x33\xCC\x00\xCC\x33\xFF\x00\xCC\x66\x00\x00\xCC\x66\x33\x00" +
          "\x99\x66\x66\x00\xCC\x66\x99\x00\xCC\x66\xCC\x00\x99\x66\xFF\x00" +
          "\xCC\x99\x00\x00\xCC\x99\x33\x00\xCC\x99\x66\x00\xCC\x99\x99\x00" +
          "\xCC\x99\xCC\x00\xCC\x99\xFF\x00\xCC\xCC\x00\x00\xCC\xCC\x33\x00" +
          "\xCC\xCC\x66\x00\xCC\xCC\x99\x00\xCC\xCC\xCC\x00\xCC\xCC\xFF\x00" +
          "\xCC\xFF\x00\x00\xCC\xFF\x33\x00\x99\xFF\x66\x00\xCC\xFF\x99\x00" +
          "\xCC\xFF\xCC\x00\xCC\xFF\xFF\x00\xCC\x00\x33\x00\xFF\x00\x66\x00" +
          "\xFF\x00\x99\x00\xCC\x33\x00\x00\xFF\x33\x33\x00\xFF\x33\x66\x00" +
          "\xFF\x33\x99\x00\xFF\x33\xCC\x00\xFF\x33\xFF\x00\xFF\x66\x00\x00" +
          "\xFF\x66\x33\x00\xCC\x66\x66\x00\xFF\x66\x99\x00\xFF\x66\xCC\x00" +
          "\xCC\x66\xFF\x00\xFF\x99\x00\x00\xFF\x99\x33\x00\xFF\x99\x66\x00" +
          "\xFF\x99\x99\x00\xFF\x99\xCC\x00\xFF\x99\xFF\x00\xFF\xCC\x00\x00" +
          "\xFF\xCC\x33\x00\xFF\xCC\x66\x00\xFF\xCC\x99\x00\xFF\xCC\xCC\x00" +
          "\xFF\xCC\xFF\x00\xFF\xFF\x33\x00\xCC\xFF\x66\x00\xFF\xFF\x99\x00" +
          "\xFF\xFF\xCC\x00\x66\x66\xFF\x00\x66\xFF\x66\x00\x66\xFF\xFF\x00" +
          "\xFF\x66\x66\x00\xFF\x66\xFF\x00\xFF\xFF\x66\x00\x21\x00\xA5\x00" +
          "\x5F\x5F\x5F\x00\x77\x77\x77\x00\x86\x86\x86\x00\x96\x96\x96\x00" +
          "\xCB\xCB\xCB\x00\xB2\xB2\xB2\x00\xD7\xD7\xD7\x00\xDD\xDD\xDD\x00" +
          "\xE3\xE3\xE3\x00\xEA\xEA\xEA\x00\xF1\xF1\xF1\x00\xF8\xF8\xF8\x00" +
          "\xF0\xFB\xFF\x00\xA4\xA0\xA0\x00\x80\x80\x80\x00\x00\x00\xFF\x00" +
          "\x00\xFF\x00\x00\x00\xFF\xFF\x00\xFF\x00\x00\x00\xFF\x00\xFF\x00" +
          "\xFF\xFF\x00\x00\xFF\xFF\xFF\x00\x0A\x0A\x0A\x0A\x0A\x0A\x23\x40" +
          "\x23\x23\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x6A\x6A\x25" +
          "\x25\x25\x6A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x23\x2B\x25\x25\x25" +
          "\x25\x25\x25\x25\x23\x0A\x0A\x0A\x0A\x0A\x0A\x2B\x25\x25\x6A\x25" +
          "\x25\x25\x6A\x25\x6A\x0A\x0A\x0A\x0A\x0A\x0A\x25\x25\x25\x6A\x25" +
          "\x01\x24\x25\x24\x40\x40\x0A\x0A\x0A\x0A\x0A\x25\x25\x25\x6A\x25" +
          "\x0A\x01\x6A\x6A\x25\x40\x0A\x0A\x0A\x0A\x6A\x25\x25\x25\x6A\x25" +
          "\x0A\x25\x40\x6A\x6A\x40\x0A\x0A\x0A\x0A\x6A\x6A\x6A\x6A\x6A\x25" +
          "\x01\x6A\x6A\x6A\x40\x94\x0A\x0A\x0A\x0A\x01\x24\x17\x40\x6A\x24" +
          "\x6A\x6A\x6A\x40\x40\xB1\x0A\x0A\x0A\x0A\x0A\x6A\x23\x17\x6A\x25" +
          "\x2B\x6A\x6A\x25\x2B\x2B\x0A\x0A\x0A\x0A\x0A\xBD\xFF\x40\x40\x25" +
          "\x40\x40\x6A\x40\x40\x40\x0A\x0A\x0A\x0A\x0A\xF6\xFF\x25\x25\x25" +
          "\x25\x6A\x6A\x40\x6A\x40\x0A\x0A\x0A\x0A\x0A\x94\x17\x25\x25\x25" +
          "\x25\x25\x24\x24\x6A\x23\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x6A\x6A\x6A" +
          "\x6A\x25\x01\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x23\x6A\x17" +
          "\x17\x40\x40\x23\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\xFC\x3F\x00\x00\xF8\x1F\x00\x00" +
          "\xE0\x07\x00\x00\xE0\x07\x00\x00\xE0\x03\x00\x00\xE0\x03\x00\x00" +
          "\xC0\x03\x00\x00\xC0\x03\x00\x00\xC0\x03\x00\x00\xE0\x03\x00\x00" +
          "\xE0\x03\x00\x00\xE0\x03\x00\x00\xE0\x03\x00\x00\xF8\x07\x00\x00" +
          "\xF8\x0F\x00\x00\xFF\xFF\x00\x00"
      end

    end # DefaultIcon16x16x256

    class DefaultIcon32x32x16 < Icon

      def initialize
        super()
        @width     = 32
        @height    = 32
        @bit_count = 4
        @value     = 
          "\x28\x00\x00\x00\x20\x00\x00\x00\x40\x00\x00\x00\x01\x00\x04\x00" +
          "\x00\x00\x00\x00\x80\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" +
          "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00" +
          "\x00\x80\x00\x00\x00\x80\x80\x00\x80\x00\x00\x00\x80\x00\x80\x00" +
          "\x80\x80\x00\x00\xC0\xC0\xC0\x00\x80\x80\x80\x00\x00\x00\xFF\x00" +
          "\x00\xFF\x00\x00\x00\xFF\xFF\x00\xFF\x00\x00\x00\xFF\x00\xFF\x00" +
          "\xFF\xFF\x00\x00\xFF\xFF\xFF\x00\x00\x00\x00\x00\x00\x00\x00\x00" +
          "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" +
          "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" +
          "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" +
          "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" +
          "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x11\x11" +
          "\x11\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x11\x11\x11" +
          "\x11\x11\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x91\x11\x11" +
          "\x11\x11\x11\x00\x00\x00\x00\x00\x00\x00\x00\x00\x19\x11\x11\x11" +
          "\x11\x11\x11\x10\x00\x00\x00\x00\x00\x00\x00\x01\x91\x11\x11\x11" +
          "\x11\x11\x11\x10\x00\x00\x00\x00\x00\x00\x00\x01\x11\x11\x11\x11" +
          "\x11\x11\x11\x11\x00\x00\x00\x00\x00\x00\x00\x01\x11\x11\x11\x11" +
          "\x11\x11\x11\x11\x00\x00\x00\x00\x00\x00\x00\x11\x11\x11\x11\x11" +
          "\x11\x11\x11\x11\x10\x00\x00\x00\x00\x00\x00\x11\x11\x11\x11\x11" +
          "\x11\x11\x11\x11\x10\x00\x00\x00\x00\x00\x00\x11\x11\x11\x91\x11" +
          "\x11\x11\x91\x11\x10\x00\x00\x00\x00\x00\x00\x11\x11\x11\x11\x11" +
          "\x11\x19\x91\x19\x90\x00\x00\x00\x00\x00\x00\x11\x11\x11\x11\x11" +
          "\x19\x11\x91\x17\x70\x00\x00\x00\x00\x00\x00\x11\x11\x91\x11\x11" +
          "\x19\x11\x11\x17\x90\x00\x00\x00\x00\x00\x00\x11\x19\x79\x11\x11" +
          "\x99\x11\x11\x99\x90\x00\x00\x00\x00\x00\x00\x19\x19\x79\x11\x11" +
          "\x99\x11\x11\x11\x10\x00\x00\x00\x00\x00\x00\x09\x97\x71\x11\x11" +
          "\x11\x11\x11\x11\x10\x00\x00\x00\x00\x00\x00\x07\xFF\x71\x11\x11" +
          "\x11\x11\x19\x11\x10\x00\x00\x00\x00\x00\x00\x01\xFF\x91\x11\x11" +
          "\x11\x11\x19\x91\x00\x00\x00\x00\x00\x00\x00\x00\x77\x11\x11\x11" +
          "\x11\x11\x11\x71\x00\x00\x00\x00\x00\x00\x00\x00\x09\x91\x11\x11" +
          "\x11\x11\x11\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x11\x11\x11" +
          "\x11\x11\x11\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x19\x99" +
          "\x11\x11\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x11" +
          "\x11\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" +
          "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" +
          "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" +
          "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" +
          "\x00\x00\x00\x00\x00\x00\x00\x00\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF" +
          "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xF0\x1F\xFF" +
          "\xFF\xC0\x0F\xFF\xFF\x80\x03\xFF\xFF\x00\x01\xFF\xFE\x00\x01\xFF" +
          "\xFE\x00\x00\xFF\xFE\x00\x00\xFF\xFC\x00\x00\x7F\xFC\x00\x00\x7F" +
          "\xFC\x00\x00\x7F\xFC\x00\x00\x7F\xFC\x00\x00\x7F\xFC\x00\x00\x7F" +
          "\xFC\x00\x00\x7F\xFC\x00\x00\x7F\xFE\x00\x00\x7F\xFE\x00\x00\x7F" +
          "\xFE\x00\x00\xFF\xFF\x00\x00\xFF\xFF\x80\x01\xFF\xFF\xC0\x03\xFF" +
          "\xFF\xE0\x07\xFF\xFF\xF8\x1F\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF" +
          "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
      end

    end # DefaultIcon32x32x16

    class DefaultIcon32x32x256 < Icon

      def initialize
        super()
        @width     = 32
        @height    = 32
        @bit_count = 8
        @value     = 
          "\x28\x00\x00\x00\x20\x00\x00\x00\x40\x00\x00\x00\x01\x00\x08\x00" +
          "\x00\x00\x00\x00\x80\x04\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" +
          "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00" +
          "\x00\x80\x00\x00\x00\x80\x80\x00\x80\x00\x00\x00\x80\x00\x80\x00" +
          "\x80\x80\x00\x00\xC0\xC0\xC0\x00\xC0\xDC\xC0\x00\xF0\xCA\xA6\x00" +
          "\x04\x04\x04\x00\x08\x08\x08\x00\x0C\x0C\x0C\x00\x11\x11\x11\x00" +
          "\x16\x16\x16\x00\x1C\x1C\x1C\x00\x22\x22\x22\x00\x29\x29\x29\x00" +
          "\x55\x55\x55\x00\x4D\x4D\x4D\x00\x42\x42\x42\x00\x39\x39\x39\x00" +
          "\x80\x7C\xFF\x00\x50\x50\xFF\x00\x93\x00\xD6\x00\xFF\xEC\xCC\x00" +
          "\xC6\xD6\xEF\x00\xD6\xE7\xE7\x00\x90\xA9\xAD\x00\x00\x00\x33\x00" +
          "\x00\x00\x66\x00\x00\x00\x99\x00\x00\x00\xCC\x00\x00\x33\x00\x00" +
          "\x00\x33\x33\x00\x00\x33\x66\x00\x00\x33\x99\x00\x00\x33\xCC\x00" +
          "\x00\x33\xFF\x00\x00\x66\x00\x00\x00\x66\x33\x00\x00\x66\x66\x00" +
          "\x00\x66\x99\x00\x00\x66\xCC\x00\x00\x66\xFF\x00\x00\x99\x00\x00" +
          "\x00\x99\x33\x00\x00\x99\x66\x00\x00\x99\x99\x00\x00\x99\xCC\x00" +
          "\x00\x99\xFF\x00\x00\xCC\x00\x00\x00\xCC\x33\x00\x00\xCC\x66\x00" +
          "\x00\xCC\x99\x00\x00\xCC\xCC\x00\x00\xCC\xFF\x00\x00\xFF\x66\x00" +
          "\x00\xFF\x99\x00\x00\xFF\xCC\x00\x33\x00\x00\x00\x33\x00\x33\x00" +
          "\x33\x00\x66\x00\x33\x00\x99\x00\x33\x00\xCC\x00\x33\x00\xFF\x00" +
          "\x33\x33\x00\x00\x33\x33\x33\x00\x33\x33\x66\x00\x33\x33\x99\x00" +
          "\x33\x33\xCC\x00\x33\x33\xFF\x00\x33\x66\x00\x00\x33\x66\x33\x00" +
          "\x33\x66\x66\x00\x33\x66\x99\x00\x33\x66\xCC\x00\x33\x66\xFF\x00" +
          "\x33\x99\x00\x00\x33\x99\x33\x00\x33\x99\x66\x00\x33\x99\x99\x00" +
          "\x33\x99\xCC\x00\x33\x99\xFF\x00\x33\xCC\x00\x00\x33\xCC\x33\x00" +
          "\x33\xCC\x66\x00\x33\xCC\x99\x00\x33\xCC\xCC\x00\x33\xCC\xFF\x00" +
          "\x33\xFF\x33\x00\x33\xFF\x66\x00\x33\xFF\x99\x00\x33\xFF\xCC\x00" +
          "\x33\xFF\xFF\x00\x66\x00\x00\x00\x66\x00\x33\x00\x66\x00\x66\x00" +
          "\x66\x00\x99\x00\x66\x00\xCC\x00\x66\x00\xFF\x00\x66\x33\x00\x00" +
          "\x66\x33\x33\x00\x66\x33\x66\x00\x66\x33\x99\x00\x66\x33\xCC\x00" +
          "\x66\x33\xFF\x00\x66\x66\x00\x00\x66\x66\x33\x00\x66\x66\x66\x00" +
          "\x66\x66\x99\x00\x66\x66\xCC\x00\x66\x99\x00\x00\x66\x99\x33\x00" +
          "\x66\x99\x66\x00\x66\x99\x99\x00\x66\x99\xCC\x00\x66\x99\xFF\x00" +
          "\x66\xCC\x00\x00\x66\xCC\x33\x00\x66\xCC\x99\x00\x66\xCC\xCC\x00" +
          "\x66\xCC\xFF\x00\x66\xFF\x00\x00\x66\xFF\x33\x00\x66\xFF\x99\x00" +
          "\x66\xFF\xCC\x00\xCC\x00\xFF\x00\xFF\x00\xCC\x00\x99\x99\x00\x00" +
          "\x99\x33\x99\x00\x99\x00\x99\x00\x99\x00\xCC\x00\x99\x00\x00\x00" +
          "\x99\x33\x33\x00\x99\x00\x66\x00\x99\x33\xCC\x00\x99\x00\xFF\x00" +
          "\x99\x66\x00\x00\x99\x66\x33\x00\x99\x33\x66\x00\x99\x66\x99\x00" +
          "\x99\x66\xCC\x00\x99\x33\xFF\x00\x99\x99\x33\x00\x99\x99\x66\x00" +
          "\x99\x99\x99\x00\x99\x99\xCC\x00\x99\x99\xFF\x00\x99\xCC\x00\x00" +
          "\x99\xCC\x33\x00\x66\xCC\x66\x00\x99\xCC\x99\x00\x99\xCC\xCC\x00" +
          "\x99\xCC\xFF\x00\x99\xFF\x00\x00\x99\xFF\x33\x00\x99\xCC\x66\x00" +
          "\x99\xFF\x99\x00\x99\xFF\xCC\x00\x99\xFF\xFF\x00\xCC\x00\x00\x00" +
          "\x99\x00\x33\x00\xCC\x00\x66\x00\xCC\x00\x99\x00\xCC\x00\xCC\x00" +
          "\x99\x33\x00\x00\xCC\x33\x33\x00\xCC\x33\x66\x00\xCC\x33\x99\x00" +
          "\xCC\x33\xCC\x00\xCC\x33\xFF\x00\xCC\x66\x00\x00\xCC\x66\x33\x00" +
          "\x99\x66\x66\x00\xCC\x66\x99\x00\xCC\x66\xCC\x00\x99\x66\xFF\x00" +
          "\xCC\x99\x00\x00\xCC\x99\x33\x00\xCC\x99\x66\x00\xCC\x99\x99\x00" +
          "\xCC\x99\xCC\x00\xCC\x99\xFF\x00\xCC\xCC\x00\x00\xCC\xCC\x33\x00" +
          "\xCC\xCC\x66\x00\xCC\xCC\x99\x00\xCC\xCC\xCC\x00\xCC\xCC\xFF\x00" +
          "\xCC\xFF\x00\x00\xCC\xFF\x33\x00\x99\xFF\x66\x00\xCC\xFF\x99\x00" +
          "\xCC\xFF\xCC\x00\xCC\xFF\xFF\x00\xCC\x00\x33\x00\xFF\x00\x66\x00" +
          "\xFF\x00\x99\x00\xCC\x33\x00\x00\xFF\x33\x33\x00\xFF\x33\x66\x00" +
          "\xFF\x33\x99\x00\xFF\x33\xCC\x00\xFF\x33\xFF\x00\xFF\x66\x00\x00" +
          "\xFF\x66\x33\x00\xCC\x66\x66\x00\xFF\x66\x99\x00\xFF\x66\xCC\x00" +
          "\xCC\x66\xFF\x00\xFF\x99\x00\x00\xFF\x99\x33\x00\xFF\x99\x66\x00" +
          "\xFF\x99\x99\x00\xFF\x99\xCC\x00\xFF\x99\xFF\x00\xFF\xCC\x00\x00" +
          "\xFF\xCC\x33\x00\xFF\xCC\x66\x00\xFF\xCC\x99\x00\xFF\xCC\xCC\x00" +
          "\xFF\xCC\xFF\x00\xFF\xFF\x33\x00\xCC\xFF\x66\x00\xFF\xFF\x99\x00" +
          "\xFF\xFF\xCC\x00\x66\x66\xFF\x00\x66\xFF\x66\x00\x66\xFF\xFF\x00" +
          "\xFF\x66\x66\x00\xFF\x66\xFF\x00\xFF\xFF\x66\x00\x21\x00\xA5\x00" +
          "\x5F\x5F\x5F\x00\x77\x77\x77\x00\x86\x86\x86\x00\x96\x96\x96\x00" +
          "\xCB\xCB\xCB\x00\xB2\xB2\xB2\x00\xD7\xD7\xD7\x00\xDD\xDD\xDD\x00" +
          "\xE3\xE3\xE3\x00\xEA\xEA\xEA\x00\xF1\xF1\xF1\x00\xF8\xF8\xF8\x00" +
          "\xF0\xFB\xFF\x00\xA4\xA0\xA0\x00\x80\x80\x80\x00\x00\x00\xFF\x00" +
          "\x00\xFF\x00\x00\x00\xFF\xFF\x00\xFF\x00\x00\x00\xFF\x00\xFF\x00" +
          "\xFF\xFF\x00\x00\xFF\xFF\xFF\x00\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x23\x23\x40\x40\x23\x23\x23\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x23\x40\x40\x25\x25\x25\x25\x25\x40\x40\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x40\x6A\x25\x25\x25\x25\x25\x25\x25\x25\x25\x40\x23\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x23\x6A\x25\x25\x25\x25\x25\x25\x25\x25\x25\x25\x25\x40\x23\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x23" +
          "\x6A\x25\x25\x25\x25\x25\x25\x25\x25\x25\x25\x25\x25\x24\x40\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x40" +
          "\x25\x25\x25\x25\x40\x40\x25\x24\x24\x25\x40\x25\x25\x40\x40\x23" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x40" +
          "\x25\x25\x25\x25\x40\x25\x25\x24\x01\x24\x25\x25\x24\x40\x40\x40" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x23\x40" +
          "\x25\x24\x25\x25\x25\x25\x25\x25\x01\x01\x25\x40\x25\x25\x24\x40" +
          "\x23\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x23\x25" +
          "\x25\x24\x25\x25\x40\x25\x25\x25\x01\x01\x40\x40\x40\x25\x25\x40" +
          "\x01\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x24\x40" +
          "\x25\x25\x25\x25\x6A\x40\x25\x25\x01\x25\x40\x40\x6A\x40\x25\x40" +
          "\x40\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x40\x40" +
          "\x40\x40\x25\x25\x25\x25\x25\x24\x01\x40\x40\x6A\x6A\x40\x40\x17" +
          "\x17\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x40\x25" +
          "\x25\x40\x40\x40\x40\x25\x25\x24\x24\x6A\x40\x40\x6A\x40\x40\x94" +
          "\xB1\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x01\x24" +
          "\x24\x40\x17\x40\x40\x25\x24\x24\x40\x6A\x40\x25\x40\x40\x40\xB1" +
          "\x17\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x01\x40" +
          "\x25\x6A\x94\x17\x25\x25\x25\x25\x6A\x6A\x25\x24\x25\x40\x6A\x6A" +
          "\x17\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x23\x17" +
          "\x40\x17\x94\x17\x40\x25\x25\x40\x6A\x6A\x40\x24\x24\x25\x25\x25" +
          "\x23\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x17" +
          "\x17\x94\x94\x40\x40\x40\x25\x40\x40\x40\x40\x25\x40\x40\x40\x40" +
          "\x23\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x94" +
          "\xFF\xFF\x94\x25\x24\x25\x24\x25\x25\x40\x25\x40\x40\x6A\x40\x40" +
          "\x23\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x23" +
          "\xF6\xF6\x17\x25\x25\x25\x25\x25\x25\x25\x40\x25\x40\x6A\x17\x40" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x94\x07\x40\x25\x25\x25\x25\x25\x25\x25\x24\x24\x24\x40\xB1\x23" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x17\x6A\x25\x25\x25\x25\x25\x25\x25\x24\x01\x01\x40\x23\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x40\x40\x40\x40\x40\x40\x25\x25\x01\x01\x40\x40\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x23\x40\x6A\x17\x17\x40\x40\x40\x40\x23\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x23\x40\x40\x40\x40\x23\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A" +
          "\x0A\x0A\x0A\x0A\x0A\x0A\x0A\x0A\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF" +
          "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xF0\x1F\xFF" +
          "\xFF\xC0\x0F\xFF\xFF\x80\x03\xFF\xFF\x00\x01\xFF\xFE\x00\x01\xFF" +
          "\xFE\x00\x00\xFF\xFE\x00\x00\xFF\xFC\x00\x00\x7F\xFC\x00\x00\x7F" +
          "\xFC\x00\x00\x7F\xFC\x00\x00\x7F\xFC\x00\x00\x7F\xFC\x00\x00\x7F" +
          "\xFC\x00\x00\x7F\xFC\x00\x00\x7F\xFE\x00\x00\x7F\xFE\x00\x00\x7F" +
          "\xFE\x00\x00\xFF\xFF\x00\x00\xFF\xFF\x80\x01\xFF\xFF\xC0\x03\xFF" +
          "\xFF\xE0\x07\xFF\xFF\xF8\x1F\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF" +
          "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
      end

    end # DefaultIcon32x32x256

  end # Resource

end # Exerb

#==============================================================================#
#==============================================================================#
