// $Id: exerb.cpp,v 1.101 2003/03/07 06:00:58 yuya Exp $

////////////////////////////////////////////////////////////////////////////////

#include "exerb.h"

////////////////////////////////////////////////////////////////////////////////

static void  ExInitRuby(int argc, char **argv);
static void  ExReplaceRequire();
static bool  ExMapping();
static void  ExUnMapping();
static bool  ExSetup();
static bool  ExSetupTable(PEXERB_ARCHIVE_HEADER archive);
static bool  ExSetupKcode(PEXERB_ARCHIVE_HEADER archive);
static int   ExExecute();
#ifdef RUBY18
static VALUE ExExecuteStartupScript(VALUE data);
#else
static VALUE ExExecuteStartupScript();
#endif
static void  ExCleanup();
static void  ExFinalRuby();
static void  ExOnErrorID(int resource_id);
static void  ExCreateTemporaryFile(char *filepath_buffer);
static void  ExWriteFile(char *filepath, void *buffer, DWORD size);
static void* ExReadFile(char *filepath);
static bool  ExLoadDll(char *filepath, VALUE filename);
static HMODULE ExLoadDll0(char *filepath, VALUE filename);
static VALUE ExGetInitFunctionName(VALUE filename);
static VALUE ExEvalString(VALUE code, VALUE filename);
static bool  ExAddFeaturesList(VALUE filename);

static VALUE ExRequire(VALUE obj, VALUE filename);
static bool  ExFindInsideFile(VALUE filename, BYTE *type, VALUE *realname, DWORD *id);
static bool  ExLoadInsideScript(VALUE fname, DWORD id);
static bool  ExLoadInsideLibrary(VALUE fname, DWORD id);
#ifdef USE_BRUBY
static bool  ExLoadInsideBinary(VALUE fname, DWORD id);
#endif
static VALUE ExRequireOutside(VALUE fname);
static bool  ExFindOutsideFile(VALUE fname, VALUE *realname, VALUE *filename);
static void  ExLoadOutsideScript(VALUE fname, VALUE name);
static void  ExLoadOutsideLibrary(VALUE fname, VALUE name);
#ifdef USE_BRUBY
static void  ExLoadOutsideBinary(VALUE fname, VALUE name);
#endif

VALUE WINAPI ex_require(const char *fname);
VALUE WINAPI ex_f_require(VALUE obj, VALUE fname);

////////////////////////////////////////////////////////////////////////////////

extern "C" VALUE ruby_errinfo;
extern "C" VALUE rb_load_path;

////////////////////////////////////////////////////////////////////////////////

static char* g_archive = NULL;

static DWORD g_startup_script_id = 0;

static vector<HMODULE>    g_loaded_dll_list;
static map<string, DWORD> g_name2id_table;
static map<DWORD, string> g_id2name_table;
static map<DWORD, BYTE>   g_entry_type_table;
static map<DWORD, DWORD>  g_entry_ptr_table;
static map<DWORD, DWORD>  g_entry_size_table;

static ON_ERROR_FNC ExOnError = NULL;
static ON_FAIL_FNC  ExOnFail  = NULL;

char g_phi_so_filename[MAX_PATH] = "";

////////////////////////////////////////////////////////////////////////////////

int
ExMain(int argc, char **argv, ON_ERROR_FNC onerror, ON_FAIL_FNC onfail)
{
	int result_code = 1;

	ExOnError = onerror;
	ExOnFail  = onfail;

	::ExInitRuby(argc, argv);
	::ExReplaceRequire();
	
	if ( ::ExMapping() ) {
		if ( ::ExSetup() ) {
			result_code = ::ExExecute();
			::ExFinalRuby();
			::ExCleanup();
		}
		::ExUnMapping();
	}

	return result_code;
}

static void
ExInitRuby(int argc, char **argv)
{
#ifdef NT
	::NtInitialize(&argc, &argv);
#endif
	::ruby_init();
	::ruby_set_argv(argc - 1, argv + 1);

	::rb_ary_push(rb_load_path, ::rb_str_new2("."));

	::Init_Exerb();
}

static void
ExReplaceRequire()
{
	::rb_define_global_function("require", (RUBYPROC)ExRequire, 1);
}

static bool
ExMapping()
{
	PEXERB_DOS_HEADER dos_header = (PEXERB_DOS_HEADER)::GetModuleHandle(NULL);
	if ( !dos_header->offset_to_archive || !dos_header->size_of_archive ) {
	    ::ExOnErrorID(IDS_FAIL_FIND_HEADER);
	    return false;
	}

	const int archive_size = (dos_header->size_of_compressed_archive > 0 ? dos_header->size_of_compressed_archive : dos_header->size_of_archive);
	g_archive = new char[archive_size];

	char filepath[MAX_PATH] = "";
	::ExGetSelfFilePath(filepath, sizeof(filepath));

	FILE *file = ::fopen(filepath, "rb");
	if ( !file ) {
		::ExOnErrorID(IDS_FAIL_READ_ARCHIVE);
		return false;
	}
	::fseek(file, dos_header->offset_to_archive, SEEK_SET);
	::fread(g_archive, archive_size, 1, file);
	::fclose(file);

	if ( dos_header->size_of_compressed_archive > 0 ) {
#ifdef USE_ZLIB
		DWORD original_size   = dos_header->size_of_archive;
		DWORD compressed_size = dos_header->size_of_compressed_archive;
		char* decompressed_archive = new char[original_size];
		::uncompress((BYTE*)decompressed_archive, &original_size, (BYTE*)g_archive, compressed_size);
		delete[] g_archive;
		g_archive = decompressed_archive;
#else
	    ::ExOnError("The compressed archive isn't supported on the core.");
	    return false;
#endif
	}

	return true;
}

static void
ExUnMapping()
{
	delete[] g_archive;
}

static bool
ExSetup()
{
	PEXERB_ARCHIVE_HEADER archive = (PEXERB_ARCHIVE_HEADER)g_archive;

	if ( archive->number_of_entry == 0 ) {
		::ExOnErrorID(IDS_FAIL_FIND_SCRIPT);
		return false;
	}

	return ::ExSetupTable(archive) && ::ExSetupKcode(archive);
}

static bool
ExSetupTable(PEXERB_ARCHIVE_HEADER archive)
{
	DWORD i = 0;

	const DWORD base_of_name_table  = (DWORD)archive + archive->offset_to_name;
	const DWORD base_of_entry_table = (DWORD)archive + archive->offset_to_entry;

	PEXERB_NAME_HEADER  name_header  = (PEXERB_NAME_HEADER)base_of_name_table;
	PEXERB_ENTRY_HEADER entry_header = (PEXERB_ENTRY_HEADER)base_of_entry_table;

	for ( i = 0; i < archive->number_of_name; i++, name_header++ ) {
		const string name = (char*)(name_header->offset + base_of_name_table);
		const DWORD  id   = name_header->id;
		g_name2id_table[name] = id;
		g_id2name_table[id]   = name;	
	}

	for ( i = 0; i < archive->number_of_entry; i++, entry_header++ ) {
		const DWORD id = entry_header->id;
		g_entry_type_table[id] = entry_header->type;
		g_entry_ptr_table[id]  = entry_header->offset + base_of_entry_table;
		g_entry_size_table[id] = entry_header->size;

		if ( !g_startup_script_id ) {
			if ( entry_header->type == EXERB_ENTRY_TYPE_SCRIPT ) g_startup_script_id = entry_header->id;
#ifdef USE_BRUBY
			if ( entry_header->type == EXERB_ENTRY_TYPE_BINARY ) g_startup_script_id = entry_header->id;
#endif
		}
	}

	if ( !g_startup_script_id ) {
		::ExOnErrorID(IDS_FAIL_FIND_SCRIPT);
		return false;
	}

	return true;
}

static bool
ExSetupKcode(PEXERB_ARCHIVE_HEADER archive)
{
	switch ( archive->options.kcode ) {
	case EXERB_OPTIONS_KCODE_NONE: ::rb_set_kcode("n"); break; // None
	case EXERB_OPTIONS_KCODE_EUC:  ::rb_set_kcode("e"); break; // EUC
	case EXERB_OPTIONS_KCODE_SJIS: ::rb_set_kcode("s"); break; // Shift-JIS
	case EXERB_OPTIONS_KCODE_UTF8: ::rb_set_kcode("u"); break; // UTF-8
	}

	return true;
}

static int
ExExecute()
{
	int state = 0;

	::rb_protect(ExExecuteStartupScript, 0, &state);

	if ( state ) {
		if ( !ruby_errinfo ) {
			return 1;
		}

		if ( ::rb_obj_is_kind_of(ruby_errinfo, rb_eSystemExit) ) {
			return NUM2INT(::rb_iv_get(ruby_errinfo, "status"));
		}

		::ExOnFail(ruby_errinfo);

		return 1;
	} else {
		return 0;
	}
}

static VALUE
#ifdef RUBY18
ExExecuteStartupScript(VALUE data)
#else
ExExecuteStartupScript()
#endif
{
	const int   type = g_entry_type_table[g_startup_script_id];
	const VALUE code = ::rb_str_new((char*)g_entry_ptr_table[g_startup_script_id], g_entry_size_table[g_startup_script_id]);
	const VALUE name = ::rb_str_new2(g_id2name_table[g_startup_script_id].c_str());

	::ruby_script(RSTRING(name)->ptr);

	if ( type == EXERB_ENTRY_TYPE_SCRIPT ) {
		::ExEvalString(code, name);
#ifdef USE_BRUBY
	} else if ( type == EXERB_ENTRY_TYPE_BINARY ) {
		BRb::loadcode(code, name, false, rb_eLoadError);
#endif
	}

	return Qnil;
}

static void
ExCleanup()
{
	for ( int i = g_loaded_dll_list.size() - 1; i >= 0 ; i-- ) {
		HMODULE module = g_loaded_dll_list[i];
		char filepath[MAX_PATH] = "";
		::GetModuleFileName(module, filepath, sizeof(filepath));
		::FreeLibrary(module);
		::DeleteFile(filepath);
	}

	g_loaded_dll_list.clear();
	g_name2id_table.clear();
	g_id2name_table.clear();
	g_entry_type_table.clear();
	g_entry_ptr_table.clear();
	g_entry_size_table.clear();
}

static void
ExFinalRuby()
{
	::ruby_finalize();
}

static void
ExOnErrorID(int resource_id)
{
	char message[1024] = "";
	::LoadString(::GetModuleHandle(NULL), resource_id, message, sizeof(message));
	::ExOnError(message);
}

void
ExRaiseException(VALUE klass, int res_id)
{
	char message[1024] = "";
	::LoadString(::GetModuleHandle(NULL), res_id, message, sizeof(message));
	::rb_raise(klass, message);
}

void
ExRaiseExceptionVar(VALUE klass, int res_id, ...)
{
	char format[1024] = "";
	::LoadString(::GetModuleHandle(NULL), res_id, format, sizeof(format));

	char message[1024] = "";
	va_list args;
	va_start(args, res_id);
	::vsnprintf(message, sizeof(message), format, args);
	va_end(args);

	::rb_raise(klass, message);
}

static void
ExCreateTemporaryFile(char *filepath_buffer)
{
	char tmpdir[MAX_PATH] = "";
	::GetTempPath(sizeof(tmpdir), tmpdir);
	::GetTempFileName(tmpdir, "ruby", 0, filepath_buffer);
}

static void
ExWriteFile(char *filepath, void *buffer, DWORD size)
{
	DEBUGMSG1("ExWriteFile('%s', ...)\n", filepath);

	FILE *file = ::fopen(filepath, "wb");
	if ( !file ) {
		::rb_load_fail(filepath);
	}

	::fwrite(buffer, 1, size, file);

	::fclose(file);
}

static void*
ExReadFile(char *filepath, DWORD *size)
{
#ifdef RUBY18
	struct _stat fs = {0};
	if ( ::_stat(filepath, &fs) != 0 ) {
		::rb_load_fail(filepath);
	}
#else
	struct stat fs = {0};
	if ( ::stat(filepath, &fs) != 0 ) {
		::rb_load_fail(filepath);
	}
#endif

	void *buffer = ::xmalloc(fs.st_size);

	FILE *file = ::fopen(filepath, "rb");
	if ( !file ) {
		::rb_load_fail(filepath);
	}

	::fread(buffer, 1, fs.st_size, file);

	::fclose(file);

	*size = fs.st_size;

	return buffer;
}

static bool
ExLoadDll(char *filepath, VALUE filename)
{
	DEBUGMSG2("ExLoadDll('%s', '%s')\n", filepath, RSTRING(filename)->ptr);

	const HMODULE module = ::ExLoadDll0(filepath, filename);
	if ( !module ) return false;

	g_loaded_dll_list.push_back(module);

	VALUE basename = ::rb_funcall(rb_cFile, ::rb_intern("basename"), 1, filename);
	if ( ::stricmp(RSTRING(basename)->ptr, "phi.so") == 0 ) {
		char fullpath[MAX_PATH] = "";
		char *phiname = NULL;
		::GetFullPathName(filepath, sizeof(fullpath), fullpath, &phiname);
		::strcpy(g_phi_so_filename, phiname);
	}

	return true;
}

static HMODULE
ExLoadDll0(char *filepath, VALUE filename)
{
	DEBUGMSG2("ExLoadDll0('%s', '%s')\n", filepath, RSTRING(filename)->ptr);

	const HMODULE module = ::LoadLibraryEx(filepath, NULL, LOAD_WITH_ALTERED_SEARCH_PATH);
	const DWORD   error  = ::GetLastError();

	if ( !module ) {
		::DeleteFile(filepath);
		::ExRaiseExceptionVar(rb_eLoadError, IDS_FAIL_CALL_LOADLIBRARYEX, error);
		return NULL;
	}

	const VALUE init_proc_name = ::ExGetInitFunctionName(filename);

	INITPROC init_proc = (INITPROC)::GetProcAddress(module, RSTRING(init_proc_name)->ptr);

	if ( !init_proc ) {
		::FreeLibrary(module);
		::DeleteFile(filepath);
		::ExRaiseException(rb_eLoadError, IDS_FAIL_GET_PROCADDRESS);
		return NULL;
	}

	DEBUGMSG1("  hModule           = 0x%08X\n", module);
	DEBUGMSG1("  vInitFunctionName = '%s'\n", RSTRING(init_proc_name)->ptr);
	DEBUGMSG1("  fpInitProc        = 0x%08X\n", init_proc);
	(*init_proc)();
	DEBUGMSG0("  succeeded\n");

	return module;
}

static VALUE
ExGetInitFunctionName(VALUE filename)
{
	static const ID id_basename = ::rb_intern("basename");
	static const ID id_sub      = ::rb_intern("sub");

	VALUE basename      = ::rb_funcall(rb_cFile, id_basename, 1, filename);
	VALUE regexp_str    = ::rb_str_new2("(\\.so|\\.dll)$");
	VALUE regexp        = ::rb_reg_new(RSTRING(regexp_str)->ptr, RSTRING(regexp_str)->len, 0);
	VALUE feature_name  = ::rb_funcall(basename, id_sub, 2, regexp, ::rb_str_new2(""));
	VALUE function_name = ::rb_str_concat(::rb_str_new2("Init_"), feature_name);

	return function_name;
}

void
ExGetSelfFilePath(LPSTR filepath_buffer, UINT size)
{
	::GetModuleFileName(NULL, filepath_buffer, size);
}

void
ExGetSelfFileName(LPSTR filename_buffer, UINT size)
{
	char self_filepath[MAX_PATH]     = "";
	char self_filepath_tmp[MAX_PATH] = "";
	char *self_filename              = NULL;

	::ExGetSelfFilePath(self_filepath, sizeof(self_filepath));
	::GetFullPathName(self_filepath, sizeof(self_filepath_tmp), self_filepath_tmp, &self_filename);
	::strncpy(filename_buffer, self_filename, size);
}

static VALUE
ExEvalString(VALUE code, VALUE filename)
{
	static const ID    id_eval = ::rb_intern("eval");
	static const VALUE binding = ::rb_const_get(rb_mKernel, ::rb_intern("TOPLEVEL_BINDING"));
	static const VALUE line    = INT2FIX(1);
	return ::rb_funcall(rb_mKernel, id_eval, 4, code, binding, filename, line);
}

static bool
ExAddFeaturesList(VALUE filename)
{
	const VALUE features = ::rb_gv_get("$\"");
	if ( !::rb_ary_includes(features, filename) ) {
		::rb_ary_push(features, filename);
		return true;
	} else {
		return false;
	}
}

static VALUE
ExRequire(VALUE obj, VALUE filename)
{
	::Check_SafeStr(filename);

	BYTE  type     = 0;
	VALUE realname = Qnil;
	DWORD id       = 0;
	if ( !::ExFindInsideFile(filename, &type, &realname, &id) ) {
		DEBUGMSG1("'%s' was not found in the archive.\n", RSTRING(filename)->ptr);
		return ::ExRequireOutside(filename);
	}

	DEBUGMSG1("'%s' was found in the archive.\n", RSTRING(filename)->ptr);
	DEBUGMSG2("the realname of '%s' is '%s'.\n", RSTRING(filename)->ptr, RSTRING(realname)->ptr);

	if ( !::ExAddFeaturesList(realname) ) {
		return Qfalse;
	}

	if ( type == EXERB_ENTRY_TYPE_SCRIPT ) {
		DEBUGMSG1("'%s' is a script.\n", RSTRING(filename)->ptr);
		::ExLoadInsideScript(realname, id);
#ifdef USE_BRUBY
	} else if ( type == EXERB_ENTRY_TYPE_BINARY ) {
		DEBUGMSG1("'%s' is a binary code.\n", RSTRING(filename)->ptr);
		::ExLoadInsideBinary(realname, id);
#endif
	} else if ( type == EXERB_ENTRY_TYPE_LIBRARY ) {
		DEBUGMSG1("'%s' is a shared library.\n", RSTRING(filename)->ptr);
		::ExLoadInsideLibrary(realname, id);
	}

	return Qtrue;
}

static bool
ExFindInsideFile(VALUE filename, BYTE *type, VALUE *realname, DWORD *id)
{
	const string fname     = RSTRING(filename)->ptr;
	const string fname_rb  = string(fname).append(".rb");
	const string fname_brb = string(fname).append(".brb");
	const string fname_so  = string(fname).append(".so");
	const string fname_dll = string(fname).append(".dll");

	if ( (*id = g_name2id_table[fname]) ) {
		*realname = ::rb_str_new2(fname.c_str());
	} else if ( (*id = g_name2id_table[fname_rb]) ) {
		*realname = ::rb_str_new2(fname_rb.c_str());
#ifdef USE_BRUBY
	} else if ( (*id = g_name2id_table[fname_brb]) ) {
		*realname = ::rb_str_new2(fname_brb.c_str());
#endif
	} else if ( (*id = g_name2id_table[fname_so]) ) {
		*realname = ::rb_str_new2(fname_so.c_str());
	} else if ( (*id = g_name2id_table[fname_dll]) ) {
		*realname = ::rb_str_new2(fname_dll.c_str());
	} else {
		*realname = Qnil;
	}

	if ( *realname != Qnil ) {
		*type = g_entry_type_table[*id];
		return true;
	} else {
		return false;
	}
}

static bool
ExLoadInsideScript(VALUE fname, DWORD id)
{
	DEBUGMSG2("ExLoadInsideScript('%s', %i)\n", RSTRING(fname)->ptr, id);

	const DWORD offset = g_entry_ptr_table[id];
	const DWORD size   = g_entry_size_table[id];
	if ( !offset || !size ) return false;

	VALUE script = ::rb_str_new((char*)offset, size);
	::ExEvalString(script, fname);

	return true;
}

static bool
ExLoadInsideLibrary(VALUE fname, DWORD id)
{
	DEBUGMSG2("ExLoadInsideLibrary('%s', %i)\n", RSTRING(fname)->ptr, id);

	const DWORD offset = g_entry_ptr_table[id];
	const DWORD size   = g_entry_size_table[id];
	if ( !offset || !size ) return false;

	if ( !::ExReplaceImportTable((void*)offset) ) {
		::ExRaiseException(rb_eLoadError, IDS_FAIL_MODIFY_IMPORT_TABLE);
	}

	char tmpfile[MAX_PATH] = "";
	::ExCreateTemporaryFile(tmpfile);

	::ExWriteFile(tmpfile, (void*)offset, size);

	return ::ExLoadDll(tmpfile, fname);
}

#ifdef USE_BRUBY

static bool
ExLoadInsideBinary(VALUE fname, DWORD id)
{
	DEBUGMSG2("ExLoadInsideBinary('%s', %i)\n", RSTRING(fname)->ptr, id);

	const DWORD offset = g_entry_ptr_table[id];
	const DWORD size   = g_entry_size_table[id];
	if ( !offset || !size ) return false;

	VALUE code = ::rb_str_new((char*)offset, size);
	BRb::loadcode(code, fname, false, rb_eLoadError);

	return true;
}

#endif

static VALUE
ExRequireOutside(VALUE fname)
{
	DEBUGMSG0("ExRequireOutside(...)\n");

	VALUE realname, filename;
	if ( !::ExFindOutsideFile(fname, &realname, &filename) ) {
		::ExRaiseExceptionVar(rb_eLoadError, IDS_FAIL_FIND_FILE, RSTRING(fname)->ptr);
	}

	DEBUGMSG1("  fname    => %s\n", RSTRING(::rb_inspect(fname))->ptr);
	DEBUGMSG1("  realname => %s\n", RSTRING(::rb_inspect(realname))->ptr);
	DEBUGMSG1("  filename => %s\n", RSTRING(::rb_inspect(filename))->ptr);

	if ( !::ExAddFeaturesList(filename) ) {
		return Qfalse;
	}

	const char* ext = ::strrchr(RSTRING(realname)->ptr, '.');

	if ( ext ) {
		if ( ::stricmp(ext, ".rb") == 0 ) {
			::ExLoadOutsideScript(realname, filename);
#ifdef USE_BRUBY
		} else if ( ::stricmp(ext, ".brb") == 0 ) {
			::ExLoadOutsideBinary(realname, filename);
#endif
		} else if ( ::stricmp(ext, ".so") == 0 || ::stricmp(ext, ".dll") == 0 ) {
			::ExLoadOutsideLibrary(realname, filename);
		} else {
			::ExRaiseExceptionVar(rb_eLoadError, IDS_FAIL_CHECK_TYPE, RSTRING(fname)->ptr);
		}
	} else {
		::rb_bug("has not extension");
	}

	return Qtrue;
}

static bool
ExFindOutsideFile(VALUE fname, VALUE *realname, VALUE *filename)
{
	if ( *realname = ::rb_find_file(fname) ) {
		*filename = fname;
		return true;
	} else {
		const VALUE name_rb  = ::rb_str_concat(::rb_str_dup(fname), ::rb_str_new2(".rb"));
		const VALUE name_brb = ::rb_str_concat(::rb_str_dup(fname), ::rb_str_new2(".brb"));
		const VALUE name_so  = ::rb_str_concat(::rb_str_dup(fname), ::rb_str_new2(".so"));
		const VALUE name_dll = ::rb_str_concat(::rb_str_dup(fname), ::rb_str_new2(".dll"));

		if ( *realname = ::rb_find_file(name_rb) ) {
			*filename = name_rb;
			return true;
#ifdef USE_BRUBY
		} else if ( *realname = ::rb_find_file(name_brb) ) {
			*filename = name_brb;
			return true;
#endif
		} else if ( *realname = ::rb_find_file(name_so) ) {
			*filename = name_so;
			return true;
		} else if ( *realname = ::rb_find_file(name_dll) ) {
			*filename = name_dll;
			return true;
		} else {
			*realname = Qnil;
			*filename = Qnil;
			return false;
		}
	}
}

static void
ExLoadOutsideScript(VALUE fname, VALUE name)
{
	DEBUGMSG2("ExLoadOutsideScript(%s, %s)\n", RSTRING(fname)->ptr, RSTRING(name)->ptr);

	::rb_load(fname, 0);
}

static void
ExLoadOutsideLibrary(VALUE fname, VALUE name)
{
	DEBUGMSG2("ExLoadOutsideLibrary(%s, %s)\n", RSTRING(fname)->ptr, RSTRING(name)->ptr);

	DWORD size    = 0;
	void  *buffer = ::ExReadFile(RSTRING(fname)->ptr, &size);

	if ( !::ExReplaceImportTable(buffer) ) {
		::ExRaiseException(rb_eLoadError, IDS_FAIL_MODIFY_IMPORT_TABLE);
	}

	char tmpfile[MAX_PATH] = "";
	::ExCreateTemporaryFile(tmpfile);

	::ExWriteFile(tmpfile, buffer, size);

	::free(buffer);

	::ExLoadDll(tmpfile, name);
}

#ifdef USE_BRUBY

static void
ExLoadOutsideBinary(VALUE fname, VALUE name)
{
	DEBUGMSG2("ExLoadOutsideBinary(%s, %s)\n", RSTRING(fname)->ptr, RSTRING(name)->ptr);

	BRb::loadfile(fname, false, rb_eLoadError);
}

#endif

////////////////////////////////////////////////////////////////////////////////

VALUE WINAPI
ex_require(const char *fname)
{
	DEBUGMSG1("ex_require('%s')\n", fname);

	return ::ExRequire(Qnil, ::rb_str_new2(fname));
}

VALUE WINAPI
ex_f_require(VALUE obj, VALUE fname)
{
	DEBUGMSG1("ex_f_require('%s')\n", RSTRING(fname)->ptr);

	return ::ExRequire(Qnil, fname);
}

////////////////////////////////////////////////////////////////////////////////
