#! /usr/bin/ruby

#==============================================================================#
# exerb.rb
# $Id: exerb.rb,v 1.29 2003/03/11 02:35:29 yuya Exp $
#==============================================================================#

require 'getoptlong'
require 'exerb/exerb'
require 'exerb/config'

#==============================================================================#
# ExerbCommand Module
module ExerbCommand

  @@verbose = false

  def self.main
    options = parse_argv

    if options[:version]
      print_version
      exit(1)
    end

    if options[:help] || ARGV.size < 1
      print_usage
      exit(1)
    end

    verbose("Exerb #{Exerb::VERSION}")

    rbcfile = ARGV[0]
    rbafile = make_filename(rbcfile, 'rba')
    verbose("Recipe file = #{rbcfile}")
    verbose("Archive file = #{rbafile}")

    rbc = Exerb::RecipeFile.new(rbcfile)
    arc = rbc.archive

    corefile = options[:corefile]
    corefile ||= lookup_corefile(options[:corename]) if options[:corename]
    corefile ||= rbc.corefile                        if rbc.corefile
    corefile ||= lookup_corefile(rbc.corename)       if rbc.corename
    corefile ||= lookup_corefile('cui')
    verbose("Core file = #{corefile}")

    exefile = options[:output]
    exefile ||= rbc.output                           if rbc.output
    exefile ||= make_filename(rbcfile, 'exe')
    verbose("Executable file = #{exefile}")

    if options[:debug]
      verbose("Writing archive file...")
      File.open(rbafile, 'wb') { |file|
        arc.output(file)
      }
    end

    compress = options[:compress]
    compress ||= rbc.compress

    verbose("Building archive...")
    arcbin  = Exerb::Binary::Archive.new(arc.pack, compress)
    verbose("Loading core file...")
    corebin = Exerb::Binary::Core.new_from_file(corefile)
    verbose("Building executable file...")
    exebin  = Exerb::Binary::Executable.new(arcbin, corebin)
    exebin.selfcheck

    verbose("Writing executable file...")
    File.open(exefile, 'wb') { |file|
      exebin.output(file)
    }

    File.chmod(0755, exefile)

    verbose("ok")
  end

  def self.lookup_corefile(name)
    if Exerb::CORE.has_key?(name)
      return Exerb::CORE[name]
    else
      STDERR.puts "#{name} was not found in exerb/config.rb"
      exit(1)
    end
  end

  def self.make_filename(filename, extension)
    return filename.sub(/(\.rbc$|$)/, '.' + extension)
  end

  def self.parse_argv
    parser = GetoptLong.new
    parser.set_options(
      ['--corename', '-c', GetoptLong::REQUIRED_ARGUMENT],
      ['--corefile', '-C', GetoptLong::REQUIRED_ARGUMENT],
      ['--output',   '-o', GetoptLong::REQUIRED_ARGUMENT],
      ['--compress', '-z', GetoptLong::NO_ARGUMENT],
      ['--verbose',  '-v', GetoptLong::NO_ARGUMENT],
      ['--debug',    '-g', GetoptLong::NO_ARGUMENT],
      ['--version',  '-V', GetoptLong::NO_ARGUMENT],
      ['--help',     '-h', GetoptLong::NO_ARGUMENT]
    )

    options = {}
    parser.each_option { |name, argument|
      options[name.sub(/^--/, '').intern] = argument
    }

    options[:compress] = !!options[:compress]
    options[:verbose]  = !!options[:verbose]
    options[:debug]    = !!options[:debug]
    options[:version]  = !!options[:version]
    options[:help]     = !!options[:help]

    @@verbose = options[:verbose]

    return options
  rescue GetoptLong::InvalidOption
    exit(1)
  end

  def self.print_usage
    puts "Exerb #{Exerb::VERSION}"
    puts ""

    puts "Usage: exerb.rb [options] recipe-file"
    puts ""

    puts "Options:"
    puts "  -c  --corename  specifies exerb-core name."
    puts "  -C  --corefile  specifies exerb-core file."
    puts "  -o  --output    specifies output file."
    puts "  -z  --compress  enable archive compression. (using Ruby/zlib)"
    puts "  -v  --verbose   enable verbose mode."
    puts "  -g  --debug     enable debug mode."
    puts "  -V  --version   display version number."
    puts "  -h  --help      display this information."
    puts ""

    puts "Registered Core:"
    Exerb::CORE.keys.sort.each { |name|
      puts "  #{name}\t#{Exerb::CORE[name] || '<not found>'}"
    }
    puts ""
  end

  def self.print_version
    puts "Exerb #{Exerb::VERSION}"
  end

  def self.verbose(msg)
    puts(msg) if @@verbose
  end

end # ExerbCommand

#==============================================================================#

ExerbCommand.main

#==============================================================================#
#==============================================================================#
