;;; cmail-nicknameb.el --- Look up a nickname with BBDB.

;; Author: Keisuke ICHIHASHI <ksuke@tky2.3web.ne.jp>
;; Keywords: mail
;; Create date: 2000-08-09
;; $Id: cmail-nicknameb.el,v 1.1 2000/12/25 01:32:11 yutopia Exp $

;;; Code:
(require 'bbdb-cmail)
(eval-when-compile (require 'cmail-vars)
		   (require 'advice))

;;; user variables
(cmail-i18n-defcustom cmail-nickname-bbdb-prop 'nickname
  ((ja_JP . "*BBDB $B$K$*$1$k%K%C%/%M!<%`%W%m%Q%F%#$NL>A0!#(B")
   (en_US . "*The name of nickname's property in BBDB."))
  :type 'sexp
  :group 'cmail-all-variables
  :group 'cmail-use-nickname-group)

;;; internal variables
(defvar cmail-nickname-nickname-alist nil)
(defvar cmail-nickname-no-nickname-list nil)

;;;
(if cmail-summarize-format
    (save-excursion
      (let ((buf (get-buffer-create " *cmail-temp*")))
	(set-buffer buf)
	(delete-region (point-min) (point-max))
	(insert "(setq cmail-summarize-format \"" cmail-summarize-format "\")")
	(goto-char (point-min))
	(while (re-search-forward "%[-0-9]*n" nil t)
	  (delete-char -1)
	  (insert "e"))
	(eval-buffer)
	(kill-buffer buf)))
  (setq cmail-summarize-format "%d [%-17e] %I%j\n"))

(let ((ja '((ask-nickname . "%s $B$N%K%C%/%M!<%`$O(B? ")
	    (nickname-ask-make-summary . "$B?7$7$$%K%C%/%M!<%`$G%5%^%j$r:FI=<($7$^$9$+(B? ")))
      (en '((ask-nickname . "Nickname of %s ? ")
	    (nickname-ask-make-summary . "Redisplay summary with updated nickname? "))))
  (or (and cmail-i18n-message
	   (boundp 'current-language-environment)
	   (or (featurep 'xemacs)
	       (and (boundp 'emacs-major-version)
		    (boundp 'emacs-minor-version)
		    (or (>= emacs-major-version 21)
			(and (eq emacs-major-version 20) (>= emacs-minor-version 3)))))
	   (cond ((equal current-language-environment "Japanese")
		  (setq cmail-resource-alist (append ja cmail-resource-alist)))
		 (t
		  (setq cmail-resource-alist (append en cmail-resource-alist)))))
      (setq cmail-resource-alist (append en cmail-resource-alist))))

;; (mapcar 'cmail-add-resource
;; 	'((ask-nickname
;; 	   ("Japanese" . "%s $B$N%K%C%/%M!<%`$O(B? ")
;; 	   (t . "Nickname of %s ? "))
;; 	  (nickname-ask-make-summary
;; 	   ("Japanese" . "$B?7$7$$%K%C%/%M!<%`$G%5%^%j$r:FI=<($7$^$9$+(B? ")
;; 	   (t . "Redisplay summary with updated nickname? "))))

(defsubst cmail-nickname-get-list-nickname (addr)
  ;; return value
  ;; (NICKNAME ADDRESS1 ADDRESS2...) or nil
  (let ((nal cmail-nickname-nickname-alist)
	ret)
    (while (and (not ret) nal)
      (if (member addr (cdr (car nal)))
	  (setq ret (car nal))
	(setq nal (cdr nal))))
    ret))

(defsubst cmail-nickname-remove-nickname (addr)
  ;; remove net addresses from cmail-nickname-nickname-alist
  (let ((entry (cmail-nickname-get-list-nickname addr))
	(record (bbdb-search-simple nil addr))
	al)
    (if record
	(setq al (bbdb-record-net record))
      (setq al (list addr)))
    (if entry
	(while al
	  (setq entry (delete (car al) entry))
	  (setq al (cdr al))))
    (if (null (cdr entry))		; no address
	(setq cmail-nickname-nickname-alist (delete entry cmail-nickname-nickname-alist)))))

(defsubst cmail-nickname-add-nickname (addr nick)
  ;; add nickname and net addresses to cmail-nickname-nickname-alist
  (cmail-nickname-remove-nickname addr)
  (let ((entry (assoc nick cmail-nickname-nickname-alist))
	(record (bbdb-search-simple nil addr))
	al)
    (if record
	(setq al (bbdb-record-net record))
      (setq al (list addr)))
    (if entry				; same nickname
	(setcdr entry (append al (cdr entry)))
      (setq cmail-nickname-nickname-alist (cons (cons nick al) cmail-nickname-nickname-alist)))))

(defsubst cmail-nickname-add-no-nickname (addr)
  ;; add net address to cmail-nickname-no-nickname-list
  (let ((record (bbdb-search-simple nil addr))
	al)
    (if record
	(setq al (bbdb-record-net record))
      (setq al (list addr)))
    (or (member (car al) cmail-nickname-no-nickname-list)
	(setq cmail-nickname-no-nickname-list (append al cmail-nickname-no-nickname-list)))))

(defsubst cmail-nickname-remove-no-nickname (addr)
  ;; remove net address from cmail-nickname-no-nickname-list
  (let ((record (bbdb-search-simple nil addr))
	al)
    (if record
	(setq al (bbdb-record-net record))
      (setq al (list addr)))
    (while al
      (setq cmail-nickname-no-nickname-list (delete (car al) cmail-nickname-no-nickname-list))
      (setq al (cdr al)))))

(defun cmail-nicknameb ()
  (let ((addr (cmail-summary-fp-value ?a)))
    (if (member addr cmail-nickname-no-nickname-list)
	(cmail-summary-fp-value ?n)
      (or (cmail-nickname-look-up-nickname addr)
	  (cmail-summary-fp-value ?n)))))

(defun cmail-nickname-look-up-nickname (addr)
  "Look up a nickname with BBDB."
  (let ((nick (car (cmail-nickname-get-list-nickname addr))))
    (if (null nick)
	(let ((record (bbdb-search-simple nil addr)))
	  (if record
	      (setq nick (or (bbdb-record-getprop record cmail-nickname-bbdb-prop)
			     (bbdb-record-name record))))
	  (if nick
	      (cmail-nickname-add-nickname addr nick)
	    (cmail-nickname-add-no-nickname addr))))
    nick))

(defun cmail-nickname-set-bbdb (from addr nick)
  "Add a nickname to BBDB."
  (let ((bbdb-notice-hook nil)
	(bbdb-elided-display nil)
	(record (bbdb-annotate-message-sender from t t t)))
    (if record
	(progn
	  (bbdb-record-putprop record cmail-nickname-bbdb-prop nick)
	  (bbdb-change-record record nil))))
  (cmail-nickname-add-nickname addr nick)
  (cmail-nickname-remove-no-nickname addr))

(defun cmail-nickname-remove-bbdb (addr)
  "Remove a nickname from BBDB."
  (let ((bbdb-notice-hook nil)
	(bbdb-elided-display nil)
	(record (bbdb-search-simple nil addr)))
    (if record
	(progn
	  (bbdb-record-putprop record cmail-nickname-bbdb-prop nil)
	  (bbdb-change-record record nil)
	  (and (get-buffer bbdb-buffer-name)
	       (set-buffer bbdb-buffer-name)
	       (bbdb-redisplay-one-record record)))))
  ;; update cache by deleting it and then looking it up 
  (cmail-nickname-remove-nickname addr)
  (cmail-nickname-look-up-nickname addr))

(defun cmail-nickname-register ()
  "Register a nickname.
If the nickname is empty string, then remove a nickname.
It prompts for redisplay summary after update."
  (interactive)
  (save-excursion
    (let ((inhibit-read-only t)
	  from addr nickname)
      (bbdb/cmail-open-header)
      (setq from (mail-extract-address-components (mail-fetch-field "From")))
      (when from
	(setq addr (car (cdr from)))
	(setq nickname (read-string
			(cmail-format-resource1 'ask-nickname addr)))
	(if (string-equal nickname "")
	    (cmail-nickname-remove-bbdb addr)
	  (cmail-nickname-set-bbdb from addr nickname)))))
  (if (y-or-n-p (cmail-get-resource 'nickname-ask-make-summary))
      (cmail-disp-summary)))

(defun cmail-nickname-bbdb-after-change-hook (record)
  "Update nickname cache when BBDB record has been changed"
  (let ((net (bbdb-record-net record))
	(nick-in-bbdb (or (bbdb-record-getprop record cmail-nickname-bbdb-prop)
			  (bbdb-record-name record)))
	nick-in-cache)
    (while net
      (if nick-in-bbdb
	  (cmail-nickname-remove-no-nickname (car net)))
      (setq nick-in-cache (car (cmail-nickname-get-list-nickname (car net))))
      (if (not (and (stringp nick-in-bbdb)
		    (stringp nick-in-cache)
		    (string= nick-in-bbdb nick-in-cache)))
	  (cmail-nickname-remove-nickname (car net)))
      (setq net (cdr net)))))

(add-hook 'bbdb-after-change-hook 'cmail-nickname-bbdb-after-change-hook)

;;;
(provide 'cmail-nicknameb)

;;; cmail-nicknameb.el ends here
