#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <asm/param.h>
#include <sched.h>
#include "common.h"

#define COUNT		0x01
#define SLEEP		0x02
#define SLEEP_X		0x03
#define CPU		0x04
#define ERROR		0x05
#define PERCENT		0x06
#ifdef _POSIX_PRIORITY_SCHEDULING
#define RR		0xA0
#define FIFO		0xA1
#endif /* _POSIX_PRIORITY_SCHEDULING */

#define DEFAULT_COUNT 1
#define DEFAULT_SLEEP 10
#define DEFAULT_ERROR 10 /*  (--cpuФ%) */
#define DEFAULT_PERCENT 1

#define MAX_SLEEP 10

#define CURRENT 0
#define LAST    1
#define PROC_TERM_INFO_MAX 2
typedef struct proctimeinfo {
        struct procinfo info[PROC_TERM_INFO_MAX];
        pid_t pid;
} procTimeInfo_t;

int main ( int argc, char **argv)
{
        int i, c, index = 0, pid_count=0;
        int count, cpu, cpu_error, percent;
        pid_t *pid;
        struct timeval sleep_time;
        struct timespec tmp_time;
#ifdef _POSIX_PRIORITY_SCHEDULING
	int rrOrFifo = SCHED_OTHER;	// 0:TS, 1:RR, 2:FIFO
	int priority = 1;
#endif /* _POSIX_PRIORITY_SCHEDULING */

        struct option *options;
        static struct option ctps_options[] = {
                {"count", required_argument, 0, COUNT},
                {"sleep", required_argument, 0, SLEEP},
                {"sleep_x", required_argument, 0, SLEEP_X},
                {"cpu", required_argument, 0, CPU},
                {"error", required_argument, 0, ERROR},
                {"percent", required_argument, 0, PERCENT},
#ifdef _POSIX_PRIORITY_SCHEDULING
		{"rr", required_argument, 0, RR},	// RR塼
		{"fifo", required_argument, 0, FIFO},	// FIFO塼
#endif /* _POSIX_PRIORITY_SCHEDULING */
                {0,0,0,0},
        };
        /* set default value */
        sleep_time.tv_sec  = DEFAULT_SLEEP;
        sleep_time.tv_usec = 0;
        cpu                = -1;
        count              = DEFAULT_COUNT;
        cpu_error          = DEFAULT_ERROR;
        percent            = DEFAULT_PERCENT;

        /* make option table */
        options = append_common_options(ctps_options, 0);
        options = append_pid_options(options, 1);

        c = 0;
        while( c != -1 ){
                int tmp;
                c = getopt_long_only (argc, argv, "", options, &index);

                switch( c ){
                case COUNT:
                        tmp = parse_int(options[index].name, optarg);
                        if (tmp>0) count = tmp;
                        break;
                case SLEEP:
                        parse_time("sleep_time", &tmp_time, optarg);
                        if (tmp_time.tv_sec > MAX_SLEEP) {
                                program_fail("sleep max %d", MAX_SLEEP);
                        }
                        sleep_time.tv_sec = tmp_time.tv_sec;
                        sleep_time.tv_usec = tmp_time.tv_nsec/(NANOSEC/MICROSEC);
                        break;
                case SLEEP_X:
                        parse_time_x("sleep_time_x", &tmp_time, optarg);
                        if (tmp_time.tv_sec > MAX_SLEEP) {
                                program_fail("sleep max %d", MAX_SLEEP);
                        }
                        sleep_time.tv_sec = tmp_time.tv_sec;
                        sleep_time.tv_usec = tmp_time.tv_nsec/(NANOSEC/MICROSEC);
                        break;
                case CPU:
                        tmp = parse_int(options[index].name, optarg);
                        if (tmp>=0) cpu = tmp;
                        break;
                case ERROR:
                        tmp = parse_int(options[index].name, optarg);
                        if (tmp>=0) cpu_error = tmp;
                        break;
                case PERCENT: {
                        tmp = parse_int(options[index].name, optarg);
                        if (tmp>0) percent *= tmp;
                        break;
                }
#ifdef _POSIX_PRIORITY_SCHEDULING
		case RR: {
			int min, max;

			min = sched_get_priority_min(SCHED_RR);
			max = sched_get_priority_max(SCHED_RR);

			if ( !strcmp("MAX", optarg) ) {
				priority = max;

			} else if ( !strcmp("MIN", optarg) ) {
				priority = min;

			} else {
				priority = parse_int("rr", optarg);
				if ( priority < min || max < priority ){
					program_fail("illegal argument:rr priority:%d <= %d <= %d", min, priority, max);
				}
			}

			rrOrFifo = SCHED_RR;
			break;
		}
		case FIFO: {
			int min, max;

			min = sched_get_priority_min(SCHED_FIFO);
			max = sched_get_priority_max(SCHED_FIFO);

			if ( !strcmp("MAX", optarg) ) {
				priority = max;

			} else if ( !strcmp("MIN", optarg) ) {
				priority = min;

			} else {
				priority = parse_int("rr", optarg);
				if ( priority < min || max < priority ){
					program_fail("illegal argument:fifo priority:%d <= %d <= %d", min, priority, max);
				}
			}

			rrOrFifo = SCHED_FIFO;
			break;
		}
#endif /* _POSIX_PRIORITY_SCHEDULING */

                case -1:        // end of argument
                        break;
                default: {
                        pid_t tmp_pid;
                        if ( !check_common(c, options, index) ){
                                // accept argument : NOP
                        } else if ( !check_pid(c, options, index, &tmp_pid) ){
                                pid_count++;
                        } else {
                                unknown_option(index, c, optarg);
                        }
                }
                }
        }

        check_setarg_pid(1, "Usage: ctps --pid=x");

        /* get pid option */
        pid = malloc( sizeof(pid_t) * pid_count );
        if (!pid) program_fail("malloc pid");
        memset(pid, 0 , sizeof(pid_t) * pid_count);

        optind = 0;
        c = 0;
        i = 0;
        while( c != -1 ){
                c = getopt_long_only (argc, argv, "", options, &index);
                if ( !check_pid(c, options, index, &pid[i]) ){
                        i++;
                }
        }

        /* setting info */
        if (cpu>=0) {
                cabi_information("cpu load : %d%%(1/%d) %d%%(1/100)\n",
                                 cpu, (100*percent), cpu/percent);
                cabi_information("range(+-): %d%%(1/%d) %d%%(1/100)\n",
                                 (cpu * cpu_error)/100,
                                 (100*percent),
                                 (cpu * cpu_error)/(100*percent));
        }
        cabi_information("cycle    : %d.%01ds\n", sleep_time.tv_sec, sleep_time.tv_usec);
        cabi_information("count    : %d\n", count);
        cabi_information("pid      : ");
        for (i=0; i<pid_count; i++) {
                cabi_information("%d ", pid[i]);
        }
        cabi_information("\n");
        cabi_information("sched    : ");
        switch ( rrOrFifo ) {
        case SCHED_RR:
                cabi_information("RR (%d)", priority);
                break;
        case SCHED_FIFO:
                cabi_information("FIFO (%d)", priority);
                break;
        case SCHED_OTHER:
        default:
                cabi_information("TS");
        }
        cabi_information("\n");

/* main loop */
{
        int pid_now;
        struct proctimeinfo *p;
        long long *avg;

        p = malloc( sizeof(struct proctimeinfo) * pid_count);
        if (!p) program_fail("malloc proctimeinfo");
        memset(p, 0 , sizeof(struct proctimeinfo) * pid_count);

        avg = malloc( sizeof(long long) * (pid_count+1)); /* pid + total */
        if (!avg) program_fail("malloc avg");
        memset(avg, 0 , sizeof(long long) * (pid_count+1));

#ifdef _POSIX_PRIORITY_SCHEDULING
        switch ( rrOrFifo ) {
        case SCHED_RR: /* fall down */
        case SCHED_FIFO: {
                struct sched_param param;
                param.sched_priority = priority;
                sched_setscheduler(getpid(), rrOrFifo, &param);
        }
        case SCHED_OTHER:
        default:
                /* blank */
                break;
        }
#endif /* _POSIX_PRIORITY_SCHEDULING */

        for ( pid_now = 0; pid_now < pid_count; pid_now++ ){
                /* inital read */
                get_proc_info(pid[pid_now], &p[pid_now].info[CURRENT]);
        }

        for ( i = 0; i < count; i++ ){
                if (i==0 || pid_count>1) {
                        cabi_information("%3d PID | JIFF | MSEC |", i);
                        cabi_information("  CPU %%");
                        if (cpu >= 0) cabi_information("[ +/- %]");
                        cabi_information("(x%d)|", percent);
                        cabi_information(" METER\n");
                }

                if (sleep_time.tv_sec)   sleep(sleep_time.tv_sec);
                if (sleep_time.tv_usec) usleep(sleep_time.tv_usec);

                for ( pid_now = 0; pid_now < pid_count; pid_now++ ){
                        int j;
                        struct procinfo *last, *current;
                        int use_jiffies;
                        long long use_time;
                        long long cpu_load;

                        last    = &p[pid_now].info[LAST];
                        current = &p[pid_now].info[CURRENT];

                        /* main read */
                        memcpy(last, current, sizeof(struct procinfo));
                        get_proc_info(pid[pid_now], current);

                        use_jiffies = get_cpu_time(current) - get_cpu_time(last);
                        use_time    = get_time_diff(last, current);

                        cabi_information("%c| %5d|", current->state, current->pid);//pid
                        cabi_information(" %5d|", use_jiffies); //jiffies
                        cabi_information(" %5lld|", use_time/MILISEC); //msec

                        cpu_load = (use_jiffies * MICROSEC) / (use_time / percent);

                        cabi_information(" %5lld%%", cpu_load); //CPU

                        avg[pid_now] += cpu_load; /* average */

                        /* print error */
                        if (cpu >= 0) {
                                int ret = cpu - cpu_load;
                                cabi_information("[%5d%%]", ret*-1); // diff
                        }

                        cabi_information("(x%d)|", percent); // percent

                        /* print graph */
                        for (j=4 ; j<(cpu_load/percent) ; j+=10 ) {
                                int k = j/10;
                                if (k >= 9)
                                        cabi_information("*");
                                else
                                        cabi_information("=");
                        }
                        cabi_information("\n");
                } /* pid_now */

        } /* count */

        cabi_information("\n");

        /* print average */
        for (i=0; i<pid_count; i++) {
                if ( count > 1 ) {
                        cabi_information(" |            %5d AVG", pid[i]);
                        cabi_information(" %5lld%%       ", avg[i]/count);
                        cabi_information(" (x%d)|",(percent));
                        cabi_information("\n");
                }
                avg[pid_count] += avg[i];
        }

        /* print total average */
        cabi_information(" |            TOTAL AVG");
        cabi_information(" %5lld%%", avg[pid_count]/(count*pid_count));

        /* check check */
        if (cpu >= 0) {
                int ret = cpu - (avg[pid_count]/(count*pid_count));
                cabi_information("[%5d%%](x%d)|\n", ret*-1, percent);
                if ( ret > (cpu * cpu_error)/100) {
                        program_fail("underrun");
                }
                if ( ret < -((cpu * cpu_error)/100) ) {
                        program_fail("overrun");
                }
        } else {
                cabi_information("\n");
        }
        cabi_result(0, 1);

        free(p);
        free(avg);
        free(pid);
}

        return 0;
}
