#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <stdarg.h>
#include <limits.h>
#include <getopt.h>
#include <unistd.h>
#include <fcntl.h>
#include <cabi/cabi_error.h>
#include <signal.h>
#include <sys/time.h>
#include "common.h"

int export_result = 0;
static int verbose = 1;
static int disp_sigtime = 0;

/* check set argment */
int setarg_cabiid;
int setarg_pid;
int setarg_pgid;
setarg_ucabi_t setarg_ucabi;

/* disp_sigtime */
static struct timeval disp_sigtime_prev;/* ưλ֤ޤϡsignalλ֤ݻ */

struct code_table {
	int code;
	char* name;
};

/*
 * CABI Error Code Table
 */
static const struct code_table error_codes[] = {
	{ 0, "CABI_SUCCESS", 	},	/* Success */
	{ 1, "CABI_ERROR", 	},	/* Error */
	{ 2, "CABI_SYSERR", 	},	/* Linux system error */
	{ 3, "CABI_EINVAL", 	},	/* Invalid argument */
	{ 4, "CABI_EACCESS", 	},	/* No use authority */
	{ 5, "CABI_ENOMEM", 	},	/* Memory allocation failed. */
	{ 6, "CABI_EATTACHED", 	},	/* Process is still attached. */
	{ 7, "CABI_ENOAVLE", 	},	/* PID is registered into another AO. */
	{ 8, "CABI_EREGIST", 	},	/* PID is already registered into AO. */
	{ 9, "CABI_ENOBIND", 	},	/* PID is not bound to AO. */
	{ 10, "CABI_ENOEXIST", 	},	/* AO dose not exist. */
	{ 11, "CABI_EPNOEXIST", },	/* PID dose not exist. */
	{ 12, "CABI_EPGNOEXIST",},	/* PGID dose not exist. */
	{ 13, "CABI_ENOCABIID", },	/* No more cabi id exist. */
	{ 14, "CABI_CREATE_ERR", },	/* cabi_account_create error. */
};

void unknown_option(int index, int c, char *optarg)
{
	program_fail("illegal argument:index=%d,c=%x:%s", index, c, optarg);
}

const char *get_error_name(int code)
{
	int i, cnt;

	cnt = sizeof(error_codes)/sizeof(struct code_table);

	for( i = 0; i < cnt; i++ ){
		if ( error_codes[i].code == code )
			return error_codes[i].name;
	}
	program_fail("unknown error code(%d", code);

	return NULL;	// never reach
}

void program_fail(char *fmt, ...)
{
	va_list ap;
	char *f;

	f = malloc(1+strlen(RESULT_NG)+1+strlen(fmt)+1+1);

	/* output result (stderr) */
	sprintf(f, "\n%s %s\n", RESULT_NG, fmt);
	va_start(ap, fmt);
	vfprintf(stderr, f, ap);
	va_end(ap);

	exit(1);
}

void cabi_fail(char *fmt, ...)
{
	va_list ap;
	char *f;

	f = malloc(strlen(RESULT_NG)+1+strlen(fmt)+1+1);

	/* output result (stderr) */
	sprintf(f, "\n%s %s\n", RESULT_NG, fmt);
	va_start(ap, fmt);
	vfprintf(stderr, f, ap);
	va_end(ap);

	exit(1);
}

void cabi_result(int code, int cr)
{
	char *crstr = "";

	if ( cr ){
		crstr = "\n";
	}
	
	if ( code == CABI_SUCCESS ){
		fprintf(stderr, "\n%s%s", RESULT_OK, crstr);
	} else {
		fprintf(stderr, "\n%s CABI_ERROR %s(%d)%s",
			RESULT_NG, get_error_name(code), code,crstr);
		exit(1);
	}
}

/*
 * åɽޤ
 *
 * export  verbose ˽äơǤ錄줿åɽޤ
 * åνϤ̵ͭȽϰʲ̤Ǥ
 *
 * 
 * exportverbose       
 * 
 * 
 *  OFF   OFF    Ϥʤ  
 * 
 *  OFF   ON     stdout      
 * 
 *  ON    OFF    Ϥʤ  
 * 
 *  ON    ON     stderr      
 *  
 */
void cabi_information(char *fmt, ...)
{
	va_list ap;
	FILE *fp;

	if ( export_result )
		fp = stderr;
	else
		fp = stdout;

	if ( verbose ){
		va_start(ap, fmt);
		vfprintf(fp, fmt, ap);
		va_end(ap);
	}
}

void cabi_information_flush(void)
{
	FILE *fp;

	if ( export_result )
		fp = stderr;
	else
		fp = stdout;

	if ( verbose ){
		fflush(fp);
	}
}

int cabi_verbose(void)
{
	return verbose;
}

int cabi_export(void)
{
	return export_result;
}

void cabi_export_int(const char *name, long long value)
{
	if ( export_result ){
		printf("export %s=%lld\n", name, value);
	}
}

struct flock *file_lock(int type, int whence)
{
	static struct flock ret;
	ret.l_type = type;
	ret.l_start = 0;
	ret.l_whence = whence;
	ret.l_len = 0;
	ret.l_pid = getpid();
	return &ret;
}
struct flock *file_unlock(void)
{
	static struct flock ret;
	ret.l_type = F_UNLCK;
	ret.l_start = 0;
	ret.l_whence = 0;
	ret.l_len = 0;
	ret.l_pid = getpid();
	return &ret;
}

/* for ctloop */
void cabi_file_export_int(char *path, int flag, const char *name, long long value)
{
	if ( export_result ){
		char buf[1024];
		int fd, count;

		memset(buf, 0, sizeof(buf));

		/* check create new file */
		if (flag) {
			/* new file */
			fd = open(path, O_CREAT|O_TRUNC|O_WRONLY);
		} else {
			fd = open(path, O_APPEND|O_WRONLY);
		}

		count = sprintf(buf, "export %s=%lld\n", name, value);
		if (fd < 0 || count < 0) program_fail("file export");

		lseek(fd, 0, SEEK_END);

		/* file lock */
		if ( fcntl(fd, F_SETLKW, file_lock(F_WRLCK, SEEK_SET)) < 0 ) {
			program_fail("file export lock");
		}
//		cabi_information("[%d] locked\n", (int)value);

		/* move file pointer */
		lseek(fd, 0, SEEK_END);

		/* output */
		if (write(fd, buf, count) < 0) program_fail("file export write");
//		cabi_information("[%d] write\n", (int)value);

		close(fd);
	}
}

long long timeval_diff(struct timeval *t1, struct timeval *t2)
{
        long long timediff;

        timediff = t2->tv_sec - t1->tv_sec;
        timediff *= 1000000;
        timediff += t2->tv_usec - t1->tv_usec;

        return timediff;
}




/***********************************************************************
 * getopt parameter
 */

static struct option common_options[] = {
    /* export */
    {"export", no_argument, 0, OPT_CODE_EXPORT},
    {"noexport", no_argument, 0, OPT_CODE_NOEXPORT},
    /* verbose */
    {"verbose", no_argument, 0, OPT_CODE_VERBOSE},
    {"noverbose", no_argument, 0, OPT_CODE_NOVERBOSE},
    /* help */
    {"help", no_argument, 0, OPT_CODE_HELP},
    /* display control */
    {"sigtime", no_argument, 0, OPT_CODE_SIGTIME},
    {"nosigtime", no_argument, 0, OPT_CODE_NOSIGTIME},
    {0,0,0,0},
};

static struct option ucabi_options[] = {
	/* cabi_id */
	{"ucabiid", required_argument, 0, OPT_CODE_UCABI_CABI_ID},

	/* cabi_param.term_act */
	{"TERM_NONE", no_argument, 0, OPT_CODE_UCABI_TERM_NONE},
	{"TERM_BLOCK", no_argument, 0, OPT_CODE_UCABI_TERM_BLOCK},
	{"TERM_SIGNAL", no_argument, 0, OPT_CODE_UCABI_TERM_SIGNAL},
	{"TERM_UNKNOWN", no_argument, 0, OPT_CODE_UCABI_TERM_UNKNOWN},
	{"term_act", required_argument, 0, OPT_CODE_UCABI_TERM_ACT},
	
	/* cabi_param.cabi_signal.pid */
	{"ucabipid", required_argument, 0, OPT_CODE_UCABI_PID},
	
	/* cabi_param.cabi_signal.sig */
	{"sig", required_argument, 0, OPT_CODE_UCABI_SIG},
	
	/* cabi_param.cabi_signal.flag */
	{"flag", required_argument, 0, OPT_CODE_UCABI_FLAG},
	
	/* cabi_param.bind_proc_type */
	{"BIND_NORMAL_PROC", no_argument, 0, OPT_CODE_UCABI_BIND_NORMAL_PROC},
	{"BIND_IDLE_PROC", no_argument, 0, OPT_CODE_UCABI_BIND_IDLE_PROC},
	{"bind_proc_type", required_argument, 0, OPT_CODE_UCABI_BIND_PROC_TYPE},
	
	/* cpu_time */
	{"cpu_time", required_argument, 0, OPT_CODE_UCABI_CPU_TIME},
	{"cpu_time_x", required_argument, 0, OPT_CODE_UCABI_CPU_TIME_X},
	
	/* cpu_period */
	{"cpu_period", required_argument, 0, OPT_CODE_UCABI_CPU_PERIOD},
	{"cpu_period_x", required_argument, 0, OPT_CODE_UCABI_CPU_PERIOD_X},
	
	/* NULL */
	{"ucabi_adr", required_argument, 0, OPT_CODE_UCABI_ADRS},
	
	{0, 0, 0, 0},
};

static struct option cabiid_options[] = {
	/* cabi_id */
	{"cabiid", required_argument, 0, OPT_CODE_CABI_ID},
	{0, 0, 0, 0},
};

static struct option pid_options[] = {
	/* cabi_id */
	{"pid", required_argument, 0, OPT_CODE_PID},
	{0, 0, 0, 0},
};

static struct option pgid_options[] = {
	/* cabi_id */
	{"pgid", required_argument, 0, OPT_CODE_PGID},
	{0, 0, 0, 0},
};

			


/*
 * getopt_long_only()裴˻ꤹͤޤ
 *
 * δؿǤϡtable ǻꤵ줿ˡcustom ǻꤵ줿
 * Ƥɲä֤ޤ
 *
 * free_flag  0 ʳꤹȡtable ΰ褬 free ޤ
 *
 * table, custom Τ NULL ꤹ뤳ȤǽǤ
 *
 * ꥿ͤϡmalloc()ưŪ˳Ƥ줿ΰǤ
 * ΰβϸƤӽФ¦ǹԤɬפޤ
 */
struct option *append_options(struct option *table, const struct option *custom, int free_flag)
{
	struct option *ret;
	int cnt, i, j;

	cnt = 0;
	if ( table != NULL ){
		for ( i = 0; table[i].name != NULL; i++, cnt++ )
			;
	}
	if ( custom != NULL ){
		for ( i = 0; custom[i].name != NULL; i++, cnt++ )
			;
	}

	ret = malloc(sizeof(struct option)*(cnt+1));


	j = 0;
	if ( table != NULL ){
		for ( i = 0; table[i].name != NULL; i++, j++ )
			memcpy(&ret[j], &table[i], sizeof(struct option));
	}

	if ( custom != NULL ){
		for ( i = 0; custom[i].name != NULL; i++, j++ )
			memcpy(&ret[j], &custom[i], sizeof(struct option));
	}

	memset(&ret[j], 0, sizeof(struct option));

	if ( free_flag != 0 && table != NULL ){
		free(table);
	}

	return ret;
}

/*
 * getopt_long_only()裴˻ꤹͤޤ
 *
 * δؿǤϡtable ǻꤵ줿ޥɰ¸Υץ
 * ̥ץɲästruct option ֤ޤ
 *
 * free_flag  0 ʳꤹ custom ΰ褬 free ޤ
 *
 * table  NULL ꤹ뤳ȤǽǤ
 *
 * ꥿ͤϡmalloc()ưŪ˳Ƥ줿ΰǤ
 * ΰβϸƤӽФ¦ǹԤɬפޤ
 */
struct option *append_common_options(struct option *table, int free_flag)
{
	gettimeofday(&disp_sigtime_prev, NULL);
	return append_options(table, common_options, free_flag);
}


/*
 * getopt_long_only()裴˻ꤹͤޤ
 *
 * δؿǤϡtable ǻꤵ줿 ucabi ¤
 * ѤΥץɲästruct option ֤ޤ
 *
 * free_flag  0 ʳꤹ table ΰ褬 free ޤ
 * table  NULL ꤹ뤳ȤǽǤ
 *
 * ꥿ͤϡmalloc()ưŪ˳Ƥ줿ΰǤ
 * ΰβϸƤӽФ¦ǹԤɬפޤ
 */
struct option *append_ucabi_options(struct option *table, int free_flag)
{
	memset(&setarg_ucabi, 0, sizeof(setarg_ucabi));
	return append_options(table, ucabi_options, free_flag);
}

/*
 * getopt_long_only()裴˻ꤹͤޤ
 *
 * δؿǤϡtable ǻꤵ줿 cabipid ѤΥץɲä
 * struct option ֤ޤ
 *
 * free_flag  0 ʳꤹ table ΰ褬 free ޤ
 * table  NULL ꤹ뤳ȤǽǤ
 *
 * ꥿ͤϡmalloc()ưŪ˳Ƥ줿ΰǤ
 * ΰβϸƤӽФ¦ǹԤɬפޤ
 */
struct option *append_cabiid_options(struct option *table, int free_flag)
{
	setarg_cabiid = 0;
	return append_options(table, cabiid_options, free_flag);
}

/*
 * getopt_long_only()裴˻ꤹͤޤ
 *
 * δؿǤϡtable ǻꤵ줿 pid ѤΥץɲä
 * struct option ֤ޤ
 *
 * free_flag  0 ʳꤹ table ΰ褬 free ޤ
 * table  NULL ꤹ뤳ȤǽǤ
 *
 * ꥿ͤϡmalloc()ưŪ˳Ƥ줿ΰǤ
 * ΰβϸƤӽФ¦ǹԤɬפޤ
 */
struct option *append_pid_options(struct option *table, int free_flag)
{
	setarg_pid = 0;
	return append_options(table, pid_options, free_flag);
}

/*
 * getopt_long_only()裴˻ꤹͤޤ
 *
 * δؿǤϡtable ǻꤵ줿 pgid ѤΥץɲä
 * struct option ֤ޤ
 *
 * free_flag  0 ʳꤹ table ΰ褬 free ޤ
 * table  NULL ꤹ뤳ȤǽǤ
 *
 * ꥿ͤϡmalloc()ưŪ˳Ƥ줿ΰǤ
 * ΰβϸƤӽФ¦ǹԤɬפޤ
 */
struct option *append_pgid_options(struct option *table, int free_flag)
{
	setarg_pgid = 0;
	return append_options(table, pgid_options, free_flag);
}



void list_option(const struct option *table)
{
	int i;
	
	fprintf(stderr, "    %-20s %s\n", "option", "required argment");
	for ( i= 0; table[i].name != NULL; i++ ){
		fprintf(stderr, "    %-20s %s\n",
			table[i].name,
			(table[i].has_arg == no_argument)?"":"*"
			);
	}
}




/***********************************************************************
 * command line argument parser
 */
unsigned long parse_cabi_id(const char *name, const char *arg)
{
	unsigned long id = 0;
	char *pp;

	id = strtoul(arg, &pp, 0);
	if ( *pp != '\0' ){
		program_fail("illegal argument:%s=%s", name, arg);
		// never reach
	}
	return id;
}

int parse_int(const char *name, const char *arg)
{
	long id = 0;
	char *pp;

	id = strtol(arg, &pp, 0);
	if ( arg == pp || *pp != '\0' || id > INT_MAX || id < INT_MIN ){
		program_fail("illegal argument:%s=%s", name, arg);
		// never reach
	}
	return id;
}

int parse_pid_t(const char *name, const char *arg)
{
	return parse_int(name, arg);
}

static long period_sec_to_nsec(const char *src, char ** last)
{
	long long val = 0;
	int minus = 1;
	long long order = 100000000;	/* 1 = 100,000,000ns = 0.1s */

	if ( *src == '-' ){
		minus = -1;
		src ++;
	}

	for ( ; *src && order > 0 ; src++, order /= 10 ){
		int c;

		c = *src;
		if ( '0' <= c && c <= '9' ){ 
			c = c - '0';
		} else {
			break;
		}

		val += order * c;

		if ( 999999999 < val ){
			break;
		}
	}
	*last = (char*)src;

	return (long)(val * minus);

}


void parse_time(const char *name, struct timespec *timespec, const char *arg)
{
	char *p, *pp;
	struct timespec ts;
	
	p = strchr(arg, '.');
	if ( p == NULL ){
		ts.tv_sec = strtol(arg, &pp, 0);
		if ( *pp != '\0' ){
			program_fail("illegal argument:%s=%s", name, arg);
			// never reach
		}
		ts.tv_nsec = 0;
	} else {
		ts.tv_sec = strtol(arg, &pp, 0);
		if ( *pp != '.' ){
			program_fail("illegal argument:%s=%s", name, arg);
			// never reach
		}
		ts.tv_nsec = period_sec_to_nsec(p+1, &pp);
		if ( *pp != '\0' ){
			program_fail("illegal argument:%s=%s", name, arg);
			// never reach
		}
	}
	memcpy(timespec, &ts, sizeof(*timespec));
}

/*
 * perse_time Ȥΰ㤤ϡperse_time 10.1  10s100ms Ȳ᤹ΤФ
 * perse_timex ϡ10.1  10s1ns Ȳ᤹뤳ȤǤ
 * perse_timex ϡtimespec ¤Τ tv_nsec  1 ðʾͤꤹ뤿
 * Ѥޤ
 */
void parse_time_x(const char *name, struct timespec *timespec, const char *arg)
{
	char *p, *pp;
	struct timespec ts;
	
	p = strchr(arg, '.');
	if ( p == NULL ){
		ts.tv_sec = strtol(arg, &pp, 0);
		if ( *pp != '\0' ){
			program_fail("illegal argument:%s=%s", name, arg);
			// never reach
		}
		ts.tv_nsec = 0;
	} else {
		ts.tv_sec = strtol(arg, &pp, 0);
		if ( *pp != '.' ){
			program_fail("illegal argument:%s=%s", name, arg);
			// never reach
		}
		ts.tv_nsec = strtol(p+1, &pp, 0);
		if ( *pp != '\0' ){
			program_fail("illegal argument:%s=%s", name, arg);
			// never reach
		}
	}
	memcpy(timespec, &ts, sizeof(*timespec));
}

/*
 * signal ѡ
 *
 * Ƭʸ'S'ξϡʥ̤ʸ(SIGHUP)ǻꤵ줿ȲϤ
 * 'S'ʳξϿͤǻꤵ줿ΤȤƲϤ
 */
int parse_signal(const char *name, const char *arg)
{
	int sig = 0;

	if ( *arg == 'S' ){
		sig = get_signal_code(arg);
	} else {
		sig = parse_int(name, arg);
	}

	return sig;
}

/*
 * command line argument parser
 ***********************************************************************/


/*
 * ̰ޤ
 *
 * c Ϥ줿̰ξѿꤷơ
 * 0 ֤ޤ̰ʳξϤʤˤ⤷ʤ 0 ʳ֤ޤ
 */
int check_common(int c, const struct option *table, int index)
{

	switch(c){
	case OPT_CODE_EXPORT:		export_result = 1;	return 0;
	case OPT_CODE_NOEXPORT:		export_result = 0;	return 0;
	case OPT_CODE_VERBOSE:		verbose = 1;		return 0;
	case OPT_CODE_NOVERBOSE:	verbose = 0;		return 0;
	case OPT_CODE_HELP:		list_option(table);	exit(0);
	case OPT_CODE_SIGTIME:		disp_sigtime = 1;	return 0;
	case OPT_CODE_NOSIGTIME:	disp_sigtime = 0;	return 0;
	default:						return 1;
	}
}


/*
 * ucabi ޤ
 *
 * c Ϥ줿 ucabi ξ ucabi ǻꤵ줿ΰ
 * ͤꤷ 0 ֤ޤucabi ʳξϤʤˤ
 * ʤ 0 ʳ֤ޤ
 */
int check_ucabi(int c, const struct option *options, int index, struct cabi_uaccount **ucabipp)
{
	struct cabi_uaccount *ucabi = *ucabipp;
	
	switch( c ){
	case OPT_CODE_UCABI_CABI_ID:
		ucabi->cabi_id = parse_cabi_id(options[index].name, optarg);
		set_setarg_ucabi_cabiid();
		break;
	case OPT_CODE_UCABI_TERM_NONE:
		ucabi->cabi_param.term_act = CABI_TERM_NONE;
		set_setarg_ucabi_term_act();
		break;
	case OPT_CODE_UCABI_TERM_BLOCK:
		ucabi->cabi_param.term_act = CABI_TERM_BLOCK;
		set_setarg_ucabi_term_act();
		break;
	case OPT_CODE_UCABI_TERM_SIGNAL:
		ucabi->cabi_param.term_act = CABI_TERM_SIGNAL;
		set_setarg_ucabi_term_act();
		break;
	case OPT_CODE_UCABI_TERM_UNKNOWN:
		ucabi->cabi_param.term_act = CABI_TERM_UNKNOWN;
		set_setarg_ucabi_term_act();
		break;
	case OPT_CODE_UCABI_TERM_ACT:
		ucabi->cabi_param.term_act = parse_int(options[index].name, optarg);
		set_setarg_ucabi_term_act();
		break;
	case OPT_CODE_UCABI_PID:
		ucabi->cabi_param.cabi_signal.pid = parse_pid_t(options[index].name, optarg);
		set_setarg_ucabi_pid();
		break;
	case OPT_CODE_UCABI_SIG:
		ucabi->cabi_param.cabi_signal.sig = parse_signal(options[index].name, optarg);
		set_setarg_ucabi_sig();
		break;
	case OPT_CODE_UCABI_FLAG:
		ucabi->cabi_param.cabi_signal.flag = parse_int(options[index].name, optarg);
		set_setarg_ucabi_flag();
		break;
	case OPT_CODE_UCABI_BIND_NORMAL_PROC:
		ucabi->cabi_param.bind_proc_type = BIND_NORMAL_PROC;
		set_setarg_ucabi_bind_proc();
		break;
	case OPT_CODE_UCABI_BIND_IDLE_PROC:
		ucabi->cabi_param.bind_proc_type = BIND_IDLE_PROC;
		set_setarg_ucabi_bind_proc();
		break;
	case OPT_CODE_UCABI_BIND_PROC_TYPE:
		ucabi->cabi_param.bind_proc_type = parse_int(options[index].name, optarg);
		set_setarg_ucabi_bind_proc();
		break;
	case OPT_CODE_UCABI_CPU_TIME:
		parse_time(options[index].name, &ucabi->cpu_time, optarg);
		set_setarg_ucabi_cpu_time();
		break;
	case OPT_CODE_UCABI_CPU_TIME_X:
		parse_time_x(options[index].name, &ucabi->cpu_time, optarg);
		set_setarg_ucabi_cpu_time();
		break;
	case OPT_CODE_UCABI_CPU_PERIOD:
		parse_time(options[index].name, &ucabi->cpu_period, optarg);
		set_setarg_ucabi_cpu_period();
		break;
	case OPT_CODE_UCABI_CPU_PERIOD_X:
		parse_time_x(options[index].name, &ucabi->cpu_period, optarg);
		set_setarg_ucabi_cpu_period();
		break;
	case OPT_CODE_UCABI_ADRS:
		*ucabipp = (void*)parse_int(options[index].name, optarg);
		break;
	default:
		return 1;
	}
	return 0;
}

/*
 * cabiid ޤ
 *
 * c Ϥ줿 cabiid ξ cabiid ǻꤵ줿ΰ
 * ͤꤷ 0 ֤ޤcabiid ʳξϤʤˤ
 * ʤ 0 ʳ֤ޤ
 */
int check_cabiid(int c, const struct option *options, int index, unsigned long *cabiid)
{
	switch( c ){
	case OPT_CODE_CABI_ID:
		*cabiid = parse_cabi_id(options[index].name, optarg);
		set_setarg_cabiid();
		break;
	default:
		return 1;
	}
	return 0;
}

/*
 * pid ޤ
 *
 * c Ϥ줿 pid ξ pid ǻꤵ줿ΰ
 * ͤꤷ 0 ֤ޤpid ʳξϤʤˤ
 * ʤ 0 ʳ֤ޤ
 */
int check_pid(int c, const struct option *options, int index, pid_t *pid)
{
	switch( c ){
	case OPT_CODE_PID:
		*pid = parse_pid_t(options[index].name, optarg);
		set_setarg_pid();
		break;
	default:
		return 1;
	}
	return 0;
}

/*
 * pgid ޤ
 *
 * c Ϥ줿 pgid ξ pgid ǻꤵ줿ΰ
 * ͤꤷ 0 ֤ޤpgid ʳξϤʤˤ
 * ʤ 0 ʳ֤ޤ
 */
int check_pgid(int c, const struct option *options, int index, pid_t *pgid)
{
	switch( c ){
	case OPT_CODE_PGID:
		*pgid = parse_pid_t(options[index].name, optarg);
		set_setarg_pgid();
		break;
	default:
		return 1;
	}
	return 0;
}

void check_setarg_cabiid(int end, char *str)
{
        if (!setarg_cabiid) {
                if (end) program_fail("cabiid not found. %s", str);
                else cabi_information("cabiid not found. %s\n", str);
        }
}
void check_setarg_pid(int end, char *str)
{
        if (!setarg_pid) {
                if (end) program_fail("pid not found. %s", str);
                else cabi_information("pid not found. %s\n", str);
        }
}
void check_setarg_pgid(int end, char *str)
{
        if (!setarg_pgid) {
                if (end) program_fail("pgid not found. %s", str);
                else cabi_information("pgid not found. %s\n", str);
        }
}
void check_setarg_ucabi_cabiid(int end, char *str)
{
        if (!setarg_ucabi.cabiid) {
                if (end) program_fail("ucabi cabiid not found. %s", str);
                else cabi_information("ucabi cabiid not found. %s\n", str);
        }
}
void check_setarg_ucabi_term_act(int end, char *str)
{
        if (!setarg_ucabi.term_act) {
                if (end) program_fail("term_act not found. %s", str);
                else cabi_information("term_act not found. %s\n", str);
        }
}
void check_setarg_ucabi_pid(int end, char *str)
{
        if (!setarg_ucabi.pid) {
                if (end) program_fail("pid not found. %s", str);
                else cabi_information("pid not found. %s\n", str);
        }
}
void check_setarg_ucabi_sig(int end, char *str)
{
        if (!setarg_ucabi.sig) {
                if (end) program_fail("sig not found. %s", str);
                else cabi_information("sig not found. %s\n", str);
        }
}
void check_setarg_ucabi_flag(int end, char *str)
{
        if (!setarg_ucabi.flag) {
                if (end) program_fail("flag not found. %s", str);
                else program_fail("flag not found. %s\n", str);
        }
}
void check_setarg_ucabi_bind_proc(int end, char *str)
{
        if (!setarg_ucabi.bind_proc) {
                if (end) program_fail("proc not found. %s", str);
                else cabi_information("proc not found. %s\n", str);
        }
}
void check_setarg_ucabi_cpu_time(int end, char *str)
{
        if (!setarg_ucabi.cpu_time) {
                if (end) program_fail("cpu_time not found. %s", str);
                else cabi_information("cpu_time not found. %s\n", str);
        }
}
void check_setarg_ucabi_cpu_period(int end, char *str)
{
        if (!setarg_ucabi.cpu_period) {
                if (end) program_fail("cpu_period not found. %s", str);
                else cabi_information("cpu_period not found. %s\n", str);
        }
}


const char *get_code_name(const struct code_table *table, int code)
{
	int i;
	static char buf[ 128 ];

	for( i = 0; table[i].name; i++ ){
		if ( table[i].code == code )
			return table[i].name;
	}
	sprintf(buf, "unknown enum code(%d:0x%x)", code, code);

	return buf;
}

const struct code_table cabi_bind_proc_type[] = {
	{ 0x0,	"BIND_NORMAL_PROC" },
	{ 0x1,	"BIND_IDLE_PROC" },
	{ -1,	NULL },
};


const struct code_table cabi_terminate_action[] = {
	{ 0x0,	"CABI_TERM_NONE" },
	{ 0x1,	"CABI_TERM_BLOCK" },
	{ 0x2,	"CABI_TERM_SIGNAL" },
	{ 0x3,	"CABI_TERM_UNKNOWN" },
	{ -1,	NULL },
};


void dump_ucabi(const struct cabi_uaccount *ucabi)
{
	cabi_information("UCABI-DUMP:cabi_id=%lu\n", ucabi->cabi_id);
	cabi_information("UCABI-DUMP:cabi_param.term_act=%s\n", get_code_name(cabi_terminate_action, ucabi->cabi_param.term_act));
	cabi_information("UCABI-DUMP:cabi_param.cabi_signal.pid=%d\n", ucabi->cabi_param.cabi_signal.pid);
	cabi_information("UCABI-DUMP:cabi_param.cabi_signal.sig=%d(%s)\n", ucabi->cabi_param.cabi_signal.sig,
					 get_signal_name(ucabi->cabi_param.cabi_signal.sig, 1));
	cabi_information("UCABI-DUMP:cabi_param.cabi_signal.flag=%d\n", ucabi->cabi_param.cabi_signal.flag);
	cabi_information("UCABI-DUMP:cabi_param.bind_proc_type=%s\n", get_code_name(cabi_bind_proc_type, ucabi->cabi_param.bind_proc_type));
	cabi_information("UCABI-DUMP:cpu_time=  %3ld.%09lds[%lds.%3ldms/%6ldus/%9ldns]\n",
                         ucabi->cpu_time.tv_sec,
                         ucabi->cpu_time.tv_nsec,
                         ucabi->cpu_time.tv_sec,
                         ucabi->cpu_time.tv_nsec/(NANOSEC/MILISEC),
                         ucabi->cpu_time.tv_nsec/(NANOSEC/MICROSEC),
                         ucabi->cpu_time.tv_nsec);
	cabi_information("UCABI-DUMP:cpu_period=%3ld.%09lds[%lds.%3ldms/%6ldus/%9ldns]\n",
                         ucabi->cpu_period.tv_sec,
                         ucabi->cpu_period.tv_nsec,
                         ucabi->cpu_period.tv_sec,
                         ucabi->cpu_period.tv_nsec/(NANOSEC/MILISEC),
                         ucabi->cpu_period.tv_nsec/(NANOSEC/MICROSEC),
                         ucabi->cpu_period.tv_nsec);
}


/*
 * Signal Table
 */
static const struct code_table signal_codes[] = {
	{ 0,         "SIG?", 	},
	{ SIGHUP,    "SIGHUP",		},/* Hangup (POSIX).  */
	{ SIGINT,    "SIGINT",		},/* Interrupt (ANSI).  */
	{ SIGQUIT,   "SIGQUIT",		},/* Quit (POSIX).  */
	{ SIGILL,    "SIGILL",		},/* Illegal instruction (ANSI).  */
	{ SIGTRAP,   "SIGTRAP",		},/* Trace trap (POSIX).  */
	{ SIGABRT,   "SIGABRT/SIGIOT",	},
	{ SIGABRT,   "SIGABRT",		},/* Abort (ANSI).  */
	{ SIGIOT,    "SIGIOT",		},/* IOT trap (4.2 BSD).  */
	{ SIGBUS,    "SIGBUS",		},/* BUS error (4.2 BSD).  */
	{ SIGFPE,    "SIGFPE",		},/* Floating-point exception (ANSI).  */
	{ SIGKILL,   "SIGKILL",		},/* Kill, unblockable (POSIX).  */
	{ SIGUSR1,   "SIGUSR1",		},/* User-defined signal 1 (POSIX).  */
	{ SIGSEGV,   "SIGSEGV",		},/* Segmentation violation (ANSI).  */
	{ SIGUSR2,   "SIGUSR2",		},/* User-defined signal 2 (POSIX).  */
	{ SIGPIPE,   "SIGPIPE",		},/* Broken pipe (POSIX).  */
	{ SIGALRM,   "SIGALRM",		},/* Alarm clock (POSIX).  */
	{ SIGTERM,   "SIGTERM",		},/* Termination (ANSI).  */
#ifdef SGISTKFLT
	{ SGISTKFLT, "SIGSTKFLT",	},/* Stack fault.  */
#endif /* SGISTKFLT */
	{ SIGCHLD,   "SIGCHLD",		},/* Same as SIGCHLD (System V).  */
	{ SIGCONT,   "SIGCONT",		},/* Continue (POSIX).  */
	{ SIGSTOP,   "SIGSTOP",		},/* Stop, unblockable (POSIX).  */
	{ SIGTSTP,   "SIGTSTP",		},/* Keyboard stop (POSIX).  */
	{ SIGTTIN,   "SIGTTIN",		},/* Background read from tty (POSIX).  */
	{ SIGTTOU,   "SIGTTOU",		},/* Background write to tty (POSIX).  */
	{ SIGURG,    "SIGURG",		},/* Urgent condition on socket (4.2 BSD).  */
	{ SIGXCPU,   "SIGXCPU",		},/* CPU limit exceeded (4.2 BSD).  */
	{ SIGXFSZ,   "SIGXFSZ",		},/* File size limit exceeded (4.2 BSD).  */
	{ SIGVTALRM, "SIGVTALRM",	},/* Virtual alarm clock (4.2 BSD).  */
	{ SIGPROF,   "SIGPROF",		},/* Profiling alarm clock (4.2 BSD).  */
	{ SIGWINCH,  "SIGWINCH",	},/* Window size change (4.3 BSD, Sun).  */
	{ SIGPOLL,   "SIGPOLL/SIGIO",	},
	{ SIGPOLL,   "SIGPOLL",		},/* Pollable event occurred (System V).  */
	{ SIGIO,     "SIGIO",		},/* I/O now possible (4.2 BSD).  */
	{ SIGPWR,    "SIGPWR",		},/* Power failure restart (System V).  */
	{ SIGSYS,    "SIGSYS",		},/* Bad system call.  */
#ifdef SIGUNUSED
	{ SIGUNUSED, "SIGUNUSED",	},
#endif /* SIGUNUSED */
};

int get_signal_code(const char *name)
{
	int i, cnt;

	if (!name) program_fail("get_signal_code");

	cnt = sizeof(signal_codes)/sizeof(struct code_table);

	for( i = 0; i < cnt; i++ ){
		if ( !strcmp(signal_codes[i].name, name) )
			return signal_codes[i].code;
	}
	program_fail("unknown error name(%s)", name);

	return -1;	// never reach
}

const char*get_signal_name(int code, int nofail)
{
	int i, cnt;

	cnt = sizeof(signal_codes)/sizeof(struct code_table);

	for( i = 0; i < cnt; i++ ){
		if ( signal_codes[i].code == code )
			return signal_codes[i].name;
	}
	if ( !nofail ){
		program_fail("unknown error code(%d)", code);
	}

	return "(unknown)";
}

/* default signal handler */
static void cabi_sighandler(int sig)
{
	fprintf(stderr, "[%d] CATCH SIGNAL %s(%d)",
		getpid(), get_signal_name(sig, 0), sig);
	if ( disp_sigtime ){
		/* ʥ֤ɽ */
		struct timeval tv;
		long long diff;
		
		gettimeofday(&tv, NULL);
		diff = timeval_diff(&disp_sigtime_prev, &tv);
		fprintf(stderr, " TIMEDIFF=%3lld.%06llds[%llds.%3lldms/%6lldus] ",
			diff / MICROSEC,
			diff % MICROSEC,
			diff / MICROSEC,
			(diff / MILISEC) - (diff / MICROSEC * MILISEC),
			diff);
		fprintf(stderr, " SIGTIME=%3ld.%06lds[%lds.%3ldms/%6ldus]",
                         tv.tv_sec,
                         tv.tv_usec,
                         tv.tv_sec,
                         tv.tv_usec/MILISEC,
			tv.tv_usec);
		gettimeofday(&disp_sigtime_prev, NULL);
	}
	fprintf(stderr, "\n");
}

/* busy signal end flag */
static volatile int busySignalEnd;


/* default busy signal handler */
void cabi_busysighandler(int sig)
{
	busySignalEnd = 0;
	
	fprintf(stderr, "[%d] CATCH SIGNAL %s(%d) [BUSY SIGNAL Handler]\n",
		getpid(), get_signal_name(sig, 0), sig);

	while( !busySignalEnd ){
		/* NOP */
	}
}

/* default busy end signal handler */
void cabi_busyendsighandler(int sig)
{
	fprintf(stderr, "[%d] CATCH SIGNAL %s(%d)  [BUSY END SIGNAL Handler]\n",
		getpid(), get_signal_name(sig, 0), sig);
	
	busySignalEnd = 1;
}

/* default null signal handler */
void cabi_nullsighandler(int sig)
{
	fprintf(stderr, "[%d] CATCH SIGNAL %s(%d)  [NULL SIGNAL Handler]\n",
		getpid(), get_signal_name(sig, 0), sig);

	*(char*)0 = 1;
}


/* regist signal handler */
int cabi_signal(int signum, void (*sighandler)(int))
{
	if (!sighandler) sighandler = cabi_sighandler;
	if (signal(signum, sighandler) < 0) program_fail("signal handler");
	return 0;
}

/* proc file */
int get_proc_info(pid_t pidno, struct procinfo *pi)
{
        int ret;
        FILE *fp;
        char filename[256];

        if (!pi) {
                program_fail("get procfs");
        }

        sprintf(filename, "/proc/%d/stat", pidno);

        memset(pi, 0, sizeof(*pi));
        fp = fopen(filename, "r");
        if ( !fp ){
                program_fail("not found pid %d", pidno);
        }

	/*
	 * ѥˤäơ'*''l'Ʊ˻ꤹ warning Ф뤿
	 * '*','l'Ʊ˻Ѥʤ褦ˤƤ
	 * λϸǻѤ뤳ȤˤʤäȤΤ else ¦˻Ĥ
	 */
#define NO_AST_AND_L_TOGETHER_IN_FORMAT
        ret = fscanf(fp,
                     "%d %s %c "        /* pid command state[RSDZTW] */
                     "%d %d "           /* ppid pgrp */
                     "%*d "             /* session */
#ifdef NO_AST_AND_L_TOGETHER_IN_FORMAT
                     "%*d %*d %*u "    /* tty_nr tpqid flags */
                     "%*u %*u %*u %*u "     /* minflt cminflt majflt cmajflt */
#else /* !NO_AST_AND_L_TOGETHER_IN_FORMAT */
                     "%*d %*d %*lu "    /* tty_nr tpqid flags */
                     "%*lu %*lu %*lu %*lu "     /* minflt cminflt majflt cmajflt */
#endif /* NO_AST_AND_L_TOGETHER_IN_FORMAT */
                     "%lu %lu %ld "     /* utime stime cutime */
#ifdef NO_AST_AND_L_TOGETHER_IN_FORMAT
                     "%*d %*d "               /* cstime priority */
                     "%*d %*d "               /* nice 0 */
                     "%*d %lu "                /* intealvalue starttime */
                     "%*u %*d %*u "  /* vsize rss rlim */
#else /* !NO_AST_AND_L_TOGETHER_IN_FORMAT */
                     "%*ld %*ld "               /* cstime priority */
                     "%*ld %*ld "               /* nice 0 */
                     "%*ld %lu "                /* intealvalue starttime */
                     "%*lu %*ld %*lu "  /* vsize rss rlim */
#endif /* NO_AST_AND_L_TOGETHER_IN_FORMAT */
                     ,
                     &pi->pid, pi->command, &pi->state,
                     &pi->ppid, &pi->pgrp,
                     &pi->utime, &pi->stime, &pi->cutime,
                     &pi->starttime
                );

        gettimeofday(&pi->tv, NULL);

        fclose(fp);

        return 0;
}

// User+SystemCPUѻ(jiffies)
int get_cpu_time(struct procinfo *pi)
{
        return pi->utime + pi->stime;
}

long long get_time_diff(struct procinfo *pi1, struct procinfo *pi2)
{
        long long timediff;

        timediff = pi2->tv.tv_sec - pi1->tv.tv_sec;
        timediff *= 1000000;
        timediff += pi2->tv.tv_usec - pi1->tv.tv_usec;

        return timediff;
}
