/*
 *  Breeze  --  An application launcher with command-line style
 *  Copyright (C) 2005, 2006, 2008 Hironao Komatsu
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/wait.h>

#include <glib/gi18n.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <X11/keysym.h>

#include "keyconfig.h"

/*
 *  Edit here to grab another key.
 *
 *  Default is "Windows key + space" (PC/AT keyboard)
 */
#define DEFAULT_KEY XK_space
#define DEFAULT_KEY_MASK Mod4Mask

static void init_keyboard(Display *dpy, KeyConfig *kconf);
static void grab_key(Display *dpy, Window root, KeyConfig *kconf);
static unsigned int get_modifiers_to_grab(const char *str);

/*
 *  Functions init_keyboard and grab_key are originally from keylaunch.c;
 *  written by Ken Lynch and Stefan Pfetzing.
 */
static void init_keyboard(Display *dpy, KeyConfig *kconf)
{
  XModifierKeymap *xmk = NULL;
  KeyCode *map;
  int m, k;

  xmk = XGetModifierMapping(dpy);
  if (xmk) {
    map = xmk->modifiermap;
    for (m = 0; m < 8; m++)
      for (k = 0; k < xmk->max_keypermod; k++, map++) {
        if (*map == XKeysymToKeycode(dpy, XK_Num_Lock))
          kconf->numLockMask = (1 << m);
        if (*map == XKeysymToKeycode(dpy, XK_Caps_Lock))
          kconf->capsLockMask = (1 << m);
        if (*map == XKeysymToKeycode(dpy, XK_Scroll_Lock))
          kconf->scrollLockMask = (1 << m);
      }
    XFreeModifiermap(xmk);
  }
}

static void grab_key(Display *dpy, Window root, KeyConfig *kconf)
{
  unsigned int mods[] = {
    0,
    kconf->numLockMask,
    kconf->capsLockMask,
    kconf->scrollLockMask,
    kconf->numLockMask | kconf->capsLockMask,
    kconf->capsLockMask | kconf->scrollLockMask,
    kconf->numLockMask | kconf->capsLockMask | kconf->scrollLockMask,
  };

  if (kconf->key_to_grab) {
    int i;

    for (i = 0; i < sizeof(mods) / sizeof(*mods); i++) {
      XGrabKey(dpy, kconf->key_to_grab, kconf->modifiers_to_grab | mods[i],
	       root, False, GrabModeAsync, GrabModeAsync); 
    }
  }
}

static unsigned int get_modifiers_to_grab(const char *str)
{
  const char *modifiers_str[] = {
    "Control", "Shift", "Mod1", "Mod2", "Mod3", "Mod4", "Mod5" };
  const int modifiers[] = {
    ControlMask, ShiftMask, Mod1Mask, Mod2Mask, Mod3Mask, Mod4Mask, Mod5Mask }; 
  const int num = sizeof(modifiers_str) / sizeof(*modifiers_str);
  int i;

  for (i = 0; i < num; i++)
    if (strcmp(str, modifiers_str[i]) == 0)
      return modifiers[i];

  return 0;
}

void kconf_initialize(KeyConfig *kconf, Display *dpy, Window root,
		      gchar *keysym_str, gchar * const *modifier_str_v)
{
  KeySym keysym = 0L;
  unsigned int modifiers = 0;

  if (keysym_str) {
    keysym = XStringToKeysym(keysym_str);
    if (keysym == NoSymbol)
      g_warning(_("unknown key: %s. ignored.\n"), keysym_str);
  }
  if (modifier_str_v) {
    gchar * const *p;

    for (p = modifier_str_v; *p; p++) {
      unsigned int mod = get_modifiers_to_grab(*p);

      if (mod == 0)
	g_warning(_("unknown modifier key: %s. ignored.\n"), *p);
      else
	modifiers |= mod;
    }
  }

  init_keyboard(dpy, kconf);

  if (keysym)
    kconf->key_to_grab = XKeysymToKeycode(dpy, keysym);
  else {
    kconf->key_to_grab = XKeysymToKeycode(dpy, DEFAULT_KEY);
    modifiers = DEFAULT_KEY_MASK;
  }

  kconf->modifiers_to_grab = modifiers;

#ifdef DEBUG
  g_debug("(key, modifier) = (%d, %d)\n",
	  kconf->key_to_grab, kconf->modifiers_to_grab);
#endif

  grab_key(dpy, root, kconf);
}
