/*
 *  Breeze  --  An application launcher with command-line style
 *  Copyright (C) 2005, 2006, 2008 Hironao Komatsu
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "mainwin.h"
#include "registwin.h"
#include "user_commands.h"
#include "commands.h"

static void main_win_class_init(MainWinClass *klass);
static void main_win_init(MainWin *self);

static void button_clicked_callback(GtkWidget *w, gpointer user_data);
static void delete_event_callback(GtkWidget *w, gpointer user_data);
static void entry_activate_callback(GtkWidget *w, gpointer user_data);

static int spawn_command(const gchar *cmd);
static void list_store_append_func(gpointer keyword, gpointer command,
				   gpointer user_data);
static GtkTreeModel *create_completion_model(void);

GType main_win_get_type()
{
  static GType type = G_TYPE_INVALID;

  if (G_UNLIKELY(type == G_TYPE_INVALID)) {
    static const GTypeInfo info = {
      sizeof(MainWinClass),
      NULL,
      NULL,
      (GClassInitFunc) main_win_class_init,
      NULL,
      NULL,
      sizeof(MainWin),
      0,
      (GInstanceInitFunc) main_win_init,
      NULL,
    };
    type = g_type_register_static(GTK_TYPE_WINDOW, "MainWin", &info, 0);
  }
  return type;
}

static GtkWindowClass *parent_class = NULL;

static void main_win_class_init(MainWinClass *klass)
{
  parent_class = g_type_class_peek_parent(klass);
}

static void main_win_init(MainWin *self)
{
  GtkTreeModel *model;
  GtkWidget *button;
  GtkWidget *hbox;
  GtkAccelGroup *accel;
  GList *focus_chain;
  const gint default_width = 240;

  GTK_WINDOW(self)->type = GTK_WINDOW_TOPLEVEL;
  gtk_window_set_title(GTK_WINDOW(self), _("Breeze"));
  gtk_widget_set_size_request(GTK_WIDGET(self), default_width, -1);
  gtk_window_set_resizable(GTK_WINDOW(self), FALSE);
  gtk_window_set_position(GTK_WINDOW(self), GTK_WIN_POS_CENTER);
  gtk_container_set_border_width(GTK_CONTAINER(self), 4);
  //gtk_window_set_decorated(GTK_WINDOW(self), FALSE);

  hbox = gtk_hbox_new(FALSE, 0);
  self->entry = gtk_entry_new();
  self->completion = gtk_entry_completion_new();
  button = gtk_button_new_with_mnemonic(_("_Cancel"));

  gtk_container_add(GTK_CONTAINER(self), hbox);
  gtk_box_pack_start(GTK_BOX(hbox), self->entry, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);

  focus_chain = NULL;
  focus_chain = g_list_append(focus_chain, self->entry);
  gtk_container_set_focus_chain(GTK_CONTAINER(hbox), focus_chain);
  g_list_free(focus_chain);

  gtk_entry_set_completion(GTK_ENTRY(self->entry), self->completion);
  gtk_entry_completion_set_inline_completion(self->completion, TRUE);
  g_object_unref(self->completion);

  model = create_completion_model();
  gtk_entry_completion_set_model(self->completion, model);
  g_object_unref(model);

  gtk_entry_completion_set_text_column (self->completion, 0);

  g_signal_connect(GTK_WIDGET(self), "delete_event",
		   G_CALLBACK(delete_event_callback), self);
  g_signal_connect(self->entry, "activate",
		   G_CALLBACK(entry_activate_callback), self);
  g_signal_connect(button, "clicked",
		   G_CALLBACK(button_clicked_callback), self);

  accel = gtk_accel_group_new();
  gtk_window_add_accel_group(GTK_WINDOW(self), accel);
  gtk_widget_add_accelerator(button, "clicked", accel, GDK_Escape, 0, 0);
}

GtkWidget *main_win_new(void)
{
  MainWin *self = g_object_new(TYPE_MAIN_WIN, NULL);

  return GTK_WIDGET(self);
}

static void button_clicked_callback(GtkWidget *w, gpointer user_data)
{
  MainWin *self = (MainWin*)user_data;
  const gchar *text = gtk_entry_get_text(GTK_ENTRY(self->entry));

  if (strlen(text) == 0) {
    gtk_widget_destroy(GTK_WIDGET(self));
    gtk_main_quit();
  }
  else
    gtk_entry_set_text(GTK_ENTRY(self->entry), "");
}

static void delete_event_callback(GtkWidget *w, gpointer user_data)
{
  MainWin *self = (MainWin*)user_data;
  gtk_widget_destroy(GTK_WIDGET(self));
  gtk_main_quit();
}

static void entry_activate_callback(GtkWidget *w, gpointer user_data)
{
  MainWin *self = (MainWin*)user_data;
  gchar *keyword = g_strdup(gtk_entry_get_text(GTK_ENTRY(self->entry)));
  gchar *keyword_to_lookup;
  gconstpointer result;

  keyword_to_lookup = g_strstrip(keyword);

  gtk_main_quit();
  gtk_widget_destroy(GTK_WIDGET(self));

  if ((result = lookup_preset_commands(keyword_to_lookup))) {
    void (*func)(void) = result;
    func();
  }
  else if ((result = lookup_user_commands(keyword_to_lookup))) {
    gchar *str = g_filename_from_utf8(result, strlen(result), NULL, NULL, NULL);

    if (str) {
      spawn_command(str);
      g_free(str);
    }
    else
      g_critical(_("failed to convert encoding\n"));
  }
  else {
    GtkWidget *dialog = regist_win_new(keyword_to_lookup);

    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
  }

  g_free(keyword);
}

static int spawn_command(const gchar *cmd)
{
  int pid = fork();

  switch (pid) {
  case 0:
    setsid();
    execlp("/bin/sh", "/bin/sh", "-c", cmd, NULL);
    g_critical(_("Failed to exec.\n"));
    exit(1);
  case -1:
    g_critical(_("Failed to fork.\n"));
    return -1;
  }

  return 0;
}

static void list_store_append_func(gpointer keyword, gpointer command,
				   gpointer user_data)
{
  GtkListStore *store = user_data;
  GtkTreeIter iter;

  gtk_list_store_append(store, &iter);
  gtk_list_store_set(store, &iter, 0, keyword, -1);
}

static GtkTreeModel *create_completion_model(void)
{
  GtkListStore *store = gtk_list_store_new(1, G_TYPE_STRING);

  commands_table_foreach(list_store_append_func, store);

  return GTK_TREE_MODEL(store);
}

gint main_win_run(MainWin *self)
{
  gtk_widget_show_all(GTK_WIDGET(self));

  /* get focus, raise and be sticky */
  gdk_window_set_keep_above(GTK_WIDGET(self)->window, TRUE);
  gtk_window_stick(GTK_WINDOW(self));
  gtk_window_present(GTK_WINDOW(self));

  gtk_main();

  return 0;
}

