/*
 *  Breeze  --  An application launcher with command-line style
 *  Copyright (C) 2005, 2006 Hironao Komatsu
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "breeze.h"
#include "config.h"

static GtkWidget *window;
static GtkWidget *entry;
static GtkEntryCompletion *completion;

static int spawn_command(const gchar *cmd)
{
  int pid = fork();

  switch (pid) {
  case 0:
    setsid();
    execlp("/bin/sh", "/bin/sh", "-c", cmd, NULL);
    fprintf(stderr, "Failed to exec.\n");
    exit(1);
  case -1:
    fprintf(stderr, "Failed to fork.\n");
    return -1;
  }

  return 0;
}

static void list_store_append_func(gpointer keyword, gpointer command,
				   gpointer user_data)
{
  GtkListStore *store = user_data;
  GtkTreeIter iter;

  gtk_list_store_append(store, &iter);
  gtk_list_store_set(store, &iter, 0, keyword, -1);
}

static GtkTreeModel *create_completion_model(void)
{
  GtkListStore *store = gtk_list_store_new(1, G_TYPE_STRING);

  commands_table_foreach(list_store_append_func, store);

  return GTK_TREE_MODEL(store);
}

static void button_clicked_callback(GtkWidget *w, gpointer user_data)
{
  const gchar *text = gtk_entry_get_text(GTK_ENTRY(entry));

  if (strlen(text) == 0) {
    gtk_widget_destroy(window);
    gtk_main_quit();
  }
  else
    gtk_entry_set_text(GTK_ENTRY(entry), "");
}

static void delete_event_callback(GtkWidget *w, gpointer user_data)
{
  gtk_widget_destroy(window);
  gtk_main_quit();
}

static void entry_activate_callback(GtkWidget *w, gpointer user_data)
{
  gchar *keyword = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
  gchar *keyword_to_lookup;
  gboolean is_empty;
  gconstpointer result;

  keyword_to_lookup = strip(keyword, &is_empty);

  /* If empty, assumes "Edit". */
  if (is_empty) {
    g_free(keyword);
    keyword = keyword_to_lookup = g_strdup("Edit");
  }

  gtk_widget_destroy(window);

  if ((result = lookup_preset_commands(keyword_to_lookup))) {
    void(*func)(void) = result;
    func();
  }
  else if ((result = lookup_user_commands(keyword_to_lookup))) {
    spawn_command(result);
    gtk_main_quit();
  }
  else
    popup_registwin(keyword_to_lookup);

  g_free(keyword);
}

static int mainwin_new(void)
{
  GtkTreeModel *model;
  GtkWidget *button;
  GtkWidget *hbox;
  GtkAccelGroup *accel;
  GList *focus_chain;

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  hbox = gtk_hbox_new(FALSE, 0);
  entry = gtk_entry_new();
  completion = gtk_entry_completion_new();
  button = gtk_button_new_with_label("Cancel");

  gtk_window_set_title(GTK_WINDOW(window), "Breeze");
  gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);
  gtk_container_set_border_width(GTK_CONTAINER(window), 4);

  gtk_container_add(GTK_CONTAINER(window), hbox);
  gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);

  focus_chain = NULL;
  focus_chain = g_list_append(focus_chain, entry);
  gtk_container_set_focus_chain(GTK_CONTAINER(hbox), focus_chain);
  g_list_free(focus_chain);

  gtk_entry_set_completion(GTK_ENTRY(entry), completion);
  gtk_entry_completion_set_inline_completion(completion, TRUE);
  g_object_unref(completion);

  model = create_completion_model();
  gtk_entry_completion_set_model(completion, model);
  g_object_unref(model);

  gtk_entry_completion_set_text_column (completion, 0);

  g_signal_connect(window, "delete_event",
		   G_CALLBACK(delete_event_callback), NULL);
  g_signal_connect(entry, "activate",
		   G_CALLBACK(entry_activate_callback), NULL);
  g_signal_connect(button, "clicked",
		   G_CALLBACK(button_clicked_callback), NULL);

  accel = gtk_accel_group_new();
  gtk_window_add_accel_group(GTK_WINDOW(window), accel);
  gtk_widget_add_accelerator(button, "clicked", accel, GDK_Escape, 0, 0);

  gtk_widget_show_all(window);

  return 0;
}

int popup_mainwin(void)
{
  mainwin_new();

  gtk_main();

  return 0;
}
