/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.annotation;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1TagMode;

/**
 * Indicates an ASN.1 tag of a type or a field.
 * 
 * <p>
 * {@code @ASN1Tag} annotation is used for providing definitions of ASN.1 tags
 * for ASN.1 types or element fields of structured types.
 * </p>
 * <p>
 * For example, following ASN.1 definition :
 * 
 * <pre>
 * Sample ::= [APPLICATION 1] SEQUENCE {
 * 	number	[0] IMPLICIT INTEGER,
 * 	bool	[1] BOOLEAN }
 * </pre>
 * 
 * can be translated as :
 * 
 * <code><pre>
 * &#064;ASN1Tag(value = 1, tagClass = ASN1TagClass.APPLICATION)
 * public class Sample extends SEQUENCE {
 * 
 * 	&#064;ASN1Element(0)
 * 	&#064;ASN1Tag(value = 0, tagMode = ASN1TagMode.IMPLICIT)
 * 	public INTEGER number;
 * 
 * 	&#064;ASN1Element(1)
 * 	&#064;ASN1Tag(1)
 * 	public BOOLEAN bool;
 * 
 * }
 * </pre></code>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1TagClass
 * @see ASN1TagMode
 */
@Target({ ElementType.TYPE, ElementType.FIELD })
@Retention(RetentionPolicy.RUNTIME)
public @interface ASN1Tag {

	/**
	 * Indicates tag number of this tag.
	 * 
	 * @return Tag number.
	 */
	public int value();

	/**
	 * Indicates tag class of this tag.
	 * 
	 * @return Tag class.
	 */
	public ASN1TagClass tagClass() default ASN1TagClass.CONTEXT_SPECIFIC;

	/**
	 * Indicates the tagging mode of this tag.
	 * 
	 * @return The tagging mode.
	 */
	public ASN1TagMode tagMode() default ASN1TagMode.DEFAULT;

}
