/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import java.util.HashMap;
import java.util.Map;

import jp.bitmeister.asn1.annotation.ASN1Anonymous;
import jp.bitmeister.asn1.annotation.ASN1BuiltIn;
import jp.bitmeister.asn1.annotation.ASN1Identifier;
import jp.bitmeister.asn1.annotation.ASN1ModuleRef;
import jp.bitmeister.asn1.annotation.ASN1Tag;
import jp.bitmeister.asn1.exception.ASN1IllegalDefinition;

/**
 * Specifications of an ASN.1 type.
 * 
 * <p>
 * An instance of this class contains static information of an ASN.1 type.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
public class TypeSpecification {

	private static final Map<Class<? extends ASN1Type>, TypeSpecification> TYPE_SPECIFICATIONS = new HashMap<Class<? extends ASN1Type>, TypeSpecification>();

	/**
	 * Creates and returns a {@code TypeSpecification} of the type.
	 * 
	 * @param type
	 *            The {@code Class} object of an ASN.1 type.
	 * @return The {@code TypeSpecification} of the type.
	 */
	static TypeSpecification getSpecification(Class<? extends ASN1Type> type) {
		if (TYPE_SPECIFICATIONS.containsKey(type)) {
			return TYPE_SPECIFICATIONS.get(type);
		}
		TypeSpecification specification = new TypeSpecification();

		// specify module.
		if (type.isAnnotationPresent(ASN1BuiltIn.class)) {
			specification.module = BuiltInModule.class;
		} else if (type.isAnnotationPresent(ASN1ModuleRef.class)) {
			specification.module = type.getAnnotation(ASN1ModuleRef.class)
					.value();
		} else if (type.isMemberClass()) {
			Class<?> enclosure = type.getDeclaringClass();
			if (ASN1Module.class.isAssignableFrom(enclosure)) {
				@SuppressWarnings("unchecked")
				Class<? extends ASN1Module> module = (Class<? extends ASN1Module>) enclosure;
				specification.module = module;
			}
			if (ASN1Type.class.isAssignableFrom(enclosure)) {
				@SuppressWarnings("unchecked")
				Class<? extends ASN1Type> enclosingType = (Class<? extends ASN1Type>) enclosure;
				specification.module = getSpecification(enclosingType).module;
			}
		}
		if (specification.module == null) {
			ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
			ex.setMessage("No ASN.1 module reference found for this type.",
					null, type, null, null);
			throw ex;
		}

		// set reference type specification.
		if (!type.isAnnotationPresent(ASN1BuiltIn.class)) {
			try {
				@SuppressWarnings("unchecked")
				Class<? extends ASN1Type> referenceType = (Class<? extends ASN1Type>) type
						.getSuperclass();
				specification.reference = getSpecification(referenceType);
			} catch (ClassCastException e) {
				ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
				ex.setMessage("Failed to cast the class to ASN1Type.", e, type,
						null, null);
				throw ex;
			}
		}

		// set identifier.
		if (!type.isAnnotationPresent(ASN1Anonymous.class)) {
			ASN1Identifier id = type.getAnnotation(ASN1Identifier.class);
			if (id != null) {
				specification.identifier = id.value();
			} else {
				specification.identifier = type.getSimpleName();
			}
		}

		// set tag.
		if (type.isAnnotationPresent(ASN1Tag.class)) {
			specification.tag = new ASN1TagValue(type, specification.module);
		}

		TYPE_SPECIFICATIONS.put(type, specification);
		return specification;
	}

	/**
	 * ASN.1 module that this type is defined.
	 */
	private Class<? extends ASN1Module> module;

	/**
	 * Identifier of this type.
	 */
	private String identifier;

	/**
	 * ASN.1 tag assigned to this type.
	 */
	private ASN1TagValue tag;

	/**
	 * Specification of the referencing type.
	 */
	private TypeSpecification reference;

	/**
	 * Instantiates a {@code TypeSpecification}.
	 */
	private TypeSpecification() {
	}

	/**
	 * Returns the default tagging mode for the ASN.1 module that this type is
	 * defined.
	 * 
	 * @return The default tagging mode.
	 */
	public ASN1TagDefault tagDefault() {
		return ASN1Module.tagDefault(module);
	}

	/**
	 * Tests if this type has own identifier.
	 * 
	 * @return {@code true} when this type has own identifier.
	 */
	public boolean hasIdentifier() {
		return identifier != null;
	}

	/**
	 * Returns the ASN.1 identifier of this type.
	 * 
	 * @return The ASN.1 identifier of this type.
	 */
	public String identifier() {
		if (!hasIdentifier()) {
			return reference.identifier();
		}
		return identifier;
	}

	/**
	 * Returns the full identifier of this type that made from ASN.1 module
	 * identifier and type identifier.
	 * 
	 * @return The full identifier of this type.
	 */
	public String fullIdentifier() {
		if (module != BuiltInModule.class) {
			return ASN1Module.identifier(module) + "." + identifier();
		}
		return identifier();
	}

	/**
	 * Tests if this type is tagged or references a tagged type.
	 * 
	 * @return {@code true} when this type or referencing types have ASN.1 tag.
	 */
	public boolean tagged() {
		if (tag != null) {
			return true;
		}
		if (reference != null) {
			return reference.tagged();
		}
		return false;
	}

	/**
	 * Returns the ASN.1 tag assigned to this type.
	 * 
	 * @return The ASN.1 tag.
	 */
	public ASN1TagValue tag() {
		return tag;
	}

	/**
	 * Tests if the tag class and the tag number matches this type.
	 * 
	 * @param tagClass
	 *            The tag class.
	 * @param tagNumber
	 *            The tag number.
	 * @return {@code true} when the tag class and the tag number matches this
	 *         type.
	 */
	public boolean matches(ASN1TagClass tagClass, int tagNumber) {
		if (tag == null) {
			return reference.matches(tagClass, tagNumber);
		}
		return tag.tagClass() == tagClass && tag.tagNumber() == tagNumber;
	}

	/**
	 * Returns the specification of the referencing type of this type.
	 * 
	 * @return The specification of the referencing type.
	 */
	public TypeSpecification reference() {
		return reference;
	}

}
