/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import java.nio.charset.Charset;
import java.text.DateFormat;
import java.util.Calendar;
import java.util.Date;

import jp.bitmeister.asn1.processor.ASN1Visitor;

/**
 * The base class for types which contains a {@code String} that represents date
 * and time.
 * 
 * <p>
 * This class provides generic interfaces and common methods for classes which
 * represents date and time types.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
public abstract class TimeType extends StringType {

	/**
	 * Converts the {@code Date} object to a formatted {@code String} that
	 * represents date and time, and set it to this instance.
	 * 
	 * @param value
	 *            The {@code Date} to be assigned.
	 */
	public void set(Date value) {
		set(form().format(value));
	}

	/**
	 * Returns a {@code Date} converted from the value of this instance.
	 * 
	 * @return A {@code Date}.
	 */
	public Date date() {
		String time;
		String differential;
		String value = stringValue();
		if (value.endsWith("Z")) {
			// represents UTC time.
			differential = "+0000";
			time = value.substring(0, value.length() - 1);
		} else if (value.indexOf("+") < 0 && value.indexOf("-") < 0) {
			// represents the local time
			differential = String.format("%tz", Calendar.getInstance());
			time = value;
		} else {
			// represents the local time, and the time differential
			differential = value.substring(value.length() - 5, value.length());
			time = value.substring(0, value.length() - 5);
		}
		return parseDate(time, differential);
	}

	/**
	 * Returns the {@code DateFormat} used for formatting the date and time.
	 * 
	 * @return The date format.
	 */
	public abstract DateFormat form();

	@Override
	public Charset charset() {
		return Charset.forName("US-ASCII");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.type.StringType#accept(jp.bitmeister.asn1.processor
	 * .ASN1Visitor)
	 */
	@Override
	public <R, E extends Throwable> R accept(ASN1Visitor<R, E> visitor) throws E {
		return visitor.visit(this);
	}
	
	/**
	 * Parses and converts the strings represents a set of date, time and time
	 * defferential to a {@code Date}.
	 * 
	 * @param time
	 *            The {@code String} represents a set of date.
	 * @param differential
	 *            The {@code String} represents a time differential.
	 * @return A {@code Date} object that converted from the time and the
	 *         differential.
	 */
	protected abstract Date parseDate(String time, String differential);

}
