/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

/**
 * The base class for primitive ASN.1 types.
 * 
 * <p>
 * This class provides common methods and a value field for classes which
 * represents simple, unconstructive ASN.1 types.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
public abstract class PrimitiveType<T> extends ASN1Type {

	/**
	 * The value assigned to this data.
	 */
	private T value;

	/**
	 * Sets the value to this data.
	 * 
	 * @param value
	 *            The value to be set.
	 */
	public void set(T value) {
		this.value = value;
	}

	/**
	 * Returns the value set to this data.
	 * 
	 * @return The value.
	 */
	public T value() {
		return value;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#clear()
	 */
	@Override
	public void clear() {
		value = null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#hasValue()
	 */
	@Override
	public boolean hasValue() {
		return value != null;
	}

	/**
	 * Creates and returns a copy of the value set to this data.
	 * 
	 * @return A copy of the value.
	 */
	protected T cloneValue() {
		return value;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#valueEquals(java.lang.Object)
	 */
	@Override
	public boolean valueEquals(Object other) {
		if (other instanceof PrimitiveType<?>) {
			Object otherValue = ((PrimitiveType<?>) other).value();
			if (value != null) {
				return value.equals(otherValue);
			}
			return otherValue == null;
		}
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#hashCode()
	 */
	@Override
	public int hashCode() {
		if (!hasValue()) {
			return 0;
		}
		return value.hashCode();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#clone()
	 */
	@Override
	public Object clone() {
		@SuppressWarnings("unchecked")
		PrimitiveType<T> clone = ASN1Type.instantiate(getClass());
		clone.set(cloneValue());
		return clone;
	}

}
