/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

/**
 * Manager of all ASN.1 modules.
 * 
 * <p>
 * This class manages all of the ASN.1 modules which are sub-class of
 * {@code ASN1Module} and provides {@code instantiate} method that instantiates
 * an ASN.1 data from an ASN.1 tag or identifier.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1Module
 */
public class ASN1Modules {

	/**
	 * The manager of ASN.1 modules.
	 */
	private static ASN1Modules instance;

	static {
		instance = new ASN1Modules();
	}

	/**
	 * Instantiates an ASN.1 data specified by the ASN.1 tag and the tag class.
	 * 
	 * @param tagClass
	 *            The tag class.
	 * @param tagNumber
	 *            The tag number.
	 * @return An instance of ASN.1 data.
	 */
	public static ASN1Type instantiate(ASN1TagClass tagClass, int tagNumber) {
		switch (tagClass) {
		case UNIVERSAL:
			return instance.universalModule.instantiate(tagClass, tagNumber);
		case APPLICATION:
		case PRIVATE:
			for (Entry<Class<? extends ASN1Module>, ASN1Module> e : instance.definedModules
					.entrySet()) {
				ASN1Type data = e.getValue().instantiate(tagClass, tagNumber);
				if (data != null) {
					return data;
				}
			}
			break;
		}
		return new UnknownType(tagClass, tagNumber);
	}

	/**
	 * Instantiates a built-in ASN.1 data specified by the type identifier.
	 * 
	 * @param typeIdentifier
	 *            The identifier of a type.
	 * @return An instance of built-in ASN.1 data.
	 */
	public static ASN1Type instanciate(String typeIdentifier) {
		return instance.universalModule.instantiate(typeIdentifier);
	}

	/**
	 * Instantiates an ASN.1 data specified by the identifier of module and
	 * type.
	 * 
	 * @param moduleIdentifier
	 *            The identifier of a module.
	 * @param typeIdentifier
	 *            The identifier of a type.
	 * @return An instance of ASN.1 data.
	 */
	public static ASN1Type instantiate(String moduleIdentifier,
			String typeIdentifier) {
		for (Entry<Class<? extends ASN1Module>, ASN1Module> e : instance.definedModules
				.entrySet()) {
			if (e.getValue().identifier().equals(moduleIdentifier)) {
				return e.getValue().instantiate(typeIdentifier);
			}
		}
		return null;
	}

	/**
	 * Registers the ASN.1 module to the manager.
	 * 
	 * @param module
	 *            The ASN.1 module to be registered.
	 */
	public static void using(ASN1Module module) {
		instance.definedModules.put(module.getClass(), module);
	}

	private ASN1Module universalModule = new BuiltInModule();

	private Map<Class<? extends ASN1Module>, ASN1Module> definedModules = new HashMap<Class<? extends ASN1Module>, ASN1Module>();

	/**
	 * Instantiates an {@code ASN1Modules}.
	 */
	private ASN1Modules() {
	}

}
