/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.stat.handler;

import java.io.File;
import java.io.IOException;
import java.util.List;

import org.xml.sax.SAXException;

import benten.cat.glossary.core.BentenGlossaryEngine;
import benten.cat.stat.CatStatPlugin;
import benten.cat.stat.model.GlossaryHitRateModel;
import benten.core.model.BentenXliff;
import benten.core.text.Strings;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * 用語集ヒット率取得ハンドラー。
 * <p>
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳統計取得機能: 用語対訳集ヒット率ビュー」に対応します。
 *
 * @author KASHIHARA Shinji
 */
public class ShowGlossaryHitRateHandler extends AbstractStatHandler<GlossaryHitRateModel> {

	@Override
	protected GlossaryHitRateModel processFile(final File file) throws IOException, SAXException {
		final GlossaryHitRateModel model = new GlossaryHitRateModel();
		model.setFile(file);
		final BentenXliff xliff = BentenXliff.loadInstance(file);
		final List<BlancoXliffTransUnit> unitList = xliff.getAllTransUnitList();
		for (final BlancoXliffTransUnit unit : unitList) {
			final String source = unit.getSource();
			if (source != null) {
				setModel(source, model);
			}
		}
		return model;
	}

	/**
	 * モデルの値をセット。
	 * @param source ソース
	 * @param model モデル
	 */
	protected void setModel(final String source, final GlossaryHitRateModel model) {
		final BentenGlossaryEngine glossaryEngine = getGlossaryEngine();
		final String[] words = Strings.splitByWhitespace(source);
		int hit = 0;
		for (final String word : words) {
			if (glossaryEngine.search(word).size() > 0) {
				hit++;
			}
		}
		final int rate = words.length == 0 ? 0 : 100 * hit / words.length;
		if (rate >= 75) {
			model.setHit75(model.getHit75() + 1);
		} else if (rate >= 50) {
			model.setHit50(model.getHit50() + 1);
		} else if (rate >= 25) {
			model.setHit25(model.getHit25() + 1);
		} else {
			model.setHitUnder25(model.getHitUnder25() + 1);
		}
	}

	/**
	 * 用語集エンジンを取得します。
	 * @return 用語集エンジン
	 */
	protected BentenGlossaryEngine getGlossaryEngine() {
		return CatStatPlugin.getGlossaryEngine();
	}
}
