/*******************************************************************************
 * Copyright (c) 2010 IGA Tosiki.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.engine.text;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import benten.core.model.HelpTransUnitId;
import benten.twa.filter.core.BentenTwaFilterEngine;
import benten.twa.filter.model.SentencePartitionList;
import benten.twa.process.BentenProcessResultInfo;
import blanco.commons.util.BlancoFileUtil;
import blanco.xliff.valueobject.BlancoXliff;
import blanco.xliff.valueobject.BlancoXliffBody;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffTarget;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * TEXT を処理します。
 * 
 * @author IGA Tosiki
 */
public class BentenTwaFilterTextEngine implements BentenTwaFilterEngine {
	/**
	 * 連続する改行をあらわす正規表現。
	 */
	protected static final Pattern CONTINUED_NEWLINE_PATTERN = Pattern.compile("\\s*\r\n\\s*\r\n|\\s*\n\\s*\n|\\s*\r\\s*\r"); //$NON-NLS-1$

	/**
	 * {@inheritDoc}
	 */
	public boolean canProcess(final File file) {
		if (file.getName().toLowerCase().endsWith(".utf8") || file.getName().toLowerCase().endsWith(".utf-8")) { //$NON-NLS-1$ //$NON-NLS-2$
			return true;
		} else {
			return false;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void extractXliff(final File sourceFile, final BlancoXliff xliff, final HelpTransUnitId id,
			final BentenProcessResultInfo resultInfo) throws IOException {
		final BlancoXliffFile xliffFile = xliff.getFileList().get(0);
		final BlancoXliffBody body = xliffFile.getBody();

		// テキストを表現します。
		xliffFile.setDatatype("x-text"); //$NON-NLS-1$

		final byte[] bytesFile = BlancoFileUtil.file2Bytes(sourceFile);
		String text = new String(bytesFile, "UTF-8"); //$NON-NLS-1$

		// 連続する改行が 2 回以上続いたら翻訳単位を分割する。
		for (;;) {
			final Matcher matcher = CONTINUED_NEWLINE_PATTERN.matcher(text);
			if (matcher.find() == false) {
				break;
			}

			final String head = text.substring(0, matcher.start());
			// text の内容を更新します。
			text = text.substring(matcher.end());

			createTransUnits(body, id, head, resultInfo);
		}

		// 分割後の最後の行を処理します。
		if (text.length() > 0) {
			createTransUnits(body, id, text, resultInfo);
		}
	}

	/**
	 * 与えられたテキストを処理します。
	 * 
	 * @param body XLIFF の body 要素モデル
	 * @param id trans-unit の id 属性モデル
	 * @param source source 要素の値
	 * @param resultInfo 翻訳結果情報
	 */
	void createTransUnits(final BlancoXliffBody body, final HelpTransUnitId id, final String source,
			BentenProcessResultInfo resultInfo) {
		final SentencePartitionList sentenceList = new SentencePartitionList(source);
		if (sentenceList.size() > 1) {
			for (final String sentence : sentenceList) {
				id.incrementSubSeq(sentenceList.size());
				createTransUnit(body, id, sentence, resultInfo);
			}
		} else {
			createTransUnit(body, id, source, resultInfo);
		}
		id.incrementSeq();
	}

	/**
	 * trans-unit の作成。
	 * 
	 * <UL>
	 * <LI>このメソッドは HTML -> XLIFF で利用されます。
	 * </UL>
	 * 
	 * @param body XLIFF の body 要素モデル
	 * @param helpId trans-unit の id 属性モデル
	 * @param source source 要素の値
	 * @param resultInfo 翻訳結果情報
	 */
	private void createTransUnit(final BlancoXliffBody body, final HelpTransUnitId helpId, final String source,
			BentenProcessResultInfo resultInfo) {
		resultInfo.setUnitCount(resultInfo.getUnitCount() + 1);

		final BlancoXliffTransUnit unit = new BlancoXliffTransUnit();
		unit.setId(helpId.toString());
		unit.setSource(source);
		body.getTransUnitList().add(unit);
	}

	/**
	 * {@inheritDoc}
	 */
	public byte[] mergeXliff(final File sourceFile, final BlancoXliff xliff, final BentenProcessResultInfo resultInfo)
			throws IOException {

		final Iterator<BlancoXliffTransUnit> units = xliff.getFileList().get(0).getBody().getTransUnitList().iterator();

		final StringBuilder target = new StringBuilder();

		final byte[] bytesFile = BlancoFileUtil.file2Bytes(sourceFile);
		String text = new String(bytesFile, "UTF-8"); //$NON-NLS-1$

		// 連続する改行が 2 回以上続いたら翻訳単位を分割する。
		for (;;) {
			final Matcher matcher = CONTINUED_NEWLINE_PATTERN.matcher(text);
			if (matcher.find() == false) {
				break;
			}

			final String skip = text.substring(matcher.start(), matcher.end());
			text = text.substring(matcher.end());

			target.append(mergeTextInner(units, resultInfo));

			target.append(skip);
		}

		// 分割後の最後の行を処理します。
		if (text.length() > 0) {
			target.append(mergeTextInner(units, resultInfo));
		}

		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
		outStream.write(target.toString().getBytes("UTF-8"));//$NON-NLS-1$
		outStream.flush();
		byte[] bytes = outStream.toByteArray();

		return bytes;
	}

	String mergeTextInner(final Iterator<BlancoXliffTransUnit> units, final BentenProcessResultInfo resultInfo) {
		final SentencePartitionList targetList = new SentencePartitionList();
		while (units.hasNext()) {
			resultInfo.setUnitCount(resultInfo.getUnitCount() + 1);

			final BlancoXliffTransUnit unit = units.next();
			final BlancoXliffTarget t = unit.getTarget();

			if (t != null && t.getTarget() != null && !t.getTarget().equals("")) { //$NON-NLS-1$
				targetList.add(t.getTarget());
			} else {
				targetList.add(unit.getSource());
			}

			if (!HelpTransUnitId.hasContinue(unit.getId())) {
				break;
			}
		}
		return targetList.join();
	}
}
