/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.glossary.core;

import java.util.ArrayList;
import java.util.List;

import benten.cat.glossary.core.messages.BentenGlossaryDriverManagerMessages;

/**
 * Benten 用語集ドライバーを管理するクラス。
 *
 * <UL>
 * <LI>Benten に登録済みの用語集ドライバーを管理するためのクラスです。
 * <LI>Benten 用語集エンジンは、それ自身の実装において BentenGlossaryDriverManager に自分自身を登録するような実装をおこないます。
 * </UL>
 *
 * @author IGA Tosiki
 */
public class BentenGlossaryDriverManager {
	/**
	 * 用語集ドライバー・マネージャのためのメッセージ。
	 */
	protected static final BentenGlossaryDriverManagerMessages fMsg = new BentenGlossaryDriverManagerMessages();

	/**
	 * 登録されている用語集ドライバーのリスト。
	 */
	protected static final List<BentenGlossaryDriver> fDriverList = new ArrayList<BentenGlossaryDriver>();

	/**
	 * 用語集ドライバーの登録。
	 *
	 * <UL>
	 * <LI>用語集ドライバーを登録します。
	 * </UL>
	 *
	 * @param driver 登録したい用語集ドライバー。
	 */
	public static void registerDriver(final BentenGlossaryDriver driver) {
		if (driver == null) {
			throw new IllegalArgumentException(fMsg.getRegisterDriverIllegalArgument1());
		}
		if (driver.getName() == null) {
			throw new IllegalArgumentException(fMsg.getRegisterDriverIllegalArgument2());
		}

		for (final BentenGlossaryDriver look : fDriverList) {
			if (look.getName().equals(driver.getName())) {
				// すでに登録済みのドライバーです。
				// 登録処理をせずに離脱します。
				return;
			}
		}

		fDriverList.add(driver);
	}

	/**
	 * 用語集ドライバーの登録解除。
	 * <UL>
	 * <LI>用語集ドライバーを登録解除します。
	 * <LI>登録を解除したドライバーは、Class.forName による static ブロックでの登録はできなくなることに注意してください。
	 * </UL>
	 *
	 * @param driver 登録解除したい用語集ドライバー。
	 */
	public static void deregisterDriver(final BentenGlossaryDriver driver) {
		if (driver == null) {
			throw new IllegalArgumentException(fMsg.getDeregisterDriverIllegalArgument1());
		}
		if (driver.getName() == null) {
			throw new IllegalArgumentException(fMsg.getDeregisterDriverIllegalArgument2());
		}

		for (final BentenGlossaryDriver look : fDriverList) {
			if (look.getName().equals(driver.getName())) {
				// 該当するドライバーが発見できました。
				// 除去のうえ離脱します。
				fDriverList.remove(look);
				return;
			}
		}
	}

	/**
	 * 指定された名前の用語集ドライバーの取得。
	 *
	 * <UL>
	 * <LI>指定された名前の用語集ドライバーを取得します。
	 * </UL>
	 *
	 * @param driverName ドライバー名
	 * @return 用語集ドライバー。
	 */
	public static BentenGlossaryDriver getDriver(final String driverName) {
		if (driverName == null) {
			throw new IllegalArgumentException(fMsg.getGetDriverIllegalArgument1());
		}

		for (final BentenGlossaryDriver driver : fDriverList) {
			if (driver.getName().equals(driverName)) {
				return driver;
			}
		}

		return null;
	}

	/**
	 * 登録されている用語集ドライバーの一覧を取得します。
	 *
	 * <UL>
	 * <LI>登録されている用語集ドライバーの一覧を取得します。
	 * </UL>
	 *
	 * @return 用語集ドライバーの配列。
	 */
	public static BentenGlossaryDriver[] getDrivers() {
		final BentenGlossaryDriver[] drivers = new BentenGlossaryDriver[fDriverList.size()];
		for (int index = 0; index < fDriverList.size(); index++) {
			drivers[index] = fDriverList.get(index);
		}

		return drivers;
	}
}
