/*******************************************************************************
 * Copyright (c) 2010 IGA Tosiki.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
/*
 * Copyright (C) 2010 IGA Tosiki.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package benten.twa.filter.engine.po;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import benten.core.model.HelpTransUnitId;
import benten.twa.filter.core.BentenTwaFilterProcessor;
import benten.twa.process.BentenProcessResultInfo;
import blanco.commons.util.BlancoFileUtil;
import blanco.commons.util.BlancoStringUtil;
import blanco.xliff.BlancoXliffParser;
import blanco.xliff.valueobject.BlancoXliff;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffNote;
import blanco.xliff.valueobject.BlancoXliffTarget;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * PO ファイルの処理。
 * 
 * @author IGA Tosiki
 * @see "http://www.gnu.org/software/hello/manual/gettext/PO-Files.html"
 */
public class BentenTwaFilterPoProcessor implements BentenTwaFilterProcessor {

	/**
	 * {@inheritDoc}
	 */
	public boolean canProcess(final File file) {
		final String fileName = file.getName().toLowerCase();
		if (fileName.endsWith(".po")) { //$NON-NLS-1$
			return true;
		} else {
			return false;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void extractXliffFromSource(final File sourceFile, final BlancoXliffFile xliffFile,
			final HelpTransUnitId id, final BentenProcessResultInfo resultInfo) throws IOException {
		xliffFile.setDatatype("po"); //$NON-NLS-1$

		final byte[] bytesFile = BlancoFileUtil.file2Bytes(sourceFile);

		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();

		final List<SimplePOEntry> entryList = new SimplePOReader().process(new ByteArrayInputStream(bytesFile));

		BlancoXliffTransUnit unit = new BlancoXliffTransUnit();

		String pushedPlural = "";
		for (SimplePOEntry entry : entryList) {
			final String literal = getLiteral(entry.fStringLiteralList);
			String value = entry.fCommand;

			if ("x-benten-whitespace".equals(entry.fCommand)) {
				if (unit.getSource() != null) {
					addTransUnit(xliffFile, id, resultInfo, unit);

					// 次の翻訳単位を生成。
					unit = new BlancoXliffTransUnit();
				}
			} else if ("msgid".equals(value)) {
				unit.setSource(unescapeString(literal));
			} else if ("msgid_plural".equals(value)) {
				pushedPlural = pushedPlural + unescapeString(literal);
			} else if (value.startsWith("msgstr")) {
				if (value.equals("msgstr") || value.equals("msgstr[0]")) {
					// そのままOK。						
				} else {

					addTransUnit(xliffFile, id, resultInfo, unit);

					final BlancoXliffTransUnit oldUnit = unit;

					// 次の翻訳単位を生成。
					unit = new BlancoXliffTransUnit();
					// source などを引継ぎ。
					unit.setSource(pushedPlural);
					// TODO source 以外の引継ぎ

					{
						final BlancoXliffNote note = new BlancoXliffNote();
						unit.getNoteList().add(note);
						note.setFrom("benten-filter-plural");
						note.setText(value);
					}
				}

				if (unit.getTarget() == null) {
					unit.setTarget(new BlancoXliffTarget());
				}
				if (unit.getTarget().getTarget() == null) {
					unit.getTarget().setTarget("");
				}
				unit.getTarget().setTarget(unit.getTarget().getTarget() + unescapeString(literal));
			} else if ("msgctxt".equals(value)) {
				final BlancoXliffNote note = new BlancoXliffNote();
				unit.getNoteList().add(note);
				note.setFrom("benten-filter-msgctxt");
				note.setText(unescapeString(literal));
			} else {
				throw new IllegalArgumentException("PO 処理中に不明な id[" + value + "]が渡されました。");
			}

			for (String comment : entry.fCommentList) {
				final BlancoXliffNote note = new BlancoXliffNote();
				unit.getNoteList().add(note);
				note.setFrom("benten-filter-comment");
				note.setText(comment);
			}
		}

		if (unit.getSource() != null) {
			addTransUnit(xliffFile, id, resultInfo, unit);
		}
	}

	private String getLiteral(List<String> list) {
		StringBuilder strbuf = new StringBuilder();
		for (String literal : list) {
			strbuf.append(literal);
		}
		return strbuf.toString();
	}

	private void addTransUnit(final BlancoXliffFile xliffFile, final HelpTransUnitId id,
			final BentenProcessResultInfo resultInfo, final BlancoXliffTransUnit unit) {
		xliffFile.getBody().getTransUnitList().add(unit);
		resultInfo.setUnitCount(resultInfo.getUnitCount() + 1);
		unit.setId(id.toString());
		id.incrementSeq();

		// status の設定
		if (unit.getTarget() != null && unit.getTarget().getTarget() != null) {
			if (unit.getTarget().getTarget().length() == 0) {
				unit.getTarget().setState("new");
			} else {
				unit.getTarget().setState("translated");
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public byte[] mergeXliffAndSource(final File sourceFile, final File xliffFile,
			final BentenProcessResultInfo resultInfo) throws IOException {
		final BlancoXliff xliff = new BlancoXliffParser().parse(xliffFile);
		if (xliff.getFileList().size() == 0) {
			// XLIFF の内容が取得できないケースは、原則としてありえないものと考えます。
			throw new IllegalArgumentException("An unexpected situation has occurred. XLIFF format is invalid."); //$NON-NLS-1$
		}

		final byte[] bytesFile = BlancoFileUtil.file2Bytes(sourceFile);

		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();

		final List<SimplePOEntry> originalEntryList = new SimplePOReader().process(new ByteArrayInputStream(bytesFile));
		int entryIndex = 0;

		for (BlancoXliffFile file : xliff.getFileList()) {
			for (BlancoXliffTransUnit unit : file.getBody().getTransUnitList()) {
				boolean isFirstEmptyStringLiteralExist = false;

				for (entryIndex++;; entryIndex++) {
					if (originalEntryList.get(entryIndex).fCommand.startsWith("msgstr")) {
						// 該当する msgstr を発見しました。
						break;
					}
				}

				if (unit.getTarget() != null && unit.getTarget().getTarget() != null) {
					// 展開
					final List<String> literalLines = escapeString(BlancoStringUtil.null2Blank(unit.getTarget()
							.getTarget()));

					final StringBuilder strbufCompareTransUnit = new StringBuilder();
					for (String line : literalLines) {
						strbufCompareTransUnit.append(line);
					}

					final StringBuilder strbufComparePO = new StringBuilder();
					final SimplePOEntry entry = originalEntryList.get(entryIndex);
					for (String line : entry.fStringLiteralList) {
						strbufComparePO.append(line);
					}

					if (strbufCompareTransUnit.toString().equals(strbufComparePO.toString())) {
						// 何もしません。
						// これは、取り込んだままの状況だからです。
					} else {
						entry.fStringLiteralList.clear();

						boolean isFirstTarget = true;

						for (String line : literalLines) {
							if (isFirstTarget) {
								isFirstTarget = false;

								if (isFirstEmptyStringLiteralExist) {
									if (line.length() > 0) {
										// po ファイルの元の空文字列による空行がある場合があります。これを再現するための特殊なコードです。
										entry.fStringLiteralList.add("");
									}
								}
							}

							entry.fStringLiteralList.add(line);
						}
					}
				}
			}
		}

		new SimplePOWriter().process(originalEntryList, outStream);

		return outStream.toByteArray();
	}

	/**
	 * 文字列のエスケープを解除します。
	 * 
	 * @param input
	 * @return
	 * @throws IOException
	 */
	String unescapeString(String input) throws IOException {
		StringReader reader = new StringReader(input);
		StringWriter writer = new StringWriter();

		for (;;) {
			reader.mark(1);
			final int iRead = reader.read();
			if (iRead < 0) {
				break;
			}
			final char cRead = (char) iRead;
			switch (cRead) {
			case '\\':
				reader.mark(1);
				final int iRead2 = reader.read();
				if (iRead2 < 0) {
					break;
				}
				final char cRead2 = (char) iRead2;
				switch (cRead2) {
				case 'n':
					writer.write("\n");
					break;
				case 'r':
					writer.write("\r");
					break;
				case 't':
					writer.write("\t");
					break;
				case '\\':
				case '"':
					writer.write(cRead2);
					break;
				default:
					reader.reset();
					writer.write(cRead2);
					break;
				}
				break;
			default:
				writer.write(cRead);
				break;
			}
		}

		writer.flush();
		return writer.toString();
	}

	/**
	 * 文字列をエスケープします。
	 * @param input
	 * @return
	 * @throws IOException
	 */
	List<String> escapeString(String input) throws IOException {
		List<String> result = new ArrayList<String>();
		StringReader reader = new StringReader(input);
		StringWriter writer = new StringWriter();

		for (;;) {
			reader.mark(1);
			final int iRead = reader.read();
			if (iRead < 0) {
				break;
			}
			final char cRead = (char) iRead;
			switch (cRead) {
			case '\\':
			case '"':
				writer.write("\\" + cRead);
				break;
			case '\t':
				writer.write("\\t");
				break;
			case '\n':
			case '\r':
				if (cRead == '\n') {
					writer.write("\\n");
				} else {
				}
				// 行変
				writer.flush();
				result.add(writer.toString());
				writer = new StringWriter();
				break;
			default:
				writer.write(cRead);
				break;
			}
		}

		writer.flush();
		final String line = writer.toString();
		if (result.size() == 0 || line.length() > 0) {
			result.add(writer.toString());
		}

		return result;
	}
}
