/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.core.model;

import java.io.File;
import java.util.LinkedList;
import java.util.List;

import benten.core.BentenConstants;
import benten.core.model.messages.BentenCoreModelMessages;
import blanco.commons.util.BlancoStringUtil;
import blanco.tmx.BlancoTmxParser;
import blanco.tmx.BlancoTmxSerializer;
import blanco.tmx.valueobject.BlancoTmx;
import blanco.tmx.valueobject.BlancoTmxTu;

/**
 * TMX モデルとファイルを紐付けて保持するクラス。
 * @author IGA Tosiki
 */
public class BentenTmx {
	/**
	 * Benten コア、モデルのためのメッセージ。
	 */
	protected static final BentenCoreModelMessages fMsg = new BentenCoreModelMessages();

	/**
	 * 指定されたファイルを BentenTmx としてロード。
	 * @param file ファイル
	 * @return BentenTmx
	 */
	public static BentenTmx loadInstance(final File file) {
		final BlancoTmx tmx = new BlancoTmxParser().parse(file);
		if (BlancoStringUtil.null2Blank(tmx.getVersion()).startsWith("1.4") == false) { //$NON-NLS-1$
			throw new IllegalArgumentException(fMsg.getBentenTmxMsg001(file.getName(), tmx.getVersion()));
		}
		return new BentenTmx(file, tmx);
	}

	/**
	 * 指定されたファイルの新規 BentenTmx インスタンスを作成。
	 * @param file ファイル
	 * @param srclang 入力言語。
	 * @return BentenTmx
	 */
	public static BentenTmx newInstance(final File file, final String srclang) {
		final BlancoTmx tmx = new BlancoTmx();
		tmx.getHeader().setSrclang(srclang);
		tmx.getHeader().setAdminlang("en-US"); //$NON-NLS-1$
		return new BentenTmx(file, tmx);
	}

	/** 処理対象ファイル。 */
	private final File fFile;
	/** 処理対象 TMX オブジェクト。 */
	private final BlancoTmx fBlancoTmx;

	private BentenTmx(final File file, final BlancoTmx blancoTmx) {
		fFile = file;
		fBlancoTmx = blancoTmx;

		// デフォルト値にプロジェクト名やバージョンを設定します。
		blancoTmx.getHeader().setCreationtool(BentenConstants.PROJECT_NAME_LOWER);
		blancoTmx.getHeader().setCreationtoolversion(BentenConstants.VERSION);
	}

	/**
	 * BlancoTmxTu リストの取得。
	 * @return BlancoTmxTu リスト
	 */
	public List<BlancoTmxTu> getTuList() {
		return fBlancoTmx.getBody().getTuList();
	}

	/**
	 * BlancoTmxTu の作成。<br>
	 * 作成された BlancoTmxTu はこのインスタンスに追加されます。
	 * @return BlancoTmxTu
	 */
	public BlancoTmxTu createBlancoTmxTu() {
		final BlancoTmxTu tu = new BlancoTmxTu();
		getTuList().add(tu);
		return tu;
	}

	/**
	 * ファイルへ保管。
	 */
	public void save() {
		final List<BlancoTmxTu> newFileList = new LinkedList<BlancoTmxTu>();
		for (final BlancoTmxTu tu : getTuList()) {
			if (tu.getTuvList().size() > 0) {
				// tuv の存在するもののみ書き込み対象とします。
				newFileList.add(tu);
			}
		}

		fBlancoTmx.getBody().setTuList(newFileList);
		new BlancoTmxSerializer().serialize(fBlancoTmx, fFile);
	}
}
