/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved.
 *******************************************************************************/
package benten.core.io;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;

import benten.core.io.messages.BentenCoreFilesMessages;

/**
 * ファイル・ユーティリティー。
 *
 * <UL>
 * <LI>このクラスは、Apache Commons IO (ver1.4) の org.apache.commons.io パッケージに
 * 含まれる IOUtils および FileUtils クラスに由来するものです。
 * </UL>
 *
 * @author KASHIHARA Shinji
 */
public class Files {
	/**
	 * Benten コア、ファイルのためのメッセージ。
	 */
	protected static final BentenCoreFilesMessages fMsg = new BentenCoreFilesMessages();

	/**
	 * デフォルト・バッファ・サイズ。
	 */
	private static final int DEFAULT_BUFFER_SIZE = 8192;

	/**
	 * 隠されたコンストラクター。
	 *
	 * <UL>
	 * <LI>このクラスはユーティリティ・クラスです。
	 * <LI>インスタンス生成を抑制するために、コンストラクターは private で修飾されています。
	 * </UL>
	 */
	private Files() {
	}

	/**
	 * ファイルを読み込みバイト配列へ変換。
	 * @param file ファイル
	 * @return バイト配列
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static byte[] readFileToByteArray(final File file) throws IOException {
		InputStream in = null;
		try {
			in = openInputStream(file);
			return toByteArray(in);
		} finally {
			closeQuietly(in);
		}
	}

	/**
	 * ファイルを読み込み文字列へ変換。
	 * @param file ファイル
	 * @return 文字列
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static String readFileToString(final File file) throws IOException {
		return readFileToString(file, null);
	}

	/**
	 * ファイルを読み込み文字列へ変換。
	 * @param file ファイル
	 * @param encoding エンコーディング
	 * @return 文字列
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static String readFileToString(final File file, final String encoding) throws IOException {
		InputStream in = null;
		try {
			in = openInputStream(file);
			return toString(in, encoding);
		} finally {
			closeQuietly(in);
		}
	}

	/**
	 * 入力ストリームをバイト配列へ変換。
	 * @param input 入力
	 * @return バイト配列
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static byte[] toByteArray(final InputStream input) throws IOException {
		final ByteArrayOutputStream output = new ByteArrayOutputStream();
		copy(input, output);
		return output.toByteArray();
	}

	/**
	 * 入力ストリームを文字列へ変換。
	 * @param input 入力
	 * @param encoding エンコーディング
	 * @return 文字列
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static String toString(final InputStream input, final String encoding) throws IOException {
		final StringWriter sw = new StringWriter();
		copy(input, sw, encoding);
		return sw.toString();
	}

	/**
	 * 入力ストリームのオープン。
	 * @param file ファイル
	 * @return ファイル入力ストリーム
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static FileInputStream openInputStream(final File file) throws IOException {
		if (file.exists()) {
			if (file.isDirectory()) {
				throw new IOException(fMsg.getOpenInputStreamIllegalArgument1(file.getName()));
			}
			if (file.canRead() == false) {
				throw new IOException(fMsg.getOpenInputStreamIllegalArgument2(file.getName()));
			}
		} else {
			throw new FileNotFoundException(fMsg.getOpenInputStreamIllegalArgument3(file.getName()));
		}
		return new FileInputStream(file);
	}

	/**
	 * バイト配列をファイルへ書き込み。
	 * @param file ファイル
	 * @param data データ
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void writeByteArrayToFile(final File file, final byte[] data) throws IOException {
		OutputStream out = null;
		try {
			out = openOutputStream(file);
			out.write(data);
		} finally {
			closeQuietly(out);
		}
	}

	/**
	 * 文字列をファイルへ書き込み
	 * @param file ファイル
	 * @param data データ
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void writeStringToFile(final File file, final String data) throws IOException {
		writeStringToFile(file, data, null);
	}

	/**
	 * 文字列をファイルへ書き込み。
	 * @param file ファイル
	 * @param data データ
	 * @param encoding エンコーディング
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void writeStringToFile(final File file, final String data, final String encoding) throws IOException {
		OutputStream out = null;
		try {
			out = openOutputStream(file);
			write(data, out, encoding);
		} finally {
			closeQuietly(out);
		}
	}

	/**
	 * 文字列を出力ストリームへ書き込み。
	 * @param data データ
	 * @param output 出力
	 * @param encoding エンコーディング
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void write(final String data, final OutputStream output, final String encoding) throws IOException {
		if (data != null) {
			if (encoding == null) {
				write(data, output);
			} else {
				output.write(data.getBytes(encoding));
			}
		}
	}

	/**
	 * 文字列を出力ストリームへ書き込み。
	 * @param data データ
	 * @param output 出力
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void write(final String data, final OutputStream output) throws IOException {
		if (data != null) {
			output.write(data.getBytes());
		}
	}

	/**
	 * ファイルのオープン。
	 * @param file ファイル
	 * @return ファイル出力ストリーム
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static FileOutputStream openOutputStream(final File file) throws IOException {
		if (file.exists()) {
			if (file.isDirectory()) {
				throw new IOException(fMsg.getOpenOutputStreamIllegalArgument1(file.getName()));
			}
			if (file.canWrite() == false) {
				throw new IOException(fMsg.getOpenOutputStreamIllegalArgument2(file.getName()));
			}
		} else {
			final File parent = file.getParentFile();
			if (parent != null && parent.exists() == false) {
				if (parent.mkdirs() == false) {
					throw new IOException(fMsg.getOpenOutputStreamIllegalArgument3(file.getName()));
				}
			}
		}
		return new FileOutputStream(file);
	}

	/**
	 * 入力ストリームを出力ストリームへコピー。
	 * @param input 入力
	 * @param output 出力
	 * @return バイト数
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static int copy(final InputStream input, final OutputStream output) throws IOException {
		final long count = copyLarge(input, output);
		if (count > Integer.MAX_VALUE) {
			return -1;
		}
		return (int) count;
	}

	/**
	 * 入力ストリームをライターへコピー。
	 * @param input 入力
	 * @param output 出力
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void copy(final InputStream input, final Writer output) throws IOException {
		final InputStreamReader in = new InputStreamReader(input);
		copy(in, output);
	}

	/**
	 * リーダーをライターへコピー。
	 * @param input 入力
	 * @param output 出力
	 * @return 文字数
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static int copy(final Reader input, final Writer output) throws IOException {
		final long count = copyLarge(input, output);
		if (count > Integer.MAX_VALUE) {
			return -1;
		}
		return (int) count;
	}

	/**
	 * 入力ストリームをライターへコピー。
	 * @param input 入力
	 * @param output 出力
	 * @param encoding エンコーディング
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void copy(final InputStream input, final Writer output, final String encoding) throws IOException {
		if (encoding == null) {
			copy(input, output);
		} else {
			final InputStreamReader in = new InputStreamReader(input, encoding);
			copy(in, output);
		}
	}

	/**
	 * 入力ストリームを出力ストリームへコピー。
	 * @param input 入力
	 * @param output 出力
	 * @return バイト数
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static long copyLarge(final InputStream input, final OutputStream output) throws IOException {
		final byte[] buffer = new byte[DEFAULT_BUFFER_SIZE];
		long count = 0;
		int n = 0;
		while (-1 != (n = input.read(buffer))) {
			output.write(buffer, 0, n);
			count += n;
		}
		return count;
	}

	/**
	 * リーダーをライターへコピー。
	 * @param input 入力
	 * @param output 出力
	 * @return 文字数
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static long copyLarge(final Reader input, final Writer output) throws IOException {
		final char[] buffer = new char[DEFAULT_BUFFER_SIZE];
		long count = 0;
		int n = 0;
		while (-1 != (n = input.read(buffer))) {
			output.write(buffer, 0, n);
			count += n;
		}
		return count;
	}

	/**
	 * ファイルのコピー。
	 * @param srcFile ソース・ファイル
	 * @param destFile 宛先ファイル
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void copyFile(final File srcFile, final File destFile) throws IOException {
		copyFile(srcFile, destFile, true);
	}

	/**
	 * ファイルのコピー。
	 * @param srcFile ソース・ファイル
	 * @param destFile 宛先ファイル
	 * @param preserveFileDate ファイル日付の保存
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void copyFile(final File srcFile, final File destFile, final boolean preserveFileDate)
			throws IOException {
		if (srcFile == null) {
			throw new NullPointerException(fMsg.getCopyFileIllegalArgument1());
		}
		if (destFile == null) {
			throw new NullPointerException(fMsg.getCopyFileIllegalArgument2());
		}
		if (srcFile.exists() == false) {
			throw new FileNotFoundException(fMsg.getCopyFileIllegalArgument3(srcFile.getName()));
		}
		if (srcFile.isDirectory()) {
			throw new IOException(fMsg.getCopyFileIllegalArgument4(srcFile.getName()));
		}
		if (srcFile.getCanonicalPath().equals(destFile.getCanonicalPath())) {
			throw new IOException(fMsg.getCopyFileIllegalArgument5(srcFile.getName(), destFile.getName()));
		}
		if (destFile.getParentFile() != null && destFile.getParentFile().exists() == false) {
			if (destFile.getParentFile().mkdirs() == false) {
				throw new IOException(fMsg.getCopyFileIllegalArgument6(destFile.getName()));
			}
		}
		if (destFile.exists() && destFile.canWrite() == false) {
			throw new IOException(fMsg.getCopyFileIllegalArgument7(destFile.getName()));
		}
		doCopyFile(srcFile, destFile, preserveFileDate);
	}

	/**
	 * ファイルコピーの実行。
	 * @param srcFile 入力ファイル。
	 * @param destFile 出力ファイル。
	 * @param preserveFileDate ファイルの日付を維持するかどうか。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	private static void doCopyFile(final File srcFile, final File destFile, final boolean preserveFileDate)
			throws IOException {
		if (destFile.exists() && destFile.isDirectory()) {
			throw new IOException(fMsg.getDoCopyFileIllegalArgument1(destFile.getName()));
		}

		final FileInputStream input = new FileInputStream(srcFile);
		try {
			final FileOutputStream output = new FileOutputStream(destFile);
			try {
				copy(input, output);
			} finally {
				closeQuietly(output);
			}
		} finally {
			closeQuietly(input);
		}

		if (srcFile.length() != destFile.length()) {
			throw new IOException(fMsg.getDoCopyFileIllegalArgument2(srcFile.getName(), destFile.getName()));
		}
		if (preserveFileDate) {
			// この箇所は FindBugs により、メソッドの戻り値がチェックされていないと警告を受けました。
			// しかし、ここは変更せずに、そのままおいておきます。
			destFile.setLastModified(srcFile.lastModified());
		}
	}

	/**
	 * 入力ストリームのクローズ。
	 * @param input 入力
	 */
	public static void closeQuietly(final InputStream input) {
		try {
			if (input != null) {
				input.close();
			}
		} catch (final IOException ioe) {
			// 無視！
		}
	}

	/**
	 * ライターのクローズ。
	 * @param output 出力
	 */
	public static void closeQuietly(final Writer output) {
		try {
			if (output != null) {
				output.close();
			}
		} catch (final IOException ioe) {
			// 無視！
		}
	}

	/**
	 * 出力ストリームのクローズ。
	 * @param output 出力
	 */
	public static void closeQuietly(final OutputStream output) {
		try {
			if (output != null) {
				output.close();
			}
		} catch (final IOException ioe) {
			// 無視！
		}
	}

	/**
	 * ディレクトリーの削除。
	 * @param directory ディレクトリー
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void deleteDirectory(final File directory) throws IOException {
		if (!directory.exists()) {
			return;
		}
		cleanDirectory(directory);
		if (!directory.delete()) {
			throw new IOException(fMsg.getDeleteDirectoryIllegalArgument1(directory.getName()));
		}
	}

	/**
	 * ディレクトリーのクリーン。
	 * @param directory 対象ディレクトリー。
	 * @throws IOException 入出力例外が発生した場合
	 */
	private static void cleanDirectory(final File directory) throws IOException {
		if (!directory.exists()) {
			throw new IllegalArgumentException(fMsg.getCleanDirectoryIllegalArgument1(directory.getName()));
		}
		if (!directory.isDirectory()) {
			throw new IllegalArgumentException(fMsg.getCleanDirectoryIllegalArgument2(directory.getName()));
		}
		final File[] files = directory.listFiles();
		if (files == null) { // null if security restricted
			throw new IOException(fMsg.getCleanDirectoryIllegalArgument3(directory.getName()));
		}
		IOException exception = null;
		for (int i = 0; i < files.length; i++) {
			final File file = files[i];
			try {
				forceDelete(file);
			} catch (final IOException ioe) {
				exception = ioe;
			}
		}
		if (null != exception) {
			throw exception;
		}
	}

	/**
	 * ファイルまたはディレクトリーの強制削除。
	 * @param file ファイル
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void forceDelete(final File file) throws IOException {
		if (file.isDirectory()) {
			deleteDirectory(file);
		} else {
			final boolean filePresent = file.exists();
			if (!file.delete()) {
				if (!filePresent) {
					throw new FileNotFoundException(fMsg.getForceDeleteIllegalArgument1(file.getName()));
				}
				throw new IOException(fMsg.getForceDeleteIllegalArgument2(file.getName()));
			}
		}
	}

	// これ以降は純粋な追加コード。

	/**
	 * 相対パスの取得。
	 * @param folder フォルダー
	 * @param file ファイル
	 * @return 相対パス
	 */
	public static String relativePath(final File folder, final File file) {
		try {
			final String folderPath = folder.getCanonicalPath();
			final String filePath = file.getCanonicalPath();
			return filePath.replace(folderPath, ""); //$NON-NLS-1$
		} catch (final IOException e) {
			throw new IllegalArgumentException(fMsg.getRelativePathIllegalArgument1(folder.getName(), file.getName(), e
					.toString()), e);
		}
	}

	/**
	 * ファイルから拡張子を除いた名前を取得。
	 * <pre>
	 * 例）abc.html     -> abc
	 * 例）abc.html.xlf -> abc.html
	 * </pre>
	 * @param file ファイル
	 * @return 拡張子を除いた名前
	 */
	public static String baseName(final File file) {
		return file.getName().replaceFirst("\\.[^\\.]+$", ""); //$NON-NLS-1$  //$NON-NLS-2$
	}
}
